window.FECRU = window.FECRU || {};

/**
 * Code based on work from David Black
 */
FECRU.XSRF = (function ($, contextPathGetter) {
    /**
     * Cookie name with expected XSRF token value
     * @type {string}
     */
    var EXPECTED_TOKEN_COOKIE_BASE = "atl.xsrf.token";
    /**
     * Form parameter which should contain token value
     * @type {string}
     */
    var REQUEST_PARAM_NAME = "atl_token";
    /**
     * Header name to switch off XSRF token check
     * @type {string}
     */
    var REQUEST_HEADER_NAME = "X-Atlassian-Token";
    /**
     * Header value to switch off XSRF token check
     * @type {string}
     */
    var REQUEST_HEADER_VALUE = "no-check";

    /**
     * Test if url is in the same origin as current document
     *
     * @returns {boolean} True if same origin
     */
    function isSameOriginUrl(url) {
        url = url ? url.toString() : "";
        if (url.match(new RegExp('^[a-z]+://' + document.location.host + '($|/)', "i"))) {
            // matches origin host
            return true;
        } else if (!url.match(/^(\/\/|[a-z]+:).*/i)) {
            // is relative
            return true;
        } else {
            return false;
        }
    }

    /**
     * Returns expected expected XSRF token value
     *
     * @returns {string} Token value
     */
    function atlToken() {
        return $.cookie(EXPECTED_TOKEN_COOKIE_BASE + (contextPathGetter() || "/").replace(/\//g, ".slash"));
    }

    /**
     * Returns request parameter name which should hold token value
     *
     * @returns {string} Token value
     */
    function atlTokenParam() {
        return REQUEST_PARAM_NAME;
    }

    function xsrfSubmitHandler(event) {
        if (event.isDefaultPrevented()) {
            // someone has canceled the event
            return;
        }

        var form = $(event.target);
        if (isSameOriginUrl(form.attr('action')) && (form.attr('method') || "").toString().toUpperCase() === "POST") {
            if (form.find("input[name='" + REQUEST_PARAM_NAME + "']").length === 0) {
                var xsrfToken = atlToken();
                form.append($("<input>", {
                    "type": "hidden",
                    "name": REQUEST_PARAM_NAME,
                    "value": xsrfToken
                }));
            }
        }
    }

    function xsrfAjaxPrefilter(options, originalOptions, xhr) {
        if (isSameOriginUrl(options.url)) {
            xhr.setRequestHeader(REQUEST_HEADER_NAME, REQUEST_HEADER_VALUE);
        }
    }

    /**
     * Creates temporary POST form from uri and submits it
     *
     * @param uri Uri string or object compatible with {@link parseUri()}
     */
    function postUri(uri) {
        var uriData = typeof uri === "object" ? uri : FECRU.parseUri(uri || "");
        var params = $.extend({}, uriData.params);

        var form = $("<form>", {
            "style": "display:none",
            "method": "POST",
            "action": uriData.path
        });

        $.each(params, function (name, value) {
            form.append($("<input>", {
                "type": "hidden",
                "name": name,
                "value": value
            }));
        })

        $("body").append(form);
        form.submit();
    }

    function xsrfAnchorPostClickHandler(event) {
        if (event.isDefaultPrevented()) {
            // someone has canceled the event
            return;
        }

        event.preventDefault();
        postUri($(this).data("href") || $(this).attr("href"));
    }

    function registerAjaxPrefilter() {
        $.ajaxPrefilter(xsrfAjaxPrefilter);
    }

    function registerSubmitHandler() {
        $(document).on('submit', 'form', xsrfSubmitHandler);
    }

    function registerAnchorPostClickHandler() {
        $(document).on('click', 'a.anchor-post', xsrfAnchorPostClickHandler);
    }


    registerAjaxPrefilter(); // do it right now to do it before any $.ajax() call
    registerSubmitHandler();
    registerAnchorPostClickHandler();

    return {
        isSameOriginUrl: isSameOriginUrl,
        atlToken: atlToken,
        atlTokenParam: atlTokenParam,
        postUri: postUri
    };
})(AJS.$, AJS.contextPath);
