FECRU.MIXINS = FECRU.MIXINS || {};
(function (namespace, $) {

    /*
     * Usage: $.extend(MyClass.prototype, EventProducer);
     * Will add bind, unbind, trigger, and getBinder methods to objects of MyClass.
     */
    var EventProducer = (function () {
        /*
         * @param type - a string defining the event type to bind to.
         * @param handler - a function that will be called when the event is triggered.
         */
        function bind(type, handler) {
            var handlers = this._eventHandlers || (this._eventHandlers = {});
            var typeHandlers = handlers[type] || (handlers[type] = []);
            typeHandlers.push(handler);

            return this;
        }

        /*
         * @param type - a string defining the event type to unbind from.
         * @param handlerToRemove - the handler to unbind.
         */
        function unbind(type, handlerToRemove) {
            var handlers;
            var typeHandlers;
            var index;

            if ((handlers = this._eventHandlers) &&
                (typeHandlers = handlers[type]) &&
                (index = $.inArray(handlerToRemove, typeHandlers)) !== -1) {
                Array.remove(typeHandlers, index);
            }

            return this;
        }

        /*
         * @param type - a string defining the event type to trigger.
         * @param eventObj - an object with event-related properties.  A clone of the object will be passed to the handlers.
         */
        function trigger(type, eventObj, options) {
            if (eventObj != null && !$.isPlainObject(eventObj)) {
                throw "You must trigger an event with an object";
            }

            var shouldSetTimeout = options && options.clearCallStack;
            var handlerRunner = shouldSetTimeout ? function (handler) {
                setTimeout(function () {
                    handler($.extend({type: type}, eventObj));
                }, 0);
            } : function (handler) {
                handler($.extend({type: type}, eventObj));
            };

            var handlers;
            var typeHandlers;

            if ((handlers = this._eventHandlers) &&
                (typeHandlers = handlers[type])) {
                var immutableHandlerList = typeHandlers.slice(0);
                Array.each(immutableHandlerList, handlerRunner);
            }
        }

        function makeBinder(target, binder) {
            binder.bind = function () {
                bind.apply(target, arguments);
            };
            binder.unbind = function () {
                unbind.apply(target, arguments);
            };

            if (target._eventNames) {
                Array.each(target._eventNames, function (eventName) {
                    binder[eventName] = function (func) {
                        bind.call(target, eventName, func);
                    };
                });
            }

            return binder;
        }

        /**
         * Returns an object with only a bind method.  When called, events are bound to the EventProducer instead.
         */
        function getBinder() {
            return makeBinder(this, {});
        }

        function EventProducer(eventNamesArray) {
            if (!this instanceof EventProducer) {
                return new EventProducer(eventNamesArray);
            }

            if (eventNamesArray && eventNamesArray.length) {
                this._eventNames = eventNamesArray;

                var self = this;
                Array.each(eventNamesArray, function (eventName) {
                    self[eventName] = function (func) {
                        if (func && typeof func === 'function') {
                            bind.call(self, eventName, func);
                        } else {
                            var args = Array.prototype.slice.call(arguments);
                            args.unshift(eventName);
                            trigger.apply(self, args);
                        }
                    };
                });
            }
        }

        EventProducer.bind = EventProducer.prototype.bind = bind;
        EventProducer.unbind = EventProducer.prototype.unbind = unbind;
        EventProducer.trigger = EventProducer.prototype.trigger = trigger;
        EventProducer.getBinder = EventProducer.prototype.getBinder = getBinder;

        return EventProducer;
    })();

    $.extend(namespace, {
        EventProducer: EventProducer
    });
})(FECRU.MIXINS, AJS.$);
/*[{!mixins_js_o21453v!}]*/