//built-in changes
(function () {
    // Array Remove - By John Resig (MIT Licensed)
    Array.remove = function (array, from, to) {
        var rest = array.slice((to || from) + 1 || array.length);
        array.length = from < 0 ? array.length + from : from;
        return array.push.apply(array, rest);
    };

    // Seems like this method doesn't work as expected. Will just ignore eslint errors
    /*eslint-disable no-undef*/
    Array.splice = function (array, index, count/*, replacement items... */) {
        var rest = array.slice((to || from) + 1 || array.length);
        array.length = from < 0 ? array.length + from : from;
        array.push.apply(array, arguments.slice(3));
        return array.push.apply(array, rest);
    };
    /*eslint-enable*/

    /*
     * Traverse an array, calling func on each array value in indexed order.
     *
     * @param array  array to traverse
     * @param func   function taking an array value as its first parameter and an array index as its second.
     */
    Array.each = function (array, func) {
        for (var i = 0, l = array.length; i < l && func(array[i], i) !== false; i++) {
        }
    };

    /*
     * Given an array, and a function that returns true/false, Array.any returns
     * true if there is at least one value in the array for which the function returns true.
     *
     * @param array      array to traverse
     * @param predicate  function taking an array value as its first parameter and an array index as its second.
     * @return           true if predicate returns true for any array value, false otherwise.
     */
    Array.any = function (array, predicate) {
        var i = 0;
        var l = array.length;

        for (; i < l && !predicate(array[i], i); i++) {
        }
        return i !== l; // return false if empty array
    };
    /*
     * Given an array, and a function that returns true/false, Array.all returns
     * true if the function returns true for all array values (or if the array is empty).
     *
     * @param array      array to traverse
     * @param predicate  function taking an array value as its first parameter and an array index as its second.
     * @return           true if array is empty or predicate returns true for all array values, false otherwise.
     */
    Array.all = function (array, predicate) {
        var i = 0;
        var l = array.length;

        for (; i < l && predicate(array[i], i); i++) {
        }
        return i === l; // return true if empty array
    };
    /*
     * @return the first value in the array for which predicate is true, or undefined if predicate never returns true.
     */
    Array.first = function (array, predicate) {
        var i = 0;
        var l = array.length;

        for (; i < l && !predicate(array[i], i); i++) {
        }
        return i !== l ? array[i] : undefined;
    };
    /*
     * @return the last value in the array for which predicate is true, or undefined if predicate never returns true.
     */
    Array.last = function (array, predicate) {
        var i = array.length - 1;
        for (; i >= 0 && !predicate(array[i], i); i--) {
        }
        return i >= 0 ? array[i] : undefined;
    };

    /*
     * Returns a new array containing the results of func applied to each value in the given array.
     * E.g. if output = Array.map(input, func),
     *    then output[i] = func(input[i]), where i is any index in the input array.
     *
     * @param array  An array to traverse
     * @param func   A func to apply to each array value and whose return value will be used in the resulting array
     * @return       An array containing the result of func applied to each array value.
     */
    Array.map = function (array, func) {
        var result = [];
        Array.each(array, function (element) {
            result.push(func(element));
        });
        return result;
    };
    /*
     * Combines all array values by calling combine on a resultSoFar and the next element. E.g.,
     * Array.reduce([1,2,3], 0, function(a,b) { return a + b; }) === 6,
     * Array.reduce([2,3,4], 1, function(a,b) { return a * b; }) === 24
     *
     * @param array    An array to reduce
     * @param base     A starting value
     * @param combine  A function taking two values and returning a new value.
     * @return         A single value resulting from cumulatively calling combine on the result of the last
     *                 combine call(or base) and the next array element.
     */
    Array.reduce = function (array, base, combine) {
        Array.each(array, function (element) {
            base = combine(base, element);
        });
        return base;
    };
})();

window.FECRU = window.FECRU || {};
FECRU.DATA_STRUCTURES = FECRU.DATA_STRUCTURES || {};
(function (namespace, $, window) {

    function indexOf(array, item) {
        for (var i = 0, l = array.length; i < l; i++) {
            if (item === array[i].item) {
                return i;
            }
        }
        return -1;
    }

    function compare(a, b) {
        return a.comparable < b.comparable ? -1 :
            a.comparable > b.comparable ? 1 :
                0;
    }

    var PriorityQueue = function () {
        var m_queue = [];
        var dirty = false;

        this.save = function (item, comparableValue) {
            var index = indexOf(m_queue, item);
            if (index !== -1) {
                dirty = true;
                m_queue[index].item = item;
                m_queue[index].comparable = comparableValue;
            } else {
                // set dirty if it was already dirty, or if this one doesn't belong at the end.
                dirty = dirty || (m_queue.length && compare(comparableValue, m_queue[m_queue.length - 1].comparable) < 0);
                m_queue.push({
                    item: item,
                    comparable: comparableValue
                });
            }
            return this;
        };
        this.remove = function () {
            if (dirty) {
                m_queue.sort(compare);
                dirty = false;
            }
            return m_queue.shift().item;
        };
        this.length = function () {
            return m_queue.length;
        };
    };
    var LRUQueue = function () {
        PriorityQueue.call(this);
        var superSave = this.save;
        this.save = function (item, optionalDate) {
            superSave.call(this,
                item, (optionalDate && optionalDate.getTime) ? optionalDate : new Date());
        };
    };

    //Object.prototype.hasOwnProperty can still be overwritten, but this is about as safe as we can get.
    var hasOwnProperty = window.Object.prototype.hasOwnProperty;
    /*
     Map is a safer way to create a hash map.
     Using an object directly won't support keys named 'hasOwnProperty' without breaking the very method
     you use to determine if the object has the key!

     ALL KEYS ARE CONVERTED TO STRING.  If using Map with non-String keys, ensure your key's toString function
     will create unique string representations.
     * */
    var Map = function () {
        var holder = {};
        this.has = function (key) {
            //holder.hasOwnProperty(key) would fail if someone had called .set('hasOwnProperty', 'break it')
            return hasOwnProperty.call(holder, key);
        };
        this.get = function (key) {
            return hasOwnProperty.call(holder, key) ? holder[key] : undefined;
        };
        this.set = function (key, value) {
            holder[key] = value;
        };
        this.remove = function (key) {
            return delete holder[key];
        };
        this.getKeys = function () {
            var arr = [];
            for (var key in holder) {
                if (hasOwnProperty.call(holder, key)) {
                    arr.push(key);
                }
            }
            return arr;
        };
        this.getValues = function () {
            var arr = [];
            for (var key in holder) {
                if (hasOwnProperty.call(holder, key)) {
                    arr.push(holder[key]);
                }
            }
            return arr;
        };
        this.clear = function () {
            holder = {};
        };
    };

    var CacheMap = function (entriesToStore) {
        Map.call(this);

        // if they pass in an invalid value, act like a normal map.
        if (entriesToStore <= 0) {
            return;
        }

        var deletionQueue = new LRUQueue();

        var oldSet = this.set;
        var oldGet = this.get;
        var oldRemove = this.remove;

        this.get = function (key) {
            deletionQueue.save(key);
            return oldGet.call(this, key);
        };

        this.set = function (key, value) {
            deletionQueue.save(key);
            oldSet.call(this, key, value);
            while (deletionQueue.length() > entriesToStore) {
                oldRemove.call(this, deletionQueue.remove());
            }
        };

        var sendToFront = new Date(0);
        this.remove = function (key) {
            oldRemove.call(this, key);
            deletionQueue.save(key, sendToFront);
        }
    };

    $.extend(namespace, {
        LeastRecentlyUsedQueue: LRUQueue,
        Map: Map,
        CacheMap: CacheMap
    });
})(FECRU.DATA_STRUCTURES, AJS.$, this);

/*[{!data_structures_js_c6b453l!}]*/