/**
 *
 * This file is for enabling the functionality in dashboard v2:
 *  The components in dashboard are :
 *      quick access widget
 *      review table
 *      extra quick search
 *
 *  When there is a need to re-use the component, it can be make as plugin and be moved into a separate js file in the future.
 */

;
(function ($, g) {

    /**
     * This backbone class provide more decent templating method for views, until we introduce a sophisticated templating
     * engine to FECru, e.g. soy template
     *
     * Any View extending this class can utilise the functionality by simply having <something data-tpl></something> in
     * the view's html and specify the tplConfig
     *
     * tplConfig: {}
     *      $template_name as #tpl_config: {}
     *          _: #temp_function
     *          || function( data, render ) as #temp_func
     *              @data: #any
     *              -- the the data to be injected
     *              @render: function( tplKey, direcitve )
     *              -- the function of renderTpl
     *                  @tplKey: string
     *                  -- the key of the template, set by data-tpl
     *                  @directive: #directive
     *                  -- the directive to help render method to inject value to jquery object
     *
     *  #directive: {}
     *      $sel: #directive
     *      -- $sel is the selector for the render method to drill down to a/some sub jquery object(s)
     *      html: #any
     *      -- set the html of the current target to #any
     *      text: string
     *      val: string
     *      data: {}
     *          $key: #any
     *      attr: {}
     *          $key: string
     *      func: function( target )
     *          @target: #jqr_object
     *
     *
     * @type {*}
     */
    var ViewWithJqrTemplate = Backbone.View.extend({
        initialize: function () {
            this._prepareTpls();
        },
        _parseTpls: function () {
            var $tpls = this.$el.find('[data-tpl]');
            var _tplJqrs = this._tplJqrs = {};
            $tpls.each(function () {
                var $t = $(this);
                _tplJqrs[$t.attr('data-tpl')] = $t;
                $t.css('display', '');
            });

            $tpls.remove().removeAttr('data-tpl');
        },

        /*eslint-disable max-depth*/
        _renderTpl: function (key, directive, $tpl) {
            $tpl = $tpl || this._tplJqrs[key].clone();

            for (var i in directive) {
                if (directive.hasOwnProperty(i)) {
                    var dc = directive[i];
                    switch (i) {
                        case 'attr':
                            for (var attr in dc) {
                                if (dc.hasOwnProperty(attr)) {
                                    $tpl.attr(attr, dc[attr]);
                                }
                            }
                            break;
                        case 'html':
                            $tpl.html(dc);
                            break;
                        case 'val':
                            $tpl.val(dc);
                            break;
                        case 'text':
                            $tpl.text(dc);
                            break;
                        case 'func':
                            dc($tpl);
                            break;
                        case 'data':
                            $tpl.data(dc);
                        default:
                            this._renderTpl(key, dc, $tpl.find(i));
                            break;
                    }
                }
            }
            return $tpl;
        },
        /*eslint-enable*/

        _getTplFn: function (_this, fn, render) {
            return function (data) {
                return fn.call(_this, data, render);
            }
        },
        _prepareTpls: function (conf, render) {
            var t = this;
            var firstCall = false;
            if (!conf) {
                firstCall = true;
                t._parseTpls();
                conf = t.tplConfig;
                render = function (key, directive) {
                    return t._renderTpl(key, directive);
                }
            }
            if (conf._) {
                var tpl = function (data) {
                    return tpl._(data);
                };
                tpl._ = t._getTplFn(t, conf._, render);
            } else {
                var tpl = {};
            }

            for (var key in conf) {
                if (conf.hasOwnProperty(key) && conf[key] !== '_') {
                    if (typeof conf[key] === 'function') {
                        tpl[key] = t._getTplFn(tpl, conf[key], render);
                    } else {
                        tpl[key] = t._prepareTpls(conf[key], render);
                    }
                }
            }

            if (firstCall) {
                this.tpl = tpl;
            }
            return tpl;
        },
        tplConfig: {}
    })

    var ViewWithJqrGetter = ViewWithJqrTemplate.extend({
        initialize: function () {
            ViewWithJqrTemplate.prototype.initialize.apply(this, arguments);
            this.prepare$();
        },
        _getJqrFromFunc: function ($target, sel, parent) {
            return function () {
                var selStr = sel.apply(this, arguments);
                if (selStr.charAt(0) === '+') {
                    return parent().filter(selStr.substr(1));
                } else {
                    return parent().find(selStr);
                }
            }
        },
        _getJqr: function ($target, sel, $parent, parent) {
            if (sel.charAt(0) === '/') {
                sel = sel.substr(1);
                if (sel.charAt(0) === '+') {
                    sel = sel.substr(1);
                    var $ret = $parent.filter(sel);
                } else {
                    $ret = $parent.find(sel);
                }
                return function () {
                    return $ret;
                }
            } else {
                if (sel.charAt(0) === '+') {
                    sel = sel.substr(1);
                    return function () {
                        return parent().filter(sel);
                    }
                } else {
                    return function () {
                        return parent().find(sel);
                    }
                }
            }
        },
        prepare$: function ($target, conf, $parent, parent) {
            var firstCall = false;
            var t = this;
            var ret;

            if (!conf) {
                $parent = $target = this.$el;
                conf = this.$config;
                firstCall = true;
                ret = this.$ = function () {
                    return t.$el;
                }
            } else {
                if (conf._) {
                    ret = t._getJqr($target, conf._, $parent, parent);
                    $parent = ret();
                } else {
                    ret = {};
                }
            }

            for (var key in conf) {
                if (conf.hasOwnProperty(key) && conf[key] !== '_') {
                    if (typeof conf[key] === 'string') {
                        ret[key] = t._getJqr($target, conf[key], $parent, ret);
                    } else if (typeof conf[key] === 'function') {
                        ret[key] = t._getJqrFromFunc($target, conf[key], ret);
                    } else {
                        ret[key] = t.prepare$($target, conf[key], $parent, ret);
                    }
                }
            }

            return ret;
        },
        $config: {}
    });

    var QuickAccess = ViewWithJqrGetter.extend({
        initialize: function (options) {
            ViewWithJqrGetter.prototype.initialize.apply(this, arguments);
            var t = this;
            t._ajaxQueue = {};
            t._type = 'repository';

            this.dataSource = {
                repository: options.repoDataSource,
                project: options.projDataSource
            };

            var resultCollection = Backbone.Collection.extend({
                maxResults: -1,
                initialize: function (options) {
                    this.maxResults = options.maxResults;
                },
                reachMax: function () {
                    return this.maxResults !== -1 && this.size() >= this.maxResults;
                }
            });
            this.collection = {
                repository: new resultCollection({
                    maxResults: this.dataSource.repository._maxResults
                }),
                project: new resultCollection({
                    maxResults: this.dataSource.project._maxResults
                })
            };


            _.bindAll(t, 'render', 'processItems');

            t.collection.project.on('reset', t.render);
            t.collection.repository.on('reset', t.render);
            t.dataSource.repository.on('respond', function (rsp) {
                t._dequeueAjax('repositorySpinner', 1);
                t._dequeueAjax('repositoryInlineSpinner', 1);
                t.processItems(rsp, 'repository');
            });
            t.dataSource.project.on('respond', function (rsp) {
                t._dequeueAjax('projectSpinner', 1);
                t._dequeueAjax('projectInlineSpinner', 1);
                t.processItems(rsp, 'project');
            });

            t.queries = {
                repository: '',
                project: ''
            };

            t.lastQueryWithoutResult = {
                repository: null,
                project: null
            };

            t.queryDelayTimer = {
                repository: null,
                project: null
            };

            t.$.contentContainer().resizable({
                handles: {
                    s: t.$.resizeHandle()
                }
            });

            t.initInstanceType(options.instanceType);
        },
        _instanceType: null,
        queries: null,
        collection: null, // Backbone.Collection
        dataSource: null, // the progressiveDataSet
        _type: null, // repository (default) | project
        CON_keynavMargin: 30,
        initInstanceType: function (instanceType) {
            var t = this;
            t._instanceType = instanceType;
            switch (instanceType) {
                case 'fisheyeOnly':
                    t.$.tabs.project().remove();
                    t.$.tabs.repository().find('a').addClass('corner-tr');
                    t.switchTab('repository', {preventFocus: true});
                    break;
                case 'crucibleOnly':
                    t.$.tabs.repository().remove();
                    t.$.tabs.project().find('a').addClass('corner-tl');
                    t.switchTab('project', {preventFocus: true});
                    break;
                default:
                    t.switchTab('repository', {preventFocus: true});
                    t.dataSource.project.query('');
                    break;
            }
        },
        tplConfig: {
            item: {
                _: function (item, render) {
                    return render('item', {
                        'a': {
                            attr: {
                                'href': item.get('href'),
                                'title': item.get('name')
                            },
                            text: item.get('name')
                        },
                        'span': {
                            func: function ($t) {
                                if (item.get('key')) {
                                    $t.html(item.get('key'));
                                } else {
                                    $t.hide();
                                }
                            }
                        }
                    })
                }
            },
            moreResults: {
                _: function (item, render) {
                    return render('more-results');
                }
            }
        },
        $config: {
            inlineSpinner: '/.quick-access-spinner',
            spinner: '/.block-ui',
            filterBox: '/.filter-box',
            allContent: '/.quick-access-content',
            contentContainer: '/.content-container',
            repository: {
                _: '/.quick-access-repositories',
                recent: '/.recent-repositories',
                all: '/.all-repositories',
                noResult: '/.no-result'
            },
            project: {
                _: '/.quick-access-projects',
                recent: '/.recent-projects',
                all: '/.all-projects',
                noResult: '/.no-result'
            },
            resizeHandle: '/.resize-handle',
            tabs: {
                _: '/.menu-item',
                repository: '/+ [data-type=repository]',
                project: '/+ [data-type=project]'
            }
        },
        events: {
            'click .menu-item': 'switchTab',
            'keyup .filter-box': 'startQuery',
            'keydown .filter-box': 'keydownFilterbox',
            'mouseenter ul li': 'highlightItem',
            'mouseleave ul li': 'unhighlightItem'
        },
        highlightItem: function (e, $t) {
            $t = $(e.currentTarget);
            $t.parent().parent()
                .find('li.selected').removeClass('selected');
            $t.addClass('selected');
        },
        unhighlightItem: function (e, $t) {
            $t = $(e.currentTarget);
            $t.removeClass('selected');
        },
        switchTab: function (e, options) {
            var t = this;
            var $t;
            var type;
            var options = options || {};

            if (typeof e === 'string') {
                type = e;
                $t = t.$.tabs[type]();
            } else {
                $t = $(e.currentTarget);
                if ($t.is('.active-tab')) {
                    return;
                }

                type = $t.attr('data-type');
            }

            t.$.filterBox().val(t.queries[type])
                .attr('placeholder', 'Filter ' + (type === 'project' ? 'projects...' : 'repositories...'));

            if (!options.preventFocus) {
                t.$.filterBox().focus().select();
            }

            var $all = t.$.allContent();
            $all.hide();
            t.$[type]().show();
            t._type = type;

            t.$.tabs().removeClass('active-tab');
            $t.addClass('active-tab');

            if (!t.$[type]().data('inited')) {
                var query = t.queries[type];
                t._queueAjax(type + 'Spinner', 1);
                t.$.spinner().show();
                t.dataSource[type].query(query);
                t.$[type]().data('inited', true);
            }
        },
        lastQueryWithoutResult: null,
        queryDelayTimer: null,
        keydownFilterbox: function (e) {
            var t = this;
            var type = t._type;

            if (e.keyCode === 13) {
                t.$[type]().find('li.selected a')[0].click();
            } else if (e.keyCode === 38 || e.keyCode === 40) { // 38: up, 40: down
                var $lis = t.$[type]().find('li:not(.more-results)');
                var $li = $lis.filter('.selected').removeClass('selected');
                if (!$li.size()) {
                    $li = $lis.eq(0).addClass('selected');
                    e.preventDefault();
                    return;
                }

                var index = $lis.index($li);

                if (e.keyCode === 38) {
                    index--;
                } else {
                    index++;
                }
                if (index < 0) {
                    index = 0;
                } else if (index > $lis.size() - 1) {
                    index = $lis.size() - 1;
                }
                $li = $lis.eq(index).addClass('selected');


                // handling scrolling
                var $cont = $li.parent().parent();
                var height = $cont.height();
                var scrollTop = $cont.scrollTop();
                var top = $li.position().top;

                if (top < t.CON_keynavMargin) {
                    $cont.scrollTop(scrollTop + top - t.CON_keynavMargin);
                } else if (top > height - t.CON_keynavMargin) {
                    $cont.scrollTop(scrollTop + (top - height) + t.CON_keynavMargin);
                }

                e.preventDefault();
            }
        },
        startQuery: function (e) {
            var $t = $(e.currentTarget);
            var t = this;
            var query = $.trim($t.val());
            var type = t._type;

            if (e.keyCode === 13 || e.keyCode === 38 || e.keyCode === 40) { // 38: up, 40: down, 13: enter
                return false;
            }

            clearTimeout(t.queryDelayTimer[type]);
            t.queries['project'] = query;
            t.queries['repository'] = query;

            if (type === 'repository') {
                t.$.project().data('inited', false);
            } else {
                t.$.repository().data('inited', false);
            }

            if ((type === 'repository' && query.indexOf(t.lastQueryWithoutResult[type]) === 0)
                || (type === 'project' && query.indexOf(t.lastQueryWithoutResult[type]) === 0)) {
                return;
            }

            t.queryDelayTimer[type] = setTimeout(function () {
                t._queueAjax(type + 'InlineSpinner', 1);
                t.$.inlineSpinner().show();
                if (type === 'repository') {
                    t.dataSource[type].query(query);
                } else {
                    t.dataSource[type].query(query);
                }
            }, 200);
        },
        processItems: function (rsp, type) {
            var t = this;
            t.lastQueryWithoutResult[type] = null;

            var completed = false;
            if (!rsp.remoteQuery) {
                completed = t.dataSource[type].hasQueryCache(rsp.query) || !t.dataSource[type].shouldGetMoreResults(rsp.results);
            } else {
                completed = rsp.query === rsp.remoteQuery || (rsp.query.indexOf(rsp.remoteQuery) === 0 && !t.dataSource[type].shouldGetMoreResults(rsp.results));
            }

            if (completed) {
                t.collection[type].reset(rsp.results);
                if (rsp.query && !rsp.results.length) {
                    t.lastQueryWithoutResult[type] = rsp.query;
                }
            }
        },
        render: function () {
            var t = this;
            var hasItemShown = false;
            var $spinner = t.$.spinner();
            var $inlineSpinner = t.$.inlineSpinner();

            if (t.isFilterSearching()) {
                $inlineSpinner.show();
            } else {
                $inlineSpinner.hide();
            }

            if (t.isTabLoading()) {
                $spinner.show();
            } else {
                $spinner.hide();

                var type = t._type;
                var $recent = t.$[type].recent();
                var $all = t.$[type].all();
                var $noResult = t.$[type].noResult();
                var $item = null;
                var $firstItem = null;

                $recent.empty();
                _.each(t.recentItems[type], function (item) {
                    if (item.matches(t.queries[type])) {
                        $item = t.tpl.item(item);
                        if (!$firstItem) {
                            $firstItem = $item;
                        }
                        $recent.append($item);
                    }
                });

                if (!$item) {
                    $recent.hide()
                        .prev().hide();
                    $all.prev().addClass('first-heading');
                } else {
                    hasItemShown = true;
                    $recent.show()
                        .prev().show();
                    $all.prev().removeClass('first-heading');
                }

                $all.empty();
                $item = null;
                t.collection[type].each(function (item) {
                    $item = t.tpl.item(item);
                    if (!$firstItem) {
                        $firstItem = $item;
                    }
                    $all.append($item);
                });
                if (!$item) {
                    $all.hide()
                        .prev().hide();
                } else {
                    hasItemShown = true;
                    $all.show()
                        .prev().show();

                    if (t.collection[type].reachMax()) {
                        $all.append(t.tpl.moreResults());
                    }
                }

                if ($firstItem) {
                    $firstItem.addClass('selected');
                }
                $all.parent().scrollTop(0);

                if (!hasItemShown) {
                    $noResult.show();
                } else {
                    $noResult.hide();
                }
            }
        },
        isLoading: function (keys) {
            var t = this;
            for (var key in t._ajaxQueue) {
                if (t._ajaxQueue.hasOwnProperty(key)) {
                    if ((key in keys) && t._ajaxQueue[key]) {
                        return true;
                    }
                }
            }
            return false;
        },
        isTabLoading: function () {
            return this.isLoading({
                repositorySpinner: 1,
                projectSpinner: 1,
                recent: 1
            })
        },
        isFilterSearching: function () {
            return this.isLoading({
                repositoryInlineSpinner: 1,
                projectInlineSpinner: 1
            })
        },
        _ajaxQueue: null,
        _ajaxSeq: 1,
        _contextPath: AJS.contextPath(),
        _queueAjax: function (key, seq) {
            var t = this;
            t._ajaxQueue[key] = seq || ++t._ajaxSeq;
            return seq || t._ajaxSeq;
        },
        _dequeueAjax: function (key, seq) {
            var t = this;
            if (t._ajaxQueue[key] === seq) {
                t._ajaxQueue[key] = 0;
                return true;
            } else {
                return false;
            }
        },
        recentItems: null,
        getRecent: function () {
            var t = this;
            var seq = t._queueAjax('recent');

            t.recentItems = {
                repository: [],
                project: []
            };

            $.ajax({
                url: t._contextPath + '/rest-service-fecru/recently-visited-v1/detailed.json',
                type: 'GET',
                cache: false,
                dataType: 'json',
                success: function (rsp) {
                    if (t._instanceType !== 'crucibleOnly') {
                        $.each(rsp.repositories.repository, function (__, repo) {
                            var data = new Item({
                                name: repo.repoData.name,
                                href: t._contextPath + repo.uri
                            });
                            t.recentItems.repository.push(data);
                        });
                    }

                    if (t._instanceType !== 'fisheyeOnly') {
                        $.each(rsp.projects.project, function (__, proj) {
                            var data = new Item({
                                name: proj.projectData.name,
                                key: proj.projectData.key,
                                href: t._contextPath + proj.uri
                            });
                            t.recentItems.project.push(data);
                        });
                    }

                    t._dequeueAjax('recent', seq);
                    t.render();
                }
            })
        }
    });

    /**
     * attributes:
     *  name: str
     *  key: str
     *  href: str
     *
     * @type Item
     */
    var Item = Backbone.Model.extend({
        initialize: function () {
            var t = this;
            var context = AJS.contextPath();

            if (t.has('displayPrimary')) {
                var primary = t.get('displayPrimary');
                t.unset('displayPrimary');

                if (t.has('displaySecondary')) {
                    var secondary = t.get('displaySecondary');
                    t.set({
                        name: secondary,
                        key: primary,
                        href: context + '/project/' + primary
                    });
                } else {
                    t.set({
                        name: primary,
                        href: context + '/changelog/' + primary
                    });
                }
            }
        },
        idAttribute: "id",
        matcher: function (model, query) {
            // this method is for matching query string in start of words.
            // not used at the moment but might be used in the future.
            var E = '\x1b';
            var comps = [];

            if (!model.has('comps')) {
                var key = model.get('key');
                if (key) {
                    comps.push(key);
                }
                var name = model.get('name');

                name = name.replace(/([\s\-\_\.]+)/g, '$1' + E);

                var pts = name.split(E);
                var suffix = '';
                var i = pts.length;

                while (i-- > 0) {
                    suffix = pts[i] + suffix;
                    if (suffix) {
                        comps.push(suffix);
                    }
                }
                model.set('comps', comps);
            } else {
                comps = model.get('comps');
            }

            var res = _.find(comps, function (comp) {
                return comp.toLowerCase().indexOf(query.toLowerCase()) === 0;
            });

            return typeof res !== 'undefined';
        },
        sorter: function (itemA, itemB) {
            var nameA = itemA.get('name').toLowerCase();
            var nameB = itemB.get('name').toLowerCase();

            if (nameA > nameB) {
                return 1;
            } else if (nameA < nameB) {
                return -1;
            } else {
                return 0;
            }
        },
        matcherAnywhere: function (model, query) {
            if (model.get('name').toLowerCase().indexOf(query.toLowerCase()) !== -1) {
                return true;
            }
            if (model.get('key') && model.get('key').toLowerCase().indexOf(query.toLowerCase()) !== -1) {
                return true;
            }
            return false;
        },
        matches: function (query) {
            return this.matcher(this, query);
        }
    });

    var repoDataSource = new AJS.ProgressiveDataSet([], {
        queryEndpoint: AJS.contextPath() + "/json/fe/repositoryFinder.do?limit=100",
        queryParamKey: "q",
        maxResults: 50,
        model: Item,
        allowEmptyQuery: true,
        matcher: Item.prototype.matcher,
        sorter: Item.prototype.sorter
    });

    var projDataSource = new AJS.ProgressiveDataSet([], {
        queryEndpoint: AJS.contextPath() + "/json/cru/projectFinder.do?limit=100",
        queryParamKey: "q",
        maxResults: 50,
        model: Item,
        allowEmptyQuery: true,
        matcher: Item.prototype.matcher,
        sorter: Item.prototype.sorter
    });

    var $rssFeed;
    var rssUrl;

    $(function () {
        var quickAccess = new QuickAccess({
            el: '#quick-access',
            repoDataSource: repoDataSource,
            projDataSource: projDataSource,
            instanceType: AJS.instanceType
        });
        quickAccess.getRecent();

        $rssFeed = $('.aui-toolbar .rss-feed');
        // depends on jquery.deserialize
        rssUrl = $rssFeed.attr('href');

        var parts = rssUrl.split('?');
        var params;
        var paramStr = "";

        if (parts[1] && $.trim(parts[1])) {
            params = $.deserialize(parts[1]);
            delete params['view'];
            paramStr = "?" + $.param(params);
        }
        rssUrl = parts[0] + paramStr;

        $('#toolbar .activity-stream-filter:not(.active)').on('click', '.aui-button', function (e) {
            var $t = $(this);
            var type = $t.data('stream-type');

            $t.parent().children().removeClass('active');
            $t.addClass('active');
            FECRU.ActivityStream.loadMoreActivity($t.attr('href'));
            $rssFeed.attr('href', rssUrl + '&view=' + type);
            e.preventDefault();
        });
    })

})(AJS.$, window);
/*[{!dashboard_js_ml7x53k!}]*/
