/**
 * Simple wrapper for creating and managing dynamic stylesheets
 * Allows to set new CSS rules from javascript
 *
 * @example
 *
 * // appends <style type="text/css" id="changeset-page"></style> into <head>
 * var stylesheet = cssJs('changeset-page');
 * //NOTE: Creating separate names stylesheets is just for debugging purposes
 * // You can always use global stylesheet cssJs.getGlobalStylesheet()
 *
 * // Creates new CSS rule for "#changeset-content pre" selector
 * stylesheet.set('#changeset-content pre', {
 *      color: 'red'
 * });
 *
 * // NOTE: It's impossible to add new CSS rules for the same selector
 * // so by default all properties for same selector are override
 * stylesheet.set('#changeset-content pre', {
 *      color: 'yellow', // such yellow, wow
 *      width: '100px'
 * });
 */
var cssJs = (function () {
    var StyleSheet = function (name, doc) {
        this.document = doc || document;
        this.name = name;
        this.sheet = createSheet.call(this);
    };
    var createSheet = function () {
        var head = this.document.head || this.document.getElementsByTagName('head')[0];
        var style = this.document.createElement('style');

        style.type = 'text/css';
        style.id = this.name;
        head.appendChild(style);
        return style.sheet;
    };
    var extendStyles = function (styles, properties) {
        for (var key in properties) {
            if (properties.hasOwnProperty(key) && key in styles) {
                styles[key] = properties[key];
            }
        }
    };
    var getStyleRule = function (selector, returnIndex) {
        var rules = this.sheet.cssRules || this.sheet.rules;
        var length = rules.length;
        var rule;

        for (var i = 0; i < length; i++) {
            rule = rules[i];
            // sometimes chrome removes custom "originalSelector" property
            if ((rule.originalSelector || rule.selectorText) === selector) {
                return {
                    rule: rules[i],
                    index: i
                };
            }
        }
    };
    var createStyleRule = function (selector) {
        var rules = this.sheet.cssRules || this.sheet.rules;
        var index = rules.length;

        if ('insertRule' in this.sheet) {
            this.sheet.insertRule(selector + ' { }', index);
        } else if ('addRule' in this.sheet) {
            this.sheet.addRule(selector, ' ', index);
        }
        // for faster selector comparison since IE9 reorders selector elements in rule
        rules[index].originalSelector = selector;
        return {
            rule: rules[index],
            index: index
        };
    };
    var removeStyleRule = function (index) {
        if ('removeRule' in this.sheet) {
            this.sheet.removeRule(index);
        } else if ('deleteRule' in this.sheet) {
            this.sheet.deleteRule(index);
        }
    };


    StyleSheet.prototype = {
        constructor: StyleSheet,
        /**
         * Creates/override CSS rule and sets given CSS properties on it
         *
         * @param {String} selector CSS rule selector
         * @param {Object} properties CSS properties for rule
         */
        set: function (selector, properties) {
            var rule = getStyleRule.call(this, selector) || createStyleRule.call(this, selector);
            if (rule) {
                extendStyles(rule.rule.style, properties);
            }
        },
        /**
         * Returns CSSStyleDeclaration for given selector
         *
         * @param {String} selector
         * @returns {CSSStyleDeclaration}
         */
        get: function (selector) {
            var rule = getStyleRule.call(this, selector);
            return rule && rule.rule.style;
        },
        /**
         * Removes CSS rule for given selector
         *
         * @param {String} selector
         */
        remove: function (selector) {
            var rule = getStyleRule.call(this, selector);
            rule && removeStyleRule.call(this, rule.index);
        }
    };

    var exports = function (name, document) {
        return new StyleSheet(name, document);
    };
    var globalStyleSheet;

    /**
     * Returns global dynamic stylesheets
     *
     * @returns {StyleSheet}
     */
    exports.getGlobalStylesheet = function () {
        if (!globalStyleSheet) {
            globalStyleSheet = new StyleSheet(document, 'fecru-global');
        }
        return globalStyleSheet;
    };

    return exports;
})();
/*[{!css_js_a3r553j!}]*/