/**
 *
 * This js file contains helpers for barracuda's UI, e.g. calling the REST endpoint or creating AUI message etc.
 *
 * Dependencies:
 *
 *  HTML elements
 *      div.adg-new-message : for holding the adg style aui message
 *      div.content-view ui.info : for holding the spinner & "indexing..."
 *
 *  JS variables
 *      FECRU.INFO:
 *          repoName: string         -- repository name
 *          isPipelined: boolean    -- if the repository is using pipeline indexing (currently only SVN)

 */

;
(function ($, g) {

    if (!FECRU.INFO) {
        FECRU.INFO = {
            /**
             * Set a page variable to FECRU.INFO so that it can be visible from plugin
             * @param key
             * @param val
             */
            set: function (key, val) {
                if (typeof key === 'string') {
                    FECRU.INFO[key] = val;
                } else if (typeof key === 'object') {
                    $.extend(FECRU.INFO, key);
                }
            }
        };
    }

    var emptyFunc = function () {
    };

    /**
     * The ScanningTimer is for handling the smooth progress bar.
     * Updating the info box and control the visibility of barracuda page message & indexing message
     *
     * Usage :
     *
     * var timer = new ScanningTimer();
     * timer.setTarget(status);
     *
     * // status's structure : []
     *  scanning: integer
     *  indexing: integer
     *
     * @constructor
     */
    var ScanningTimer = function () {
    };
    ScanningTimer.prototype = {
        _timer: false,
        /**
         * 'none' : init state
         * 'scanning'   : when scanning is still on
         * 'indexing'   : when scanning is finished
         * 'complete'   : when everything is done
         */
        state: 'none',
        target: {},
        curr: null,
        delay: 0,
        step: 1,
        setState: function (state) {
            if (state === this.state) {
                return;
            }
            vars.$indexingMessage.fadeOut(500);
            if (!vars.isFromAddContentDialog) {
                vars.$pageMessage.slideUp(300);
                if (state === 'scanning') {
                    vars.$pageMessage.stop().show();
                }
            }
            if (state === 'complete') {
                this.stop();
                vars.inlineDialog.hideInfoDialog();
                vars.inlineDialog.hideAppvertisingDialog();
                vars.pollingTimer.stop();
            } else if (state === 'indexing' || vars.isFromAddContentDialog) {
                vars.pollingTimer.minInterval = 5000;
                vars.inlineDialog.hideInfoDialog();
                vars.$indexingMessage.stop().show();
            }
            this.state = state;
            AJS.trigger('barracuda-ui-updated');
        },
        setTarget: function (status, pollingTimer) {
            var t = this;
            if (status.scanning === 100 && status.indexing === 100) {
                t._inlineDialog.updateScanning(t.curr.scanning);
                t._inlineDialog.updateIndexing(t.curr.indexing);
                t.stop();
                pollingTimer.stop();
                setTimeout(function () {
                    t.setState('complete');
                }, 1500);
                return;
            }

            if (status.scanning === t.target.scanning && status.indexing === t.target.indexing) {
                if (t.curr) {
                    pollingTimer.update('up');
                }
                return;
            } else {
                pollingTimer.update('down');
                t.target = status;
            }

            if (!t.curr) {
                t.curr = $.extend({}, t.target);
            }

            var diff = t.target.scanning - t.curr.scanning;
            var sign = diff > 0 ? 1 : -1;
            if (diff === 0) {
                this.delay = 0;
                this.step = 0;
            } else {
                this.delay = pollingTimer.currInterval / (sign * diff);
                this.step = sign;
            }

            if (t.state === 'scanning' && !vars.isFromAddContentDialog) {
                if (!this._timer) {
                    this.tick();
                }
                t.curr.indexing = status.indexing;
            } else {
                t.curr = status;
                if (t.state === 'scanning') {
                    t.state = 'indexing';
                }
                t._inlineDialog.updateScanning(t.curr.scanning);
            }
            t._inlineDialog.updateIndexing(t.curr.indexing);
        },
        stop: function () {
            vars.get$progress().stop();
            this._timer = false;
        },
        tick: function (delay, step) {
            var t = this;
            if (typeof delay === 'undefined') {
                delay = this.delay;
            }
            if (typeof step === 'undefined') {
                step = this.step;
            }

            if (t.state === 'scanning') {
                vars.get$progress().stop().animate({
                    width: t.curr.scanning + step + '%'
                }, delay, function () {
                    if (step > 0) {
                        t.curr.scanning += step;
                        if (status.scanning !== t.target.scanning && t.target.scanning) {
                            t._onTargetChangedOnce();
                            t._onTargetChangedOnce = emptyFunc;
                        }
                    }
                    vars.get$progressText().html((t.curr.scanning + '%'));

                    t._inlineDialog.updateScanning(t.curr.scanning);

                    if (t.curr.scanning >= 100) {
                        setTimeout(function () {
                            t._onScanComplete();
                            vars.inlineDialog.viewWarningExplanation();
                            t.setState('indexing');
                        }, 1500);
                    } else if (t.curr.scanning !== t.target.scanning) {
                        t.tick();
                    }
                })
            }
        },
        onScanComplete: function (callback) {
            this._onScanComplete = callback;
        },
        _onScanComplete: emptyFunc,
        onTargetChangedOnce: function (callback) {
            this._onTargetChangedOnce = callback;
        },
        _onTargetChangedOnce: emptyFunc,
        setInlineDialog: function (inlineDialog) {
            this._inlineDialog = inlineDialog;
        },
        _inlineDialog: null
    };

    /**
     *
     * This class is for enabling polling at a dynamic interval.
     * Invoking .update('up') can enlarge the interval, otherwise .update('down')
     *
     * Meanwhile the interval will be kept between minInterval & maxInterval
     *
     * Usage:
     *
     *  var timer = new DynamicPolling(function() {
     *      // .. do something periodically
     *  });
     *
     *  timer.run();
     *
     *  timer.update('up');
     *  timer.update('down');
     *
     *  timer.stop();
     *
     * @param thread
     * @constructor
     */
    var DynamicPolling = function (thread) {
        this._thread = thread;
        this.run();
    };
    DynamicPolling.prototype = {
        minInterval: 2000,
        maxInterval: 60000,
        initIntervals: [1000],
        currInterval: 2000,
        count: 0,
        _timer: false,
        _thread: emptyFunc,
        stop: function () {
            clearTimeout(this._timer);
            this._timer = false;
        },
        run: function () {
            var t = this;

            var delay = 0;
            if (t.count in t.initIntervals) {
                delay = t.initIntervals;
            } else {
                delay = t.currInterval;
            }
            t.count++;

            t._timer = setTimeout(function () {
                t._thread();
                t.run();
            }, delay);
            t.currInterval = delay;
        },
        update: function (direction) {
            var t = this;

            if (direction === 'up') {
                t.currInterval *= 2;
                if (t.currInterval > t.maxInterval) {
                    t.currInterval = t.maxInterval;
                }
            } else if (direction === 'down') {
                t.currInterval /= 3;
                if (t.currInterval < t.minInterval) {
                    t.currInterval = t.minInterval;
                }
            }
        }
    };

    var ajaxs = {
        checkStatus: function (repoName, callback) {
            // REST returns: {scanningPercentage: 19, indexingPercentage: 10, userHasSeenWarning: false}
            $.ajax({
                url: CON.urls.restStatus + '/' + repoName,
                dataType: 'json',
                success: function (status) {
                    callback(status);
                },
                type: 'GET',
                cache: false
            });
        },
        isWarningViewed: function (callback) {
            var APPV = FECRU.APPVERTISING;

            // REST returns: true | false
            $.ajax({
                url: APPV.url + APPV.prefNames.barracuda_indexing,
                dataType: 'json',
                contentType: 'application/json',
                success: function (isViewed) {
                    callback(isViewed);
                },
                type: 'GET',
                cache: false
            });
        },
        setWarningViewed: function (isViewed) {
            var viewed = 'false';
            if (typeof isViewed === 'undefined' || isViewed) {
                viewed = 'true';
            }

            // REST has no return value
            $.ajax({
                url: FECRU.APPVERTISING.url,
                dataType: 'json',
                data: JSON.stringify({property: FECRU.APPVERTISING.prefNames.barracuda_indexing, value: viewed}),
                contentType: 'application/json',
                type: 'POST',
                cache: false
            });
        }
    };

    var actions = {
        createScanningMessage: function (repoName) {
            AJS.messages.warning(vars.$pageMessage, {
                title: 'The repository \'' + repoName + '\' is currently being scanned',
                body: AJS.template.load('barracuda-page-message').toString(),
                closeable: false
            });
            vars.isScanningMessageCreated = true;
        },
        createIndexingIcon: function () {
            var $index = $(AJS.template.load('barracuda-indexing-message').toString());

            vars.$indexingMessage.html($index);
        },
        createScanningInlineDialog: function () {
            var $info = $(AJS.template.load('barracuda-info-inline-dialog').toString());
            var $appvertising = $(AJS.template.load('barracuda-appvertising-inline-dialog').toString());

            var handler = function ($cont, trigger, showPopup) {

                $cont.html($info);

                $info.find('.close').unbind('click')
                    .bind('click', function () {
                        dialog.hide();
                    });

                $info.find('.refresh-page').unbind('click')
                    .click(function () {
                        location.reload();
                    });

                appvertisingDialog.hide();
                showPopup();
            };

            var dialog = AJS.InlineDialog('#barracuda-progress-info, .barracuda-indexing-message .message-text', 'barracuda-progress-info', handler, {
                onHover: false,
                showDelay: 200,
                width: 360,
                displayShadow: false
            });

            var appvertisingHandler = function ($cont, trigger, showPopup) {
                $cont.html($appvertising);

                $cont.find('.dismiss').unbind('click')
                    .click(function () {
                        appvertisingDialog.hide();
                    });
                showPopup();
            };

            var appvertisingDialog = AJS.InlineDialog(' .barracuda-indexing-message .message-text', 'barracuda-appvertising', appvertisingHandler, {
                onHover: false,
                noBind: true,
                hideDelay: null,
                width: 360,
                hideCallback: function () {
                    actions.setExplanationViewed();
                },
                displayShadow: false
            });

            /**
             * The function encapsulates the dialog object, only export three useful API for updating the status of
             * the inline dialog (info box)
             */
            return {
                hideInfoDialog: function () {
                    dialog.hide();
                },
                hideAppvertisingDialog: function () {
                    appvertisingDialog.hide();
                },
                updateScanning: function (number) {
                    if (number === 100) {
                        $info.find('.scanning .aui-lozenge')
                            .show()
                            .attr('class', 'aui-lozenge aui-lozenge-success')
                            .html('completed');
                        $info.find('.scanning .progress-bar').hide();
                        $info.find('.scanning .number').hide();
                    } else if (vars.isFromAddContentDialog) {
                        $info.find('.scanning .progress-bar').css('display', 'inline-block');
                        $info.find('.scanning .aui-lozenge').hide();
                        $info.find('.scanning .progress-bar .progress-content').css('width', number + '%');
                    }
                    $info.find('.scanning .number').html(number + '%');
                },
                updateIndexing: function (number) {
                    if (number === 100) {
                        $info.find('.indexing .aui-lozenge')
                            .attr('class', 'aui-lozenge aui-lozenge-success')
                            .html('completed');
                        $info.find('.indexing .number').hide();
                    } else if (number > 0) {
                        if (vars.scanningTimer.state === 'indexing') {
                            $info.find('.indexing .progress-bar').css('display', 'inline-block');
                            $info.find('.indexing .aui-lozenge').hide();
                            $info.find('.indexing .progress-bar .progress-content').css('width', number + '%');
                        } else {
                            $info.find('.indexing .aui-lozenge')
                                .attr('class', 'aui-lozenge aui-lozenge-current')
                                .html('in progress');
                        }
                        $info.find('.indexing .number').css('display', 'inline-block');
                    }
                    $info.find('.indexing .number').html(number + '%');
                },
                viewWarningExplanation: function () {
                    actions.checkExplanationViewed(function (explanationViewed) {
                        if (!explanationViewed) {
                            setTimeout(function () {
                                appvertisingDialog.show();
                            }, 1000);
                        }
                    });
                }
            }
        },
        checkExplanationViewed: function (callback) {
            if (FECRU.isAnon) {
                callback(FECRU.ClientStorage.getJSON(FECRU.APPVERTISING.keys.barracuda_indexing));
            } else {
                ajaxs.isWarningViewed(function (result) {
                    callback(result);
                });
            }
        },
        setExplanationViewed: function () {
            if (FECRU.isAnon) {
                FECRU.ClientStorage.setJSON(FECRU.APPVERTISING.keys.barracuda_indexing, true);
            } else {
                ajaxs.setWarningViewed(true);
            }
        },
        /**
         * The most important method: query the backend to get the indexing status, and update scanningTimer.
         * This is the thread of a dynamicTimer.
         * @param repoName
         * @param pollingTimer
         */
        updateScanningMessage: function (repoName, pollingTimer) {
            if (vars.scanningTimer.state === 'complete') {
                return;
            }

            ajaxs.checkStatus(repoName, function (status) {
                status = {
                    scanning: Math.floor(status.scanningPercentage),
                    indexing: Math.floor(status.indexingPercentage),
                    isStopped: status.isStopped
                };
                if (status.scanning > 100) {
                    status.scanning = 100;
                }
                if (status.indexing > 100) {
                    status.indexing = 100;
                }

                if (!vars.isScanningMessageCreated) {
                    if (!status.isStopped && (status.scanning !== 100 || status.indexing !== 100)) {
                        if (!vars.isFromAddContentDialog) {
                            actions.createScanningMessage(repoName);
                        }
                        actions.createIndexingIcon();
                        vars.inlineDialog = actions.createScanningInlineDialog();
                        vars.scanningTimer.setInlineDialog(vars.inlineDialog);
                        vars.isScanningMessageCreated = true;

                        if (status.scanning !== 100) {
                            vars.scanningTimer.setState('scanning');
                        } else if (status.indexing !== 100) {
                            vars.inlineDialog.viewWarningExplanation();
                            vars.scanningTimer.setState('indexing');
                        }
                    } else {
                        vars.scanningTimer.state = 'complete';
                        return;
                    }
                }

                vars.scanningTimer.setTarget(status, pollingTimer);
            });
        }
    };

    // variables for this plugin
    var vars = {
        get$progress: function () {
            return vars.$pageMessage.find('.progress-bar .progress-content');
        },
        get$progressText: function () {
            return vars.$pageMessage.find('.main span');
        },
        isScanningMessageCreated: false,
        // the timer to smoothly update the percentage change
        scanningTimer: false,
        // the interval timer for pooling the REST service to update the scanning index
        pollingTimer: false,
        inlineDialog: {
            hideInfoDialog: function () { /* init method */
            },
            hideAppvertisingDialog: function () { /* init method */
            }
        },
        // the barracuda page message
        $pageMessage: null,
        // container at the top-right corner to display "indexing" link
        $indexingMessage: null
    };

    var CON; // constants

    $(document).ready(function () {

        vars.$pageMessage = $('.barracuda-page-message').eq(0);

        if (FECRU.INFO.isFromAddContentDialog) {
            vars.$indexingMessage = $('#controls-container .indexing-message');
            vars.isFromAddContentDialog = true;
        } else if (FECRU.isChangesetPage) {
            vars.$indexingMessage = $('#barracuda-indexing-message-wrap');
        } else {
            vars.$indexingMessage = $('#content-column-panel .info').eq(0);
        }

        CON = {
            urls: {
                // the url to getting the status of the current repo
                restStatus: FECRU.pageContext + '/rest-service-fecru/indexing-status-v1/status'
            }
        };

        if (FECRU.INFO && FECRU.INFO.repoName && FECRU.INFO.isPipelined) {

            if (vars.$indexingMessage.size() === 1) {
                vars.scanningTimer = new ScanningTimer();
                vars.scanningTimer.onTargetChangedOnce(function () {
                    vars.$pageMessage.find('.refresh').css('visibility', 'visible');
                });

                vars.pollingTimer = new DynamicPolling(function () {
                    actions.updateScanningMessage(FECRU.INFO.repoName, vars.pollingTimer);
                });
                actions.updateScanningMessage(FECRU.INFO.repoName, vars.pollingTimer);
            }
        }
    });

    FECRU.INFO.set({
        /*
         * This is for telling the dirlist page not to refresh the content-view div when swithing directory, during
         * scanning/indexing.
         */
        isPipelineIndexingRunning: function () {
            return !(vars.scanningTimer === false || vars.scanningTimer.state === 'complete' || vars.scanningTimer.state === 'none');
        }
    })

})(AJS.$, window);
/*[{!barracuda_toolkit_js_ic2q53g!}]*/