FECRU.ANALYTICS = (function ($, sendAnalyticsEvent, document) {
    var createEventDescriptor = function (value) {
        if (value instanceof DOMEventDescriptor) {
            return value;
        }
        return DOMEventDescriptor.create(value);
    };
    var createAnalyticsEventDescriptor = function (value) {
        if (value instanceof Function) {
            return value;
        }
        if (value instanceof AnalyticsEventDescriptor) {
            return value;
        }
        return AnalyticsEventDescriptor.create(value);
    };
    var obtainAnalyticsEventValue = function (valueProvider, valueProviderContext, extraArguments) {
        // if function - call it
        if (valueProvider instanceof Function) {
            return valueProvider.apply(valueProviderContext, extraArguments);
        }
        // otherwise it's a value
        return valueProvider;
    };

    /**
     * @class
     *
     * @param {String} name
     * @param {String} data additional event properties
     * @constructor
     */
    var AnalyticsEvent = function (name, data) {
        /** @member {String} */
        this.name = name + '';
        /** @member {Object.<String, (String|Number)>} */
        this.data = data || {};
    };

    AnalyticsEvent.createFromObject = function (object) {
        return new AnalyticsEvent(object.name, object.data);
    };

    AnalyticsEvent.createFromString = function (name) {
        return new AnalyticsEvent(name);
    };

    AnalyticsEvent.create = function (value) {
        if (typeof value === 'string') {
            return AnalyticsEvent.createFromString(value);
        }
        return AnalyticsEvent.createFromObject(value);
    };

    /**
     * Describes analytics event to send
     * @class
     *
     * @param {ValueProvider} name
     * @param {(Object.<String, ValueProvider>|Function)} data
     * @constructor
     */
    var AnalyticsEventDescriptor = function (name, data) {
        if (!name) {
            throw new TypeError('Analytics event descriptor name cannot be empty. Provide some name or function that returns it');
        }
        /** @member {ValueProvider} */
        this.name = name;

        /** @member {(Object.<String, ValueProvider>|Function)} */
        this.data = data || {};
    };


    /**
     * Creates an analytics event based on current properties descriptors and HTMLElement
     *
     * Process of resolving looks as follows:
     * - if a property value IS NOT a function - use the value
     * - if a property value IS a function - call it in context of DOMElement and use the result
     *
     * @param {HTMLElement} element
     * @param {Array.<*>} [extraArguments] additional arguments for value providers functions
     * @return {AnalyticsEvent}
     */
    AnalyticsEventDescriptor.prototype.resolveWithDOMElement = function (element, extraArguments) {
        var eventName = obtainAnalyticsEventValue(this.name, element, extraArguments);
        var eventData = obtainAnalyticsEventValue(this.data, element, extraArguments);

        eventData = Object.getOwnPropertyNames(eventData)
            .reduce(function (currentData, propertyName) {
                currentData[propertyName] = obtainAnalyticsEventValue(eventData[propertyName], element, extraArguments);
                return currentData;
            }, eventData);

        return new AnalyticsEvent(eventName, eventData);
    };

    AnalyticsEventDescriptor.createFromObject = function (object) {
        return new AnalyticsEventDescriptor(object.name, object.data);
    };

    AnalyticsEventDescriptor.createFromString = function (name) {
        return new AnalyticsEventDescriptor(name);
    };

    AnalyticsEventDescriptor.create = function (value) {
        if (typeof value === 'string') {
            return AnalyticsEventDescriptor.createFromString(value);
        }
        return AnalyticsEventDescriptor.createFromObject(value);
    };

    /**
     * A string, number or a function that returns a value for analytics event field
     *
     * @typedef {(String|Number|Function)} ValueProvider
     */

    /**
     * @class
     * @param {String} selector
     * @param {String} type Dom event type - e.g. 'click'
     * @param {HTMLElement} target Delegation context element
     * @constructor
     */
    var DOMEventDescriptor = function (selector, type, target) {
        this.selector = selector;
        this.type = type || 'mousedown';
        this.target = target || document;
    };

    DOMEventDescriptor.createFromObject = function (object) {
        return new DOMEventDescriptor(object.selector, object.type, object.target);
    };

    DOMEventDescriptor.createFromString = function (selector) {
        return new DOMEventDescriptor(selector);
    };

    DOMEventDescriptor.create = function (value) {
        if (typeof value === 'string') {
            return DOMEventDescriptor.createFromString(value);
        }
        return DOMEventDescriptor.createFromObject(value);
    };

    /**
     * Function to call in context of DOMEvent target element.
     * Should returns a full analytics event object or a string representing an analytics event name.
     *
     * If it returns a falsy value then analytics event WON'T be send
     *
     * @typedef {Function} AnalyticsEventProvider
     * @return {(String|Number|Object|AnalyticsEvent)}
     */

    /**
     * Describes an DOMEvent to listen
     *
     * @typedef {Object} DOMEventDescriptor
     * @property {String} [selector] An event selector e.g. '.list-item a'
     * @property {String} [type=mousedown]
     * @property {HTMLElement} [target=document]
     */
    var analytics = {
        /**
         * Sends analytics event
         *
         * @param {(String|AnalyticsEvent)} eventName full analytics event object or analytics event name
         * @param {Object} [eventData] analytics event data
         */
        send: function (eventName, eventData) {
            var event;
            if (typeof eventName === 'string') {
                event = {
                    name: eventName,
                    data: eventData || {}
                };
            } else {
                event = eventName;
            }
            sendAnalyticsEvent(event);
        },

        /**
         * Listens on a DOMEvent and sends an analytics event when given DOMEvent occur
         *
         * @example On button mousedown
         * // default event type: 'mousedown'
         * an.sendOn('button.showFullContext', 'cru.review.fullContext')
         *
         * @example Listens on different element (instead document)
         * an.sendOn({
             *      selector: 'a.scrollToLine',
             *      target: $('.codeContainer')
             * }, 'cru.code-viewer.scrollToLine');
         *
         * @example On form submit
         * an.sendOn({
             *      type: 'submit',
             *      form: $('form.addTrackedBranchForm')
             * }, 'cru.review.create.new-tracked-branch');
         *
         * @example Static analytics event data
         * an.sendOn('.showMoreCommits', {
             *      name: 'cru.commits.show-more',
             *      data: {
             *          source: 'user-triggered'
             *      }
             * });
         *
         * @example Dynamic analytics event data
         * // <button class="goToFrx">reviewFile.txt</button>
         *
         * an.sendOn('.goToFrx', {
             *      name: 'cru.review.go-to-frx',
             *      data: {
             *          frxNameLength: function() { return this.innerHTML.length; } // this - the button
             *      }
             * }};
         *
         * @example Dynamic analytics event name
         * an.sendOn('.addContentMethod', function() {
             *      return 'cru.review.content.'+$(this).data('method-name');
             * });
         *
         * @example Conditional send
         * an.sendOn('.addContentMethod', function() {
             *      if (someCondition()) {
             *          return 'cru.review.content.some-method'; // send analytics event
             *      }
             *      // do not send analytics event
             * });
         *
         * @example Conditional send of full analytics event object
         * an.sendOn('.addContentMethod', function() {
             *     if (someCondition()) {
             *         return {
             *             name: 'cru.review.content.some-method',
             *             data: {
             *                 extraAttribute: 'some-value'
             *             }
             *         }
             *     }
             *     // do not send analytics event
             * });
         *
         * @param {(String|Object|DOMEventDescriptor)} eventDescriptor if string then it will be treated selector
         * @param {(String|Object|AnalyticsEventProvider|AnalyticsEventDescriptor)} analyticsEventDescriptor if string then it will be treated as analytics event name
         */
        sendOn: function (eventDescriptor, analyticsEventDescriptor) {
            eventDescriptor = createEventDescriptor(eventDescriptor);
            analyticsEventDescriptor = createAnalyticsEventDescriptor(analyticsEventDescriptor);

            $(eventDescriptor.target).on(eventDescriptor.type, eventDescriptor.selector, function () {
                var targetElement = this;
                var analyticsEvent;
                var args = Array.prototype.slice.call(arguments);

                if (analyticsEventDescriptor instanceof Function) {
                    analyticsEvent = analyticsEventDescriptor.apply(targetElement, args);

                    // if response is empty, do not send the analytics event
                    if (!analyticsEvent) {
                        return;
                    }

                    if (!(analyticsEvent instanceof AnalyticsEvent)) {
                        analyticsEvent = AnalyticsEvent.create(analyticsEvent);
                    }
                } else {
                    analyticsEvent = analyticsEventDescriptor.resolveWithDOMElement(this, args);
                }
                analytics.send(analyticsEvent);
            });
        },
        AnalyticsEventDescriptor: AnalyticsEventDescriptor,
        AnalyticsEvent: AnalyticsEvent,
        DOMEventDescriptor: DOMEventDescriptor
    };

    return analytics;
})(AJS.$, AJS.trigger.bind(AJS, 'analyticsEvent'), document);