window.FECRU = window.FECRU || {};

FECRU.AJAX = (function ($, window) {

    /**
     * We need to distinguish between two cases. A user clicking away to another page, cancelling an Ajax request. In that case we
     * want to do nothing. When the server refuses a connection, we want to display an error message.
     */
    var unloaded = false;
    $(window).bind('beforeunload', function () {
        unloaded = true;
    });

    /**
     * Factory method that creates a new SequentialAjaxExecutor instance.
     */
    function createSequentialExecutor() {
        return new SequentialAjaxExecutor();
    }

    /******************************************************************
     ******************** AJAX METHODS ********************************
     ******************************************************************/

    /**
     * This class represents a serial channel for ajax calls. An ajax call that
     * is submitted only gets run after the previous call has returned from the
     * server.
     * This is used by Crucible comments: when you click "post" while autosave
     * is waiting for the server to create and return the new commentId, the
     * Post action is queued and automatically gets evaluated after autosave
     * has returned (and populated the js comment model with the new commentId).
     */
    function SequentialAjaxExecutor() {

        var busy = false;
        var $jobs = $('<div></div>');   // private var to SequentialAjaxExecutor, never inserted into the document.

        /**
         * Makes an AJAX call, similar to FECRU.AJAX.ajaxDo(), but with the exception that
         * only one call can be active at any time. When this method is called
         * a second time, while the previous call has not yet finished, the new
         * call will be scheduled and executed automatically when the previous
         * call finishes.
         *
         * @param url
         * @param params
         * @param onCompleteFunc
         */
        this.executeAjax = function (url, params, onCompleteFunc, noDialog) {
            this.executeAjaxJob({
                url: url,
                params: params,
                callback: onCompleteFunc,
                noDialog: noDialog
            });
        };

        /**
         * Similar to SerializedAjaxDispatcher.executeAjax(), but takes a hash with
         * functions that allow callers to evaluate their url and query params
         * just-in-time.
         *
         * @param job
         */
        this.executeAjaxJob = function (job) {
            if (busy) {
                var dispatcher = this;
                $jobs.queue(function () {
                    dispatcher.executeAjaxJob(job);
                });
            } else {
                busy = true;
                ajaxDo(
                    $.isFunction(job.url) ? job.url() : job.url,
                    $.isFunction(job.params) ? job.params() : job.params,
                    function (resp) {
                        try {
                            if (job.callback) {
                                job.callback(resp);
                            }
                        } finally {
                            busy = false;
                            $jobs.dequeue();
                        }
                    },
                    $.isFunction(job.noDialog) ? job.noDialog() : job.noDialog);
            }
        };

        this.isPending = function () {
            return busy;
        };
    }

    function ajaxDo(url, params, onCompleteFunc, noDialog) {
        return ajaxUpdate(url, params, null, onCompleteFunc, noDialog);
    }

    /**
     * Makes a Ajax request with the specified parameters and runs the onCompleteFunc
     * function when the call returns. Optionally, a DOM element can be passed in
     * which will be updated with the payload of the Ajax response (the JSON response
     * object must contain a "payload" member.
     *
     * noDialog is optional: if falsy, the error message dialog is shown on error.
     * If it is a callback, it is invoked on error instead of showing the dialog.
     * Otherwise, if it is truthy, the dialog is not shown on error and no action
     * is taken. It makes sense to not report an error when the Ajax action is triggered by a timer, rather than by a
     * user action, or where the failure of the Ajax action doesn't affect what the user is doing. (e.g. ?)
     * TODO any other cases where we should not show an error?
     */
    function ajaxUpdate(url, params, elementIdToUpdate, onCompleteFunc, noDialog) {
        // Implemementation note:
        // jQuery does an eval(req.responseText) for us internally. We have an onSuccess callback to
        // intercept this, otherwise we end up eval'ing the response text twice (which is poor form).
        // The onComplete callback gives us access to the XmlHttpRequest and allows us to do extra
        // error handling if required.

        var data = false;
        var onSuccess = function (response) {
            data = response;
        };

        /*eslint-disable complexity*/
        var onComplete = function (req, textStatus) {
            // We do not want to proceed if the request was aborted
            if (textStatus === 'abort') {
                return;
            }

            var cleanUpOnFail = function () {
                if (onCompleteFunc) {
                    onCompleteFunc({worked: false});  //allow function to clean up
                }
            };
            try {

                if (req.status === 0) {
                    if (!unloaded) {
                        cleanUpOnFail();
                        if (!noDialog) {
                            showServerUnreachableBox();
                        }
                    }
                    //else this was caused by user navigating away, ignore it.
                    return;
                }
                if (!data || !isECMA(req)) {
                    noDialog || displayUnexpectedResponse(req);
                    cleanUpOnFail();
                    return;
                }

                var $updateMe;
                if (elementIdToUpdate) {
                    $updateMe = $("#" + elementIdToUpdate);
                }

                if (data.hasOwnProperty("worked") && !data.worked) {
                    if (!noDialog) {
                        appendErrorResponse(data.errorMsg, data.noEscape);
                        appendCredentialsRequests(data.credentialsRequired);
                        if (data.userError) {
                            showUserErrorBox();
                        } else {
                            showErrorBox();
                        }
                    }
                } else if ($updateMe && $updateMe.length === 1) {
                    if ($updateMe.is(':input')) {
                        $updateMe.val(data.payload);
                    } else {
                        $updateMe.html(data.payload);
                    }
                }
                if (onCompleteFunc) {
                    onCompleteFunc(data);
                }
            } catch (e) {
                console && console.error('ajaxUpdate failure: ' + e);
                displayUnexpectedResponse(req);
                cleanUpOnFail();
            }
        };
        /*eslint-enable*/

        return $.ajax({
            type: "post",
            url: url,
            data: params,
            dataType: "json",
            traditional: true, // serialize array params as rev=1&rev=2 rather than rev[]=1&rev[]=2
            success: onSuccess,
            complete: onComplete,
            beforeSend: function (req, settings) {
                req.url = settings.url;
            },
            error: noDialog ? ($.isFunction(noDialog) ? noDialog : NOOP) : ajaxFailure
        });
    }


    var NOOP = function () {
    };

    var ajaxFailure = function (req, textStatus, errorThrown) {
        try {
            if (!isECMA(req)) {
                displayUnexpectedResponse(req);
                return;
            }
            var resp = eval('(' + req.responseText + ')');
            var errorShown = false;
            if (!resp.worked || resp.errorMsg || resp.message) {
                appendErrorResponse(resp.errorMsg || resp.message, resp.noEscape);
                errorShown = true;
            }
            if (errorThrown) {
                appendErrorResponse(errorThrown);
                errorShown = true;
            }
            if (!errorShown) {
                appendErrorResponse("Unknown error");
            }
            showErrorBox();
        } catch (e) {
            console && console.error('ajaxFailure: ' + e);
            displayUnexpectedResponse(req);
        }
    };

    /******************************************************************
     ******************** HELPER METHODS ******************************
     ******************************************************************/

    var isECMA = function (req) {
        return isContentType(req, /^application\/(ecmascript|json)/);
    };

    var isHTML = function (req) {
        return isContentType(req, /^text\/html/);
    };

    var isContentType = function (req, targetType) {
        var respContentType = req.getResponseHeader("Content-Type");
        return respContentType != null && respContentType.match(targetType);
    };

    var isHTMLFrag = function (req) {
        var responseText = req.responseText;
        return responseText != null && !responseText.match(/<body/i);
    };

    //TODO use FECRU.htmlEscape || AJS.escapeHtml
    var ourHtmlEscape = function (input) {
        try {
            return input.replace(/&/g, '&amp;').
                replace(/>/g, '&gt;').
                replace(/</g, '&lt;').
                replace(/"/g, '&quot;');
        } catch (e) {
            console && console.error('escape failure: ' + e);
            return input.message || 'There was an error communicating with the server';
        }
    };

    /**
     * @deprecated use $.spin() instead which was introduced in AUI 5.0-fecru1
     */
    function startSpin(id, className, replace) {
        className = className ? className + " spinner" : "spinner";

        var $el = typeof(id) === 'object' ? $(id) : $("#" + id);
        if ($el.length === 1) {
            var $spinner = $(document.createElement("span"));
            $spinner.addClass(className)
                .html("<img src='" + FECRU.pageContext + "/" + FECRU.staticDirectory + "/2static/images/blank.gif'>");
            if (replace) {
                $el.replaceWith($spinner);
            } else {
                if (!$el.next().is('.spinner')) {
                    $spinner.insertAfter($el);
                }
            }
            return true;
        }
        return false;
    }

    /**
     * @deprecated use $.spinStop() instead which was introduced in AUI 5.0-fecru1
     */
    function stopSpin(id) {
        var $el = typeof(id) === 'object' ? $(id) : $("#" + id);

        if ($el.length === 1) {
            $el.nextAll(".spinner").remove();
        }
    }


    /******************************************************************
     ******************** ERROR HANDLING ******************************
     ******************************************************************/

    function checkError(resp) {
        return resp.error;
    }

    function appendErrorResponse(error, noEscape) {
        if (error) {
            appendErrorMessage(error + (error.stack ? ":\n" + error.stack : ""), noEscape);
        }
    }

    function appendCredentialsRequests(credentialsRequired) {
        if (credentialsRequired) {
            for (var i = 0; i < credentialsRequired.length; i++) {
                var creds = credentialsRequired[i];
                var msg = "Click <a onclick=\"FECRU.HOVER.invalidateCache(FECRU.HOVER.CACHE_FOREVER); return true;\" href='" + creds.authUrl + "' target='_new'>here</a> to authenticate with " + creds.name;
                appendErrorMessage(msg, true);
            }
        }
    }

    function appendErrorMessage(errorMsg, noEscape) {
        if (errorMsg) {
            var $err = $('#errorResponses');
            var errorCount = $err.find('.errorResponse').length;
            var escaped = (noEscape) ? errorMsg : ourHtmlEscape(errorMsg);

            var $existingErrors = $('#errorResponses .errorResponse');
            for (var i = 0, len = $existingErrors.length; i < len; i++) {
                if (escaped === $($existingErrors[i]).html()) {
                    return; // dont show an error if we have already seen it
                }
            }

            $err.show()
                .append($('<div class="errorResponse-count">Error' + (errorCount ? (' ' + ++errorCount) : '') + '</div>'))
                .append($('<div class="errorResponse"></div>').html(escaped));
        }
    }

    function appendNotificationMessage(message) {
        if (message) {
            var $err = $('#errorResponses');
            var $msg = $('<div class="notification-box-message"></div>');

            if (message.jquery) {
                $msg.append(message);
            } else {
                $msg.text(message);
            }
            $err.append($msg);
        }
    }

    var dialog = 0;

    function showErrorBox() {
        showNotificationBox('An error has occurred');
        $('#errorDetails').show();
    }

    function showUserErrorBox() {
        showNotificationBox('Cannot perform the requested action', 'usererror');
        $('#errorDetails').show();
    }

    function showServerUnreachableBox() {
        showNotificationBox('The server is not responding.');
        $('#errorDetails').show();
    }

    function showNotificationBox(title, severity) {
        severity = severity || "error";
        var util = CRU.UTIL;

        if (dialog === 0) {
            dialog = FECRU.DIALOG.create(640, 480, "error-dialog");
            dialog.addHeader(title)
                .addPanel("Information", $('#errorBox'))
                .addButton("Reload", function () {
                    window.location.reload();
                }, "reload-button")
                .addButton("Close", function () {
                    $('#errorResponses .errorResponse').remove();
                    $('#errorResponses .errorResponse-count').remove();
                    dialog.hide();
                }, "close-button");
        } else {
            dialog.addHeader(title);
        }
        var $dialog = $('#error-dialog');
        if (severity === 'error') {
            $dialog.addClass('error').removeClass('message usererror');
        } else if (severity === 'usererror' && !$dialog.hasClass('error')) {
            $dialog.find(".errorResponse-count").hide();
            $dialog.addClass('usererror').removeClass('message error');
        } else if (severity === 'message' && !$dialog.hasClass('error') && !$dialog.hasClass('usererror')) {
            $dialog.addClass('message').removeClass('error usererror');
        }

        // error overrides message and usererror, usererror overrides message
        if (severity === 'error') {
            override = true; // eslint-disable-line no-undef
        } else if (severity === 'usererror' && $dialog.hasClass('message')) {
            override = true; // eslint-disable-line no-undef
        }

        if (!util.isAnyDialogShowing()) {
            if (util.isAjaxDialogSpinning()) {
                util.stopAjaxDialogSpin();
            }
            dialog.show();
        }
        return false;
    }

    function displayUnexpectedResponse(req) {
        var error = {}; // message, noEscape, title, severity;
        if (isHTML(req) && isHTMLFrag(req)) {
            error.message = req.responseText;
        }
        else if (!req.responseText) {
            //no-op here: no responseText means most likely server is gone offline, which is handled in onComplete
        }
        else {
            try {
                var resp = eval('(' + req.responseText + ')');

                if (!resp.errorMessages[0]) {
                    resp.errorDetails = "<p>No further details were provided.</p>";
                }
                else {
                    resp.errorDetails = resp.errorMessages.join("<br>");
                }

                var startsWithVowel = resp.errorName && (-1 !== "aeiou".indexOf(resp.errorName.charAt(0).toLowerCase()));

                error.message = [
                    "<p>",
                    (resp.errorName ? (startsWithVowel ? "An " : "A ") + resp.errorName : "An error"),
                    " was encountered",
                    "</p>",
                    resp.errorDetails
                ].join("");
                error.noEscape = true;
            } catch (e) {
                if (req.status >= 400) {
                    error.message = req.status + ": " + req.statusText;
                    error.title = "Error communicating with the server";
                    error.severity = "error";
                }
            }
        }
        if (error.message) {
            appendErrorResponse("<p>URL: " + AJS.escapeHtml(req.url) + "</p>" + (error.noEscape ? error.message : AJS.escapeHtml(error.message)), true);
            if (error.title) {
                showNotificationBox(error.title, error.severity);
            } else {
                showErrorBox();
            }
        }
    }

    function displayErrorInResponse(err) {
        appendErrorResponse(err);
        showErrorBox();
    }

    return {
        createSequentialExecutor: createSequentialExecutor,

        ajaxDo: ajaxDo,
        ajaxUpdate: ajaxUpdate,

        /**
         * @deprecated use $.spin() instead which was introduced in AUI 5.0-fecru1
         */
        startSpin: startSpin,
        /**
         * @deprecated use $.spinStop() instead which was introduced in AUI 5.0-fecru1
         */
        stopSpin: stopSpin,

        checkError: checkError,

        appendErrorResponse: appendErrorResponse,
        appendErrorMessage: appendErrorMessage,
        showErrorBox: showErrorBox,
        showUserErrorBox: showUserErrorBox,
        showServerUnreachableBox: showServerUnreachableBox,
        appendNotificationMessage: appendNotificationMessage,
        showNotificationBox: showNotificationBox,
        ajaxFailure: ajaxFailure
    };
})(AJS.$, this);
/*[{!ajax_js_6oig53d!}]*/