(function ($) {
    function Diff(rawDiffData) {
        if (!(this instanceof Diff)) {
            return new Diff(rawDiffData);
        }
        var parts = rawDiffData.split(Diff.DELIMITER);

        /** Sorted array of hunks. */
        this.hunks = eval(parts[0]);

        this.pagesPerBlock = eval(parts[1]);
        this.linesPerPage = eval(parts[2]);

        this.from = new File(this.pagesPerBlock, this.linesPerPage, parts[3]);
        this.to = new File(this.pagesPerBlock, this.linesPerPage, parts[4]);

        return this;
    }

    Diff.DELIMITER = '<';

    Diff.prototype = (function () {
        var prototype = Diff.prototype;

        /**
         * Merge more file blocks into the diff. Any metadata included in the
         * raw data is ignored.
         */
        prototype.merge = function (rawDiffData) {
            var diffParts = rawDiffData.split(Diff.DELIMITER);
            if (this.from.exists()) {
                this.from.merge(diffParts[3].split(File.DELIMITER));
            }
            this.to.merge(diffParts[4].split(File.DELIMITER));
        };

        /**
         * Ensures that the lines in the hunk are viewable.
         *
         * This may need to load a max of two blocks for each file from the server
         * (since the number of lines in a block is much larger than the viewport).
         *
         * @param hunk the from and to lines to show; this will typically correspond
         *      to the lines visible in the viewport.
         * @param postload optional callback executed once the entire hunk is visible.
         */
        prototype.ensureVisible = function (hunk, postload) {
            var fromRange = hunk.fromRange;
            var toRange = hunk.toRange;
            var linesPerBlock = this.pagesPerBlock * this.linesPerPage;

            var firstFromBlock = Math.floor(fromRange.low / linesPerBlock);
            var lastFromBlock = Math.floor(fromRange.high / linesPerBlock);
            var firstToBlock = Math.floor(toRange.low / linesPerBlock);
            var lastToBlock = Math.floor(toRange.high / linesPerBlock);

            var fromBlockToFetch = this.from.exists() ? nextBlockToFetch(this.from, firstFromBlock, lastFromBlock) : null;
            var toBlockToFetch = nextBlockToFetch(this.to, firstToBlock, lastToBlock);

            if (!(fromBlockToFetch || toBlockToFetch)) {
                if (this.from.exists()) {
                    this.from.ensureVisible(fromRange);
                }
                this.to.ensureVisible(toRange);
                postload && postload();
            } else {
                var diff = this;
                if (!fromBlockToFetch && diff.from.exists()) {
                    diff.from.ensureVisible(fromRange);
                }
                if (!toBlockToFetch) {
                    diff.to.ensureVisible(toRange);
                }
                $.ajax({
                    type: 'POST',
                    url: document.location.href,
                    data: {
                        linesPerPage: diff.linesPerPage,
                        fromBlock: fromBlockToFetch,
                        toBlock: toBlockToFetch,
                        ajax: "true"
                    },
                    dataType: 'text',   // Custom data format that we selectively eval.
                    success: function (rawDiffData) {
                        diff.merge(rawDiffData);
                        // Display what is visible from the just-fetched blocks.
                        if (fromBlockToFetch && diff.from.exists()) {
                            diff.from.ensureVisible(fromRange);
                        }
                        if (toBlockToFetch) {
                            diff.to.ensureVisible(toRange);
                        }
                        // Recurse to make sure the entire hunk is visible.
                        diff.ensureVisible(hunk, postload);
                    }
                });
            }
        };

        /**
         * Returns the next unrequested/unloaded block number to load for the file or null
         * if all the blocks are already requested/loaded.
         */
        function nextBlockToFetch(file, firstBlockNumber, lastBlockNumber) {
            for (var blockNumber = firstBlockNumber; blockNumber <= lastBlockNumber; blockNumber++) {
                if (!file.blocks[blockNumber]) {
                    // Mark as loading; will be overwritten upon merging the raw file data.
                    // Any object without numeric keys will work here, since we will only
                    // insert pages from the block when they've been loaded and stored under
                    // their respective page number.
                    file.blocks[blockNumber] = nextBlockToFetch.loadingBlock;
                    return blockNumber;
                }
            }
            return null;
        }

        nextBlockToFetch.loadingBlock = {loading: true};

        return prototype;
    })();

    var $gutterBoxes;

    /**
     * Call a function for each gutter we are displaying
     *
     * @param fn a function taking an int and a jQuery element -- called for each gutter box
     */
    function forEachGutterBox(fn) {
        if (typeof $gutterBoxes === "undefined") {
            $gutterBoxes = $(".gutter-box");
        }
        for (var i = 0, len = $gutterBoxes.length; i < len; ++i) {
            fn(i, $gutterBoxes.eq(i));
        }
    }


    function File(pagesPerBlock, linesPerPage, rawFileData) {
        if (!(this instanceof File)) {
            return new File(pagesPerBlock, linesPerPage, rawFileData);
        }
        var parts = rawFileData.split(File.DELIMITER);
        var metadata = eval(parts[parts.length - 1]);

        if (!this.numLines) {
            this.numLines = 0;
        }

        if (!metadata) {
            return null;
        }

        this.name = metadata.name;
        this.position = metadata.position;
        this.numLines = metadata.numLines;
        this.longestLine = metadata.longestLine;
        this.pagesPerBlock = pagesPerBlock;
        this.linesPerPage = linesPerPage;
        this.gutterWidths = metadata.gutterWidths;

        /**
         * Map of blockNumber to an array of page data (which is eval'd on demand).
         *
         * Each 'page data' is either a raw (uneval'd) string or an object
         * containing the DOM elements for that page.
         */
        this.blocks = {};

        this.merge(parts);

        // These variables can only be calculated after the DOM is ready.
        this.scrollBoxDom = null;
        this.numbersBoxDom = null;
        this.revisionBlameBoxDom = null;
        this.lineHeightPx = null;

        return this;
    }

    File.DELIMITER = '>';

    File.prototype = (function () {
        var prototype = File.prototype;

        prototype.exists = function () {
            return !!this.blocks;
        };

        /**
         * Merge more blocks into this file given the parts that result from
         * splitting the raw file data. Any file metadata included in the
         * parts is ignored.
         */
        prototype.merge = function (parts) {
            var blockNumber = eval(parts[0]);
            if (blockNumber >= 0) {
                // Store the uneval'd pages.
                this.blocks[blockNumber] = parts.slice(1, parts.length - 1);
            }
        };

        /**
         * Ensure that the pages corresponding to the range are in the DOM. This
         * can only ensure that pages from already fetched blocks are visible
         * (otherwise it is a noop).
         *
         * @param range the range of lines to show; this will typically be
         *      the range of lines visible in the viewport for this file.
         */
        prototype.ensureVisible = function (range) {
            if (this.numLines > 0) {
                var linesPerBlock = this.pagesPerBlock * this.linesPerPage;
                var firstBlock = Math.floor((range.low - 1) / linesPerBlock);
                var lastBlock = Math.floor((range.high - 1) / linesPerBlock);

                for (var blockNumber = firstBlock; blockNumber <= lastBlock; blockNumber++) {
                    var blockLineOffset = blockNumber * linesPerBlock;
                    var firstLineInBlock = (range.low - 1) - blockLineOffset;
                    var lastLineInBlock = Math.min((range.high - 1) - blockLineOffset, (blockNumber + 1) * linesPerBlock - 1);
                    var firstPage = Math.floor(firstLineInBlock / this.linesPerPage);
                    var lastPage = Math.floor(lastLineInBlock / this.linesPerPage);

                    for (var pageNumber = firstPage; pageNumber <= lastPage; pageNumber++) {
                        // This can only be the case if the block is in memory and the page hasn't yet been inserted.
                        if (this.blocks[blockNumber] && typeof this.blocks[blockNumber][pageNumber] === 'string') {
                            insertPage(this, blockNumber, pageNumber);
                        }
                    }
                }
            }
        };

        /**
         * Insert the page and its line numbers into the DOM. No check is made as
         * to whether the page is already in the DOM.
         */
        function insertPage(file, blockNumber, pageNumber) {
            var $scrollBox = $(scrollBoxDom(file));
            var $numbersBox = $(numbersBoxDom(file));
            var $revBlameBox = $(revisionBlameBoxDom(file));
            var $authBlameBox = $(authorBlameBoxDom(file));
            var cssTop = (blockNumber * file.pagesPerBlock + pageNumber) * file.linesPerPage * file.lineHeightPx;

            // TODO Page widths should be set via a dynamic css rule.
            $scrollBox.children('.content-box')[0].appendChild(
                $(pageDom(file, blockNumber, pageNumber))
                    .css('top', cssTop)
                    .css('width', $scrollBox.data('getPageWidth')())[0]
            );
            $numbersBox.children('.content-box')[0].appendChild(
                $(numbersForPageDom(file, blockNumber, pageNumber))
                    .css('top', cssTop)[0]
            );
            if ($revBlameBox.length !== 0) {
                $revBlameBox.children('.content-box')[0].appendChild(
                    $(revisionBlameForPageDom(file, blockNumber, pageNumber))
                        .css('top', cssTop)[0]
                );
            }
            if ($authBlameBox.length !== 0) {
                $authBlameBox.children('.content-box')[0].appendChild(
                    $(authorBlameForPageDom(file, blockNumber, pageNumber))
                        .css('top', cssTop)[0]
                );
            }

            forEachGutterBox(function (i, $gutterBox) {
                $gutterBox.children('.content-box')[0].appendChild(
                    $(gutterDecoratorForPageDom(file, blockNumber, pageNumber, i))
                        .css('top', cssTop)[0]
                );
            });

            triggerOnParent('sbs-page:rendered', file.blocks[blockNumber][pageNumber]);
        }

        function scrollBoxDom(file) {
            if (!file.scrollBoxDom) {
                file.scrollBoxDom = $('#' + file.name + '-box')[0];
            }
            return file.scrollBoxDom;
        }

        function numbersBoxDom(file) {
            if (!file.numbersBoxDom) {
                file.numbersBoxDom = $('#' + file.name + '-numbers-box')[0];
            }
            return file.numbersBoxDom;
        }

        function revisionBlameBoxDom(file) {
            if (!file.revisionBlameBoxDom) {
                file.revisionBlameBoxDom = $('#' + file.name + '-revision-blame-box')[0];
            }
            return file.revisionBlameBoxDom;
        }

        function authorBlameBoxDom(file) {
            if (!file.authorBlameBoxDom) {
                file.authorBlameBoxDom = $('#' + file.name + '-auth-blame-box')[0];
            }
            return file.authorBlameBoxDom
        }

        /**
         * Get the DOM element for the page. Assumes that the block has already
         * been fetched from the server.
         */
        function pageDom(file, blockNumber, pageNumber) {
            var block = file.blocks[blockNumber];
            var data = block[pageNumber];

            if (typeof data === 'string') {
                var lines = eval(data);
                var pageDom = createPageDom(lines);
                var numbersDom = createNumbersForPageDom(file, blockNumber, pageNumber, lines);
                var revisionBlameDom = createRevisionBlameForPageDom(file, blockNumber, pageNumber, lines);
                var authorBlameDom = createAuthorBlameForPageDom(file, blockNumber, pageNumber, lines);
                var gutterDom = createGutterDecoratorForPageDom(file, blockNumber, pageNumber, lines);
                var absolutePageNumber = blockNumber * file.pagesPerBlock + pageNumber;
                var lineOffset = absolutePageNumber * file.linesPerPage;
                var lineStart = lineOffset + 1;
                var lineEnd = Math.min(lineOffset + file.linesPerPage, file.numLines);

                setIdToFileBlockPage(pageDom, file, blockNumber, absolutePageNumber, 'page');
                setIdToFileBlockPage(numbersDom, file, blockNumber, absolutePageNumber, 'numbers');
                setIdToFileBlockPage(revisionBlameDom, file, blockNumber, absolutePageNumber, 'revision-blame');
                setIdToFileBlockPage(authorBlameDom, file, blockNumber, absolutePageNumber, 'auth-blame');

                // Free the memory used by raw page data by storing the DOM elements in its place.
                block[pageNumber] = {
                    pageDom: pageDom,
                    numbersDom: numbersDom,
                    revBlameDom: revisionBlameDom,
                    authBlameDom: authorBlameDom,
                    gutterDom: gutterDom,
                    lineStart: lineStart,
                    lineEnd: lineEnd
                };
            }
            return block[pageNumber].pageDom;
        }

        function setIdToFileBlockPage(dom, file, blockNumber, pageNumber, typeString) {
            dom.id = file.name + "-" + typeString + "-" + blockNumber + "-" + pageNumber;
        }

        /**
         * Get the page of line numbers for the page. Assumes that the block has
         * alredy been fetched from the server.
         */
        function numbersForPageDom(file, blockNumber, pageNumber) {
            // Ensure the raw page data has been eval'd.
            pageDom(file, blockNumber, pageNumber);
            return file.blocks[blockNumber][pageNumber].numbersDom;
        }

        function revisionBlameForPageDom(file, blockNumber, pageNumber) {
            // Ensure the raw page data has been eval'd.
            pageDom(file, blockNumber, pageNumber);
            return file.blocks[blockNumber][pageNumber].revBlameDom;
        }

        function authorBlameForPageDom(file, blockNumber, pageNumber) {
            // Ensure the raw page data has been eval'd.
            pageDom(file, blockNumber, pageNumber);
            return file.blocks[blockNumber][pageNumber].authBlameDom;
        }

        function gutterDecoratorForPageDom(file, blockNumber, pageNumber, gutterIndex) {
            // Ensure the raw page data has been eval'd.
            pageDom(file, blockNumber, pageNumber);
            return file.blocks[blockNumber][pageNumber].gutterDom[gutterIndex];
        }

        /**
         * Create a detached page DOM element.
         */
        function createPageDom(lines) {
            var pageDom = document.createElement('div');
            pageDom.className = 'page';
            for (var i = 0, len = lines.length; i < len; i++) {
                pageDom.appendChild(createLineDom(lines[i]));
            }
            return pageDom;
        }

        /**
         * Create a detached page of line numbers DOM element.
         */
        var createNumbersForPageDom = (function () {
            var location = window.top.location;
            var baseUrl = location.href.replace(location.hash, '');

            return function (file, blockNumber, pageNumber, lines) {
                var numbersDom = document.createElement('div');
                var lineOffset = (blockNumber * file.pagesPerBlock + pageNumber) * file.linesPerPage + 1;

                numbersDom.className = 'page line-numbers';
                for (var i = 0, len = lines.length; i < len; i++) {
                    var lineNumber = lineOffset + i;
                    var lineUrl = baseUrl + '#' + file.name + lineNumber;
                    var lineDom = createLineDom(lines[i], '<a href="' + lineUrl + '" target="_blank">' + lineNumber + '</a>');

                    numbersDom.appendChild(lineDom);
                }
                return numbersDom;
            };
        })();

        var createRevisionBlameForPageDom = (function () {
            return function (file, blockNumber, pageNumber, lines) {
                var blameDom = document.createElement('div');
                blameDom.className = 'page revision-blame';
                for (var i = 0, len = lines.length; i < len; i++) {
                    var html = lines[i].start && lines[i].revision || '&nbsp;';
                    var lineDom = createLineDom(lines[i], '<span>' + html + '</span>', [lines[i]['blameclass']]);
                    blameDom.appendChild(lineDom);
                }
                return blameDom;
            };
        })();

        var createAuthorBlameForPageDom = (function () {
            return function (file, blockNumber, pageNumber, lines) {
                var blameDom = document.createElement('div');
                blameDom.className = 'page author-blame';
                for (var i = 0, len = lines.length; i < len; i++) {
                    var html = lines[i].start && lines[i].author || '&nbsp;';
                    var lineDom = createLineDom(lines[i], '<span>' + html + '</span>');
                    blameDom.appendChild(lineDom);
                }
                return blameDom;
            };
        })();

        var createGutterDecoratorForPageDom = (function () {
            return function (file, blockNumber, pageNumber, lines) {
                var gutterDom = [];
                for (var gutterIndex = 0; gutterIndex < lines[0].gutters.length; ++gutterIndex) {
                    gutterDom[gutterIndex] = document.createElement('div');
                    for (var i = 0, len = lines.length; i < len; i++) {
                        var lineDom = createLineDom(lines[i], lines[i].gutters[gutterIndex]);
                        gutterDom[gutterIndex].appendChild(lineDom);
                    }
                }
                return gutterDom;
            };
        })();

        /**
         * Create a detached line DOM element.
         *
         * @param line object with at least an html property that contains the
         *      html representation of the line (i.e., with spans for syntax
         *      highlighting and ediff, etc.)
         * @param content optional content to use instead of line.html
         */
        var createLineDom = (function () {
            var codes = ['addition', 'change', 'deletion', 'start', 'end'];
            var createElem = null;

            if ($.browser.msie) {
                // IE work around for quirk of normalizing white-space on assignment to innerHTML.
                var container = document.createElement('div');
                createElem = function (content) {
                    var c = container;
                    c.innerHTML = '<pre>' + content + '</pre>';
                    return c.removeChild(c.firstChild);
                };
            } else {
                createElem = function (content) {
                    var lineDom = document.createElement('pre');
                    lineDom.innerHTML = content;
                    return lineDom;
                };
            }

            return function (line, content, extraClasses) {
                var lineDom = createElem(content || line.html);
                // Must use typeof because the empty string is falsy.
                if ((typeof lineDom.innerText !== 'undefined' ? lineDom.innerText : lineDom.textContent).length === 0) {
                    lineDom.innerHTML = lineDom.innerHTML + '&nbsp;';
                }

                var cssClasses = [];
                for (var i = 0, len = codes.length; i < len; i++) {
                    var code = codes[i];
                    if (line.hasOwnProperty(code)) {
                        cssClasses.push(code);
                    }
                }
                extraClasses = extraClasses || [];
                for (i = 0, len = extraClasses.length; i < len; i++) {
                    cssClasses.push(extraClasses[i]);
                }
                var cssClass = cssClasses.join(' ');
                if (cssClass) {
                    lineDom.className = cssClass;
                }

                return lineDom;
            };
        })();

        /**
         * Get the DOM element for the longest line in this file.
         */
        prototype.longestLineDom = function () {
            return createLineDom(this.longestLine);
        };

        return prototype;
    })();


// Navigation can't do anything until the DOM is ready.
    window.nextSegment = function () {
    };
    window.prevSegment = function () {
    };

// This is used for annotations, not just diffs
    window.FECRU = window.FECRU || {};
    FECRU.SBS = FECRU.SBS || {};
    FECRU.SBS.main = function (rawDiffData) {

        var diff = new Diff(rawDiffData);
        var $fromBox = null;
        var $toBox = null;
        var LINE_HEIGHT = null;
        var FOCUSED_LINE_OFFSET = null;

        var lastWidth = 0;
        var lastHeight = 0;
        var emsSet = false;
        var fromNumbersOuterWidth = 0;
        var fromRevisionsOuterWidth = 0;
        var fromAuthorsOuterWidth = 0;
        var segmentsOuterWidth = 0;
        var toNumbersOuterWidth = 0;
        var toRevisionsOuterWidth = 0;
        var toAuthorsOuterWidth = 0;
        var gutterOuterWidths = [];

        /*eslint-disable complexity*/
        function setPaneDimensions(forceResize) {
            var $fromNumbers = $('#from-numbers-box');
            var $toNumbers = $('#to-numbers-box');
            var $fromRevisions = $('#from-revision-blame-box');
            var $toRevisions = $('#to-revision-blame-box');
            var $fromAuthors = $('#from-auth-blame-box');
            var $toAuthors = $('#to-auth-blame-box');
            var $segments = $('#segment-box');
            var $container = $(window.frameElement || window);
            var containerWidth = $container.width();
            var containerHeight = $container.height();

            if (!forceResize && containerWidth === lastWidth && containerHeight === lastHeight) {
                // dont do it again
                return;
            }
            lastWidth = containerWidth;
            lastHeight = containerHeight;

            var paneHeight = lastHeight;
            if (!emsSet) {
                emsSet = true;
                var numbersWidthEm = Math.max(2.5, 0.75 * String(Math.max(diff.from.numLines, diff.to.numLines)).length);
                var fromRevisionsWidthEm = 4.5;
                var toRevisionsWidthEm = 4.5;
                var fromAuthorsWidthEm = 9;
                var toAuthorsWidthEm = 9;

                $fromNumbers.add($toNumbers).width(numbersWidthEm + 'em');

                var setWidth = function($column, widthEm) {
                    $column.width(widthEm + 'em');
                };

                setWidth($fromRevisions, fromRevisionsWidthEm);
                setWidth($toRevisions, toRevisionsWidthEm);
                setWidth($fromAuthors, fromAuthorsWidthEm);
                setWidth($toAuthors, toAuthorsWidthEm);

                forEachGutterBox(function (i, $gutterBox) {
                    $gutterBox.width(diff.to.gutterWidths[i] + 'em');
                    gutterOuterWidths[i] = $gutterBox.outerWidth(true);
                });

                fromNumbersOuterWidth = $fromNumbers.outerWidth(true);
                fromRevisionsOuterWidth = $fromRevisions.outerWidth(true);
                fromAuthorsOuterWidth = $fromAuthors.outerWidth(true);
                segmentsOuterWidth = $segments.outerWidth(true);
                toNumbersOuterWidth = $toNumbers.outerWidth(true);
                toRevisionsOuterWidth = $toRevisions.outerWidth(true);
                toAuthorsOuterWidth = $toAuthors.outerWidth(true);
            }

            $fromNumbers
                .add($toNumbers)
                .add($fromRevisions)
                .add($toRevisions)
                .add($fromAuthors)
                .add($toAuthors)
                .add($segments)
                .add(".gutter-box")
                .height(paneHeight);

            var fromWidth = 0;
            if ($fromNumbers.is(':visible')) {
                fromWidth += fromNumbersOuterWidth;
            }
            if ($fromRevisions.is(':visible')) {
                fromWidth += fromRevisionsOuterWidth;
            }
            if ($fromAuthors.is(':visible')) {
                fromWidth += fromAuthorsOuterWidth;
            }

            var segmentsWidth = 0;
            if ($segments.is(':visible')) {
                segmentsWidth += segmentsOuterWidth;
            }

            var toWidth = 0;
            if ($toNumbers.is(':visible')) {
                toWidth += toNumbersOuterWidth;
            }
            if ($toRevisions.is(':visible')) {
                toWidth += toRevisionsOuterWidth;
            }
            if ($toAuthors.is(':visible')) {
                toWidth += toAuthorsOuterWidth;
            }

            forEachGutterBox(function (i, $gutterBox) {
                toWidth += gutterOuterWidths[i];
            });

            var $fileBoxes = $fromBox.add($toBox);

            $fileBoxes.height(paneHeight);

            var availableWidth = lastWidth - fromWidth - segmentsWidth - toWidth - 2;

            var fromNewWidth = Math.floor(availableWidth / $fileBoxes.length);
            var toNewWidth = Math.ceil(availableWidth / $fileBoxes.length);

            //set the real width
            $fromBox.width(fromNewWidth);
            $toBox.width(toNewWidth);

            var getFromPageWidth = $fromBox.data('getPageWidth');
            if (getFromPageWidth) {
                $fromBox.children().children().css('width', getFromPageWidth());
            }

            var getToPageWidth = $toBox.data('getPageWidth');
            if (getToPageWidth) {
                $toBox.children().children().css('width', getToPageWidth());
            }

            if (!LINE_HEIGHT) {
                var lineHtml = '<pre>        <span class="hl_comment">&lt;!-- comment --&gt;</span> <span class="hl_starttag">&lt;tag <span class="hl_attrib">key</span>=<span class="hl_string">"<span class="ediffChangedB">value</span>!default"</span></span></pre>';

                //a single line wasn't accurate enough and caused overlapping pages.
                //since we're only doing it once, this isn't so bad.
                var $controlLine = $(new Array(diff.linesPerPage + 2).join(lineHtml));
                var $dummyPage = $('<div class="page">').append($controlLine);

                $toBox.children('.content-box')
                    .append($dummyPage);
                // outerHeight() only gives integral accuracy, so use offset().top
                LINE_HEIGHT = ($dummyPage.children().last().offset().top - $dummyPage.children().first().offset().top) / diff.linesPerPage;
                $toBox.children('.content-box')
                    .empty();
            }

            var numViewableLines = Math.ceil(paneHeight / LINE_HEIGHT);
            FOCUSED_LINE_OFFSET = Math.floor(numViewableLines / 3) - 1;

            $fromBox.toggleClass('all-lines-visible', diff.from.numLines < numViewableLines);
            $toBox.toggleClass('all-lines-visible', diff.to.numLines < numViewableLines);
        }

        /*eslint-enable*/

        FECRU.SBS.resize = setPaneDimensions;
        FECRU.SBS.getDiff = function () {
            return diff;
        };

        FECRU.SBS.simulateScroll = function () {
            $toBox.trigger('scroll');
            $fromBox.trigger('scroll');
        };

        function initDimensions(file, $scrollBox) {
            var $longestLine = $(file.longestLineDom());
            var SCROLLBOX_PADDING_BOTTOM = 30;

            $scrollBox.children('.content-box')
                .append($('<div class="page">').append($longestLine));

            file.lineHeightPx = LINE_HEIGHT;

            var longestLineWidth = $longestLine.outerWidth();
            var getPageWidth = function () {
                return Math.max($scrollBox.width(), longestLineWidth)
            };
            var width = getPageWidth();
            var height = file.numLines * LINE_HEIGHT;

            $scrollBox
                .children('.content-box')
                .height(height + SCROLLBOX_PADDING_BOTTOM)
                .end()
                .scrollTop(height)
                .scrollLeft(width);

            $scrollBox
                .data('name', file.name)
                .data('getPageWidth', getPageWidth)
                .data('maxScrollTop', $scrollBox.scrollTop())
                .data('maxScrollLeft', $scrollBox.scrollLeft())
                .data('prevScrollTop', 0)
                .data('prevScrollLeft', 0);

            $scrollBox
                .scrollTop(0)
                .scrollLeft(0)
                .children('.content-box')
                .empty();

            // Add some buffer to account for horizontal scrollbars that overlay the last line.
            $('#' + file.name + '-numbers-box').children('.content-box')
                .height(height + 30);
            $('#' + file.name + '-revision-blame-box').children('.content-box')
                .height(height + 30);
            $('#' + file.name + '-auth-blame-box').children('.content-box')
                .height(height + 30);
        }

        function lineTopPx(line) {
            return (line - 1) * LINE_HEIGHT;
        }

        function lineBottomPx(line) {
            return line * LINE_HEIGHT;
        }

        var ignoreScrolls = 0;

        function makeScrollSynchHandler(numLines, otherPaneLine, otherPane) {
            return function () {
                var $thisPane = $(this);
                var $otherPane = $(otherPane);
                var scrollTop = $thisPane.scrollTop();
                var scrollLeft = $thisPane.scrollLeft();
                var isDownScroll = $thisPane.data('prevScrollTop') < scrollTop;

                $thisPane
                    .data('prevScrollTop', scrollTop)
                    .data('prevScrollLeft', scrollLeft);

                $('#' + $thisPane.data('name') + '-numbers-box').scrollTop(scrollTop);
                $('#' + $thisPane.data('name') + '-revision-blame-box').scrollTop(scrollTop);
                $('#' + $thisPane.data('name') + '-auth-blame-box').scrollTop(scrollTop);
                $('.gutter-box').scrollTop(scrollTop);

                if (ignoreScrolls) {
                    ignoreScrolls -= 1;
                    return;
                }

                var firstVisibleLine = Math[isDownScroll ? 'ceil' : 'floor'](scrollTop / LINE_HEIGHT) + 1;
                var residue = scrollTop - lineTopPx(firstVisibleLine);

                var currentFocusedLine = Math.min(numLines, firstVisibleLine + FOCUSED_LINE_OFFSET);
                var previousFocusedLine = Math.min(numLines, firstVisibleLine + (isDownScroll ? -1 : 1) + FOCUSED_LINE_OFFSET);
                var keepOtherPaneStill = otherPaneLine[previousFocusedLine] === otherPaneLine[currentFocusedLine];

                if ($otherPane.length !== 0) {
                    // Need to cap the scroll positions since requesting a position larger than
                    // physically possible only scrolls as far as the maximum possible, and no less than 0...
                    var otherPaneScrollTop = lineTopPx(otherPaneLine[currentFocusedLine] - FOCUSED_LINE_OFFSET) + (keepOtherPaneStill ? 0 : residue);
                    if (otherPaneScrollTop < 0) {
                        otherPaneScrollTop = 0;
                    } else if (otherPaneScrollTop > $otherPane.data('maxScrollTop')) {
                        otherPaneScrollTop = $otherPane.data('maxScrollTop');
                    }
                    var otherPaneScrollLeft = Math.min($otherPane.data('maxScrollLeft'), scrollLeft);

                    // ...and need to make sure the scroll position will change, otherwise the
                    // scroll event handler won't fire.
                    if ($otherPane.data('prevScrollTop') !== otherPaneScrollTop) {
                        ignoreScrolls++;
                        $otherPane.scrollTop(otherPaneScrollTop);
                    }
                    if ($otherPane.data('prevScrollLeft') !== otherPaneScrollLeft) {
                        ignoreScrolls++;
                        $otherPane.scrollLeft(otherPaneScrollLeft);
                    }
                }

                renderDiffSegments();
            };
        }

        var anchoredHunk = null;
        var renderDiffSegments = (function () {

            var canvas = null;
            var $segmentBox = null;
            var visiblePaths = [];
            var anchoredPath = null;

            var createPath = (function () {
                var add = {stroke: '#ccc', fill: '#cfc'};
                var del = {stroke: '#ccc', fill: '#ffc0cb'};
                var mod = {stroke: '#ccc', fill: '#f4f4f4'};

                return function (hunk) {
                    var path = canvas.path();
                    if (hunk.isAddition()) {
                        path.attr(add);
                    } else if (hunk.isChange()) {
                        path.attr(mod);
                    } else if (hunk.isDeletion()) {
                        path.attr(del);
                    }
                    return path;
                };
            })();

            /*eslint-disable complexity*/
            return function () {
                $segmentBox = $segmentBox || $('#segment-box');
                canvas = canvas || ($segmentBox.length === 0 ? null : Raphael('segment-box'));

                var fromScrollTop = $fromBox.scrollTop();
                var fromOffsetLines = Math.floor(fromScrollTop / LINE_HEIGHT);
                var fromResidue = fromScrollTop - lineTopPx(fromOffsetLines + 1);

                var toScrollTop = $toBox.scrollTop();
                var toOffsetLines = Math.floor(toScrollTop / LINE_HEIGHT);
                var toResidue = toScrollTop - lineTopPx(toOffsetLines + 1);

                var toBoxHeight = $toBox.height();

                var numViewableLines = Math.ceil(toBoxHeight / LINE_HEIGHT);
                var viewableFromRange = Range.ofLength(fromOffsetLines + 1, numViewableLines);
                var viewableToRange = Range.ofLength(toOffsetLines + 1, numViewableLines);
                var viewport = new Hunk(viewableFromRange, viewableToRange);
                var width = $segmentBox.length === 0 ? 0 : $segmentBox.width();
                var visibleHunks = viewport.overlapping(diff.hunks);

                diff.ensureVisible(viewport);

                // Clear the canvas of previously drawn paths.
                for (var i = 0, len = visiblePaths.length; i < len; i++) {
                    visiblePaths[i].hide();
                }
                visiblePaths = [];
                if (anchoredPath) {
                    anchoredPath.hide();
                }

                // Draw the newly visible paths.
                for (i = 0, len = visibleHunks.length; i < len; i++) {
                    var hunk = visibleHunks[i];
                    var fromRange = hunk.fromRange;
                    var toRange = hunk.toRange;
                    var path = hunk.path || (hunk.path = createPath(hunk));

                    visiblePaths.push(path);
                    if (hunk.isAddition()) {
                        path.attr('path', [
                            ['M', -1, lineBottomPx(fromRange.low - fromOffsetLines) - fromResidue - 1],
                            ['L', width, lineTopPx(toRange.low - toOffsetLines) - toResidue],
                            ['V', lineBottomPx(toRange.high - toOffsetLines) - toResidue - 1],
                            ['Z']
                        ]).show();
                    } else if (hunk.isChange()) {
                        path.attr('path', [
                            ['M', -1, lineTopPx(fromRange.low - fromOffsetLines) - fromResidue],
                            ['L', width, lineTopPx(toRange.low - toOffsetLines) - toResidue],
                            ['V', lineBottomPx(toRange.high - toOffsetLines) - toResidue - 1],
                            ['L', -1, lineBottomPx(fromRange.high - fromOffsetLines) - fromResidue - 1],
                            ['Z']
                        ]).show();
                    } else if (hunk.isDeletion()) {
                        path.attr('path', [
                            ['M', -1, lineTopPx(fromRange.low - fromOffsetLines) - fromResidue],
                            ['V', lineBottomPx(fromRange.high - fromOffsetLines) - fromResidue - 1],
                            ['L', width, lineBottomPx(toRange.low - toOffsetLines) - toResidue - 1],
                            ['Z']
                        ]).show();
                    }
                }

                if (anchoredHunk) {
                    anchoredPath = anchoredPath || (canvas ? canvas.path().attr({
                            fill: '#fffff0',
                            stroke: '#fffff0',
                            opacity: 0.75
                        }) : null);
                    if (anchoredHunk.overlaps(viewport)) {
                        fromRange = anchoredHunk.fromRange;
                        toRange = anchoredHunk.toRange;
                        anchoredPath && anchoredPath.attr('path', [
                            ['M', -1, lineTopPx(fromRange.low - fromOffsetLines) - fromResidue],
                            ['L', width, lineTopPx(toRange.low - toOffsetLines) - toResidue],
                            ['V', lineBottomPx(toRange.high - toOffsetLines) - toResidue - 1],
                            ['L', -1, lineBottomPx(fromRange.high - fromOffsetLines) - fromResidue - 1],
                            ['Z']
                        ]).show().toFront();
                    }
                }

            };
            /*eslint-enable*/
        })();

        function hunkAfterLine(sortedHunks, fromLineNumber) {
            var min = 0;
            var max = sortedHunks.length - 1;
            var after = null;

            while (min <= max) {
                var mid = Math.floor((min + max) / 2);
                var node = sortedHunks[mid];

                if (fromLineNumber < node.fromRange.low) {
                    after = node;
                    max = mid - 1;
                } else {
                    min = mid + 1;
                }
            }
            return after;
        }

        function hunkBeforeLine(sortedHunks, fromLineNumber) {
            var min = 0;
            var max = sortedHunks.length - 1;
            var before = null;

            while (min <= max) {
                var mid = Math.floor((min + max) / 2);
                var node = sortedHunks[mid];

                if (node.fromRange.low < fromLineNumber) {
                    before = node;
                    min = mid + 1;
                } else {
                    max = mid - 1;
                }
            }
            return before;
        }

        function setAnchoredFromLine(fromLineNum, animate) {
            if (diff.from.exists()) {
                setAnchoredLines(fromLineNum, diff.from.position[fromLineNum]);
                scrollToFocusedFromLine(fromLineNum, animate);
            }
        }

        function setAnchoredToLine(toLineNum, animate) {
            setAnchoredLines(diff.to.position[toLineNum], toLineNum);
            scrollToFocusedToLine(toLineNum, animate);
        }

        var setAnchoredLines = (function () {
            var $from = null;
            var $to = null;

            return function (fromLineNum, toLineNum) {
                var linesPerBlock = diff.pagesPerBlock * diff.linesPerPage;
                var fromLineNumWithinPage = (fromLineNum - 1) % linesPerBlock % diff.linesPerPage;
                var fromPageNum = Math.floor((fromLineNum - 1) % linesPerBlock / diff.linesPerPage);
                var fromBlockNum = Math.floor((fromLineNum - 1) / linesPerBlock);
                var toLineNumWithinPage = (toLineNum - 1) % linesPerBlock % diff.linesPerPage;
                var toPageNum = Math.floor((toLineNum - 1) % linesPerBlock / diff.linesPerPage);
                var toBlockNum = Math.floor((toLineNum - 1) / linesPerBlock);

                $from && $from.removeClass('anchored-line');
                $to && $to.removeClass('anchored-line');

                anchoredHunk = Hunk.ofLengths(fromLineNum, 1, toLineNum, 1);
                diff.ensureVisible(anchoredHunk, function () {
                    if (diff.from.exists()) {
                        $from =
                            $('#from-page-' + fromBlockNum + '-' + fromPageNum).children('pre:eq(' + fromLineNumWithinPage + ')').add(
                                $('#from-numbers-' + fromBlockNum + '-' + fromPageNum).children('pre:eq(' + fromLineNumWithinPage + ')'));
                        $from.addClass('anchored-line');
                    }
                    $to =
                        $('#to-page-' + toBlockNum + '-' + toPageNum).children('pre:eq(' + toLineNumWithinPage + ')').add(
                            $('#to-numbers-' + toBlockNum + '-' + toPageNum).children('pre:eq(' + toLineNumWithinPage + ')'));
                    $to.addClass('anchored-line');
                });
            };
        })();

        function scrollToFocusedFromLine(lineNum, animate) {
            if (diff.from.exists()) {
                var location = window.top.location;
                lineNum = Math.min(Math.max(1, lineNum), diff.from.numLines - 1);
                location.hash = '#from' + lineNum;
                var scrollTop = lineTopPx(lineNum - FOCUSED_LINE_OFFSET);
                var toBoxScrollTop = lineTopPx(diff.from.position[lineNum] - FOCUSED_LINE_OFFSET);
                if (animate) {
                    animateScrollTo($fromBox, scrollTop, toBoxScrollTop);
                } else {
                    syncMainPageScrollBar(toBoxScrollTop);
                    $fromBox.scrollTop(scrollTop);
                }
            }
        }

        function scrollToFocusedToLine(lineNum, animate) {
            var location = window.top.location;
            lineNum = Math.min(Math.max(1, lineNum), diff.to.numLines - 1);
            location.hash = '#to' + lineNum;
            var targetLine = lineNum - FOCUSED_LINE_OFFSET;
            var scrollTop = lineTopPx(targetLine < 0 ? 1 : targetLine);
            if (animate) {
                animateScrollTo($toBox, scrollTop);
            } else {
                syncMainPageScrollBar(scrollTop);
                $toBox.scrollTop(scrollTop);
            }
        }

        function syncMainPageScrollBar(scrollTop) {
            triggerOnParent('scroll-content', scrollTop);
        }

        function animateScrollTo($scrollBox, scrollTop, toBoxScrollTop) {
            var TOTAL_DURATION = 100;
            var SLEEP_DURATION = 10;
            var steps = TOTAL_DURATION / SLEEP_DURATION;
            var currentScrollTop = $scrollBox.data('prevScrollTop');
            var stepSize = (scrollTop - currentScrollTop) / steps;

            if (typeof toBoxScrollTop !== 'undefined') {
                var toBoxCurrentScrollTop = $toBox.data('prevScrollTop');
                var toBoxStepSize = (toBoxScrollTop - toBoxCurrentScrollTop) / steps;
            } else {
                toBoxScrollTop = scrollTop;
                toBoxCurrentScrollTop = currentScrollTop;
                toBoxStepSize = stepSize;
            }

            (function loop() {
                if (steps > 0) {
                    // Scroll to the exact final location so that other line calculations are accurate.
                    currentScrollTop += stepSize;
                    toBoxCurrentScrollTop += toBoxStepSize;

                    var nextScrollTop = steps === 1 ? scrollTop : currentScrollTop;
                    var nextToBoxScrollTop = steps === 1 ? toBoxScrollTop : toBoxCurrentScrollTop;

                    syncMainPageScrollBar(nextToBoxScrollTop);
                    $scrollBox.scrollTop(nextScrollTop);

                    steps -= 1;
                    setTimeout(loop, SLEEP_DURATION);
                }
            })();
        }

        $(document).ready(function () {
            $fromBox = $('#from-box');
            $toBox = $('#to-box');

            if (window.top.AJS) {
                window.top.AJS.bind('file-view-resize', function () {
                    FECRU.SBS.resize();
                });
            } else {
                $(window).resize(function () {
                    FECRU.SBS.resize();
                })
            }

            setPaneDimensions(true);
            // IE8 exhibits some race condition on the initial load that could cause it not ot resize properly.
            // Resizing it AFTER document.ready seems to workaround it until we determine the race condition.
            setTimeout(function () {
                setPaneDimensions(true);
            }, 1);

            if (diff.from.exists()) {
                initDimensions(diff.from, $fromBox);
            }
            initDimensions(diff.to, $toBox);

            // trigger event to the parent window to reset the iframe height
            triggerOnParent('file-view-resize-from-iframe');

            if (diff.from.exists()) {
                $fromBox.bind('scroll', makeScrollSynchHandler(diff.from.numLines, diff.from.position, $toBox[0]));
            }
            $toBox.bind('scroll', makeScrollSynchHandler(diff.to.numLines, diff.to.position, $fromBox[0]));
            renderDiffSegments();

            window.nextSegment = function () {
                var fromLineNumber = Math.floor($fromBox.data('prevScrollTop') / LINE_HEIGHT) + 1 + FOCUSED_LINE_OFFSET;
                var hunk = hunkAfterLine(diff.hunks, fromLineNumber);

                scrollToFocusedFromLine(hunk ? (diff.from.exists() ? hunk.fromRange.low : hunk.toRange.low) : (diff.from.exists() ? diff.from.numLines : diff.to.numLines) - 1);
            };

            window.prevSegment = function () {
                var fromLineNumber = Math.floor($fromBox.data('prevScrollTop') / LINE_HEIGHT) + 1 + FOCUSED_LINE_OFFSET;
                var hunk = hunkBeforeLine(diff.hunks, fromLineNumber);

                scrollToFocusedFromLine(hunk ? (diff.from.exists() ? hunk.fromRange.low : hunk.toRange.low) : 1);
            };

            // Scroll to requested position, otherwise the first hunk.
            var hash = window.top.location.hash;
            var lineNum = null;

            if (/^#seg\d+/.test(hash)) {
                // Support legacy #segN URL anchors to scroll to a specific hunk.
                var segment = parseInt(hash.replace(/^#seg(\d+).*$/, '$1'), 10);
                if (segment <= 0) {
                    lineNum = 1;
                } else if (segment > diff.hunks.length) {
                    lineNum = diff.from.numLines - 1;
                } else {
                    lineNum = diff.hunks[segment - 1].fromRange.low;
                }
                setAnchoredFromLine(lineNum);
            } else if (/^#(l|from)\d+/.test(hash)) {
                // Also support legacy #lN URL anchors to scroll to a specific line.
                lineNum = parseInt(hash.replace(/^#(l|from)(\d+).*$/, '$2'), 10);
                setAnchoredFromLine(lineNum);
            } else if (/^#to\d+/.test(hash)) {
                lineNum = parseInt(hash.replace(/^#to(\d+).*$/, '$1'), 10);
                // Workaround for Firefox scrolling the 'to' pane then giving
                // focus to the top of the 'from' pane.
                setTimeout(function () {
                    setAnchoredToLine(lineNum);
                }, 10);
            }

            // Scroll to the specified link on left-clicking the permalink.
            $('#from-numbers-box').delegate('a', 'click', function (e) {
                if (e.button === 0) {
                    var matches = /#from(\d+)$/.exec($(this).attr('href'));
                    var lineNumber = parseInt(matches[1], 10);

                    setAnchoredFromLine(lineNumber, true);
                    e.preventDefault();
                }
            });
            $('#to-numbers-box').delegate('a', 'click', function (e) {
                if (e.button === 0) {
                    var matches = /#to(\d+)$/.exec($(this).attr('href'));
                    var lineNumber = parseInt(matches[1], 10);

                    setAnchoredToLine(lineNumber, true);
                    e.preventDefault();
                }
            });

        });
    };

    function triggerOnParent(event, data) {
        var parent = window.parent;
        if (parent !== window) {
            parent.AJS && parent.AJS.trigger(event, data);
        }
    }

})(typeof AJS === 'undefined' ? jQuery : AJS.$);
/*[{!sbs_diff_js_dr0w52g!}]*/
