/**
 * A closed range of the diff line grid.
 *
 * A range will normally span whole lines, however, the concept of 'points'
 * on the line grid are required to represent and render addition and
 * deletion hunks.
 *
 * <pre>
 *   point   line      range
 *              0*
 *   [0,-1]  +-----     -+
 *           |  1        | [1, 1]
 *   [1, 0]  +-----     -+         -+
 *           |  2                   |
 *   [2, 1]  +-----                 | [2, 3]
 *           |  3                   |
 *   [3, 2]  +-----                -+
 * </pre>
 *
 * *: line zero is a special case
 *
 * @see http://www.artima.com/weblogs/viewpost.jsp?thread=164293
 *
 * @param low start of the range (inclusive)
 * @param high end of the range (inclusive); if one less than <em>low</em>
 *      then represents a point
 */
function Range(low, high) {
    if (!(this instanceof Range)) {
        return new Range(low, high);
    }
    this.low = low;
    this.high = high;
    this.length = high - low + 1;
    return this;
}

Range.ofLength = function (line, length) {
    return new Range(line, line + length - 1);
};

Range.ofLine = function (line) {
    return Range.ofLength(line, 1);
};

Range.ofLines = function (firstLine, lastLine) {
    return new Range(firstLine, lastLine);
};

Range.pointAfterLine = function (line) {
    return new Range(line, line - 1);
};

Range.pointBeforeLine = function (line) {
    return Range.pointAfterLine(line - 1);
};

Range.prototype = {
    constructor: Range,
    before: function (range) {
        return this.high < range.low;
    },
    after: function (range) {
        return range.high < this.low;
    },
    overlaps: function (range) {
        return this.low <= range.high && range.low <= this.high;
    },
    equals: function (range) {
        return this.low === range.low &&
            this.high === range.high;
    },
    toString: function () {
        return '[' + this.low + ', ' + this.high + ']';
    }
};


function Hunk(fromRange, toRange) {
    if (!(this instanceof Hunk)) {
        return new Hunk(fromRange, toRange);
    }
    this.fromRange = fromRange;
    this.toRange = toRange;
    return this;
}

Hunk.ofLengths = function (fromStart, fromLength, toStart, toLength) {
    return new Hunk(
        Range.ofLength(fromStart, fromLength),
        Range.ofLength(toStart, toLength)
    );
};

/**
 * Convenience constructor for addition hunks in normal diff output
 * format (e.g., <code>11a11,13</code>).
 *
 * @see http://www.gnu.org/software/diffutils/manual/html_node/Detailed-Normal.html
 */
Hunk.addition = function (afterFromLine, toRange) {
    return new Hunk(Range.pointAfterLine(afterFromLine), toRange);
};

/**
 * Convenience constructor for change hunks in normal diff output
 * format (e.g., <code>4,5c2,3</code>).
 *
 * @see http://www.gnu.org/software/diffutils/manual/html_node/Detailed-Normal.html
 */
Hunk.change = function (fromRange, toRange) {
    return new Hunk(fromRange, toRange);
};

/**
 * Convenience constructor for deletion hunks in normal diff output
 * format (e.g., <code>1,2d0</code>).
 *
 * @see http://www.gnu.org/software/diffutils/manual/html_node/Detailed-Normal.html
 */
Hunk.deletion = function (fromRange, afterToLine) {
    return new Hunk(fromRange, Range.pointAfterLine(afterToLine));
};

Hunk.prototype = (function () {
    var prototype = Hunk.prototype;

    prototype.isAddition = function () {
        return this.fromRange.length === 0;
    };

    prototype.isChange = function () {
        return this.fromRange.length > 0 && this.toRange.length > 0;
    };

    prototype.isDeletion = function () {
        return this.toRange.length === 0;
    };

    prototype.before = function (hunk) {
        return this.fromRange.before(hunk.fromRange) &&
            this.toRange.before(hunk.toRange);
    };

    prototype.after = function (hunk) {
        return this.fromRange.after(hunk.fromRange) &&
            this.toRange.after(hunk.toRange);
    };

    prototype.overlaps = function (hunk) {
        return this.fromRange.overlaps(hunk.fromRange) ||
            this.toRange.overlaps(hunk.toRange);
    };

    /**
     * Returns the set of hunks that overlap this hunk.
     */
    prototype.overlapping = function (sortedHunks) {
        var result = [];
        var index = binarySearch(sortedHunks, this);

        if (index >= 0) {
            var h;
            var i;
            var len;

            for (i = index - 1; i >= 0 && (h = sortedHunks[i]).overlaps(this); i -= 1) {
                result.push(h);
            }
            result.push(sortedHunks[index]);
            for (i = index + 1, len = sortedHunks.length; i < len && (h = sortedHunks[i]).overlaps(this); i++) {
                result.push(h);
            }
        }
        return result;
    };

    /**
     * Returns the index of a hunk that overlaps the given hunk.
     */
    function binarySearch(sortedHunks, hunk) {
        var min = 0;
        var max = sortedHunks.length - 1;

        while (min <= max) {
            var mid = Math.floor((min + max) / 2);
            var node = sortedHunks[mid];

            if (hunk.overlaps(node)) {
                return mid;
            } else if (hunk.before(node)) {
                max = mid - 1;
            } else if (hunk.after(node)) {
                min = mid + 1;
            } else {
                //bad hunks - either NaN values, or a mismatched from and to.
                return -1;
            }
        }
        return -1;
    }

    prototype.equals = function (hunk) {
        return this.fromRange.equals(hunk.fromRange) &&
            this.toRange.equals(hunk.toRange);
    };

    prototype.toString = (function () {
        function toHunkRangeString(range) {
            if (range.length === 1) {
                return range.low + '';
            } else {
                return range.low + ',' + range.length;
            }
        }

        return function () {
            return '-' + toHunkRangeString(this.fromRange) + ' ' +
                '+' + toHunkRangeString(this.toRange);
        };
    })();

    return prototype;
})();
/*[{!json_hunks_js_8t5z52e!}]*/