(function ($) {

    var article = FE.ARTICLE;

    var revIdKey = 'revision-id';

    var revisionIdForRow = function ($revision) {
        return $revision.data(revIdKey);
    };

    var rowForRevisionId = function (revisionId) {
        return $("#revision-" + revisionId).closest(".revision");
    };

    // Cache the focused row as it can be too slow to do $("tr.history-focus");
    var $focusedRow;
    var focusedRevisionData;
    var blurFocusedRow = function () {
        if ($focusedRow && focusedRevisionData) {
            focusedRevisionData.focussed = false;
            $focusedRow.removeClass("history-focus");
        }
    };

    var setRowFocus = function (revisionData) {
        blurFocusedRow();

        //make the switch
        focusedRevisionData = revisionData;
        $focusedRow = rowForRevisionId(revisionData.id);

        //focus the new
        focusedRevisionData.focussed = true;
        if ($focusedRow.length) {
            $focusedRow.addClass("history-focus");
            $(window).scrollTo(rowForRevisionId(focusedRevisionData.id), {
                axis: 'y',
                offset: -50
            });
        } else {
            var pageToLoad = getPageForRevisionId(focusedRevisionData.id);
            if (pageToLoad != null) {
                loadRevisionsPage(pageToLoad, focusedRevisionData.id);
            } else {
                //TODO: Remove our js error handling from the FECRU."AJAX" namespace...
                FECRU.AJAX.appendErrorMessage("Couldn't find the revision " + revisionData.revision + ".  Try reloading.");
                FECRU.AJAX.showErrorBox();
            }
        }
    };

    var revisionDataForId = function (revisionId) {
        var arrayIndex = historyTableData.revisionKeyValueIndexMap[revisionId];
        return historyTableData.revisions[arrayIndex];
    };

    var setRevisionDataCheckedFlag = function (revision, value) {
        var revisionId = historyTableData.revisionToRevisionIdMap[revision];
        var revisionData = revisionDataForId(revisionId);

        revisionData.checked = value;
    };

    /**
     * Handle revisions selection change and update the UI according to it.
     */
    var setupDiffCheckboxes = function () {
        var $revisions = $('#history-revisions');
        var $diffButton = $('#diff-selected-button');
        var baseUrl = $diffButton.data('baseurl');
        var queue = [];

        var setRevisionInputCheckedFlag = function (revision, value) {
            $revisions.find('.revision-check[value="' + revision + '"]').prop('checked', value);
        };

        var getRevisionIndexInQueue = function (revision) {
            return queue.indexOf(revision);
        };

        var diffButtonClickHandler = function () {
            var href = $diffButton.data('href');
            if (href) {
                window.location.href = href;
            }
        };

        var revisionSelectionChangeHandler = function () {
            var revision = this.value;
            var revisionInQueueIndex = getRevisionIndexInQueue(revision);
            var isInQueue = revisionInQueueIndex !== -1;
            var toBeUnchecked = [];

            setRevisionDataCheckedFlag(revision, this.checked);

            if (this.checked) {
                queue.push(revision);
            } else if (isInQueue) {
                toBeUnchecked.push.apply(toBeUnchecked, queue.splice(revisionInQueueIndex, 1));
            }

            if (queue.length > 2) {
                toBeUnchecked.push.apply(toBeUnchecked, queue.splice(0, queue.length - 2));
            }

            toBeUnchecked.forEach(function (revision) {
                setRevisionDataCheckedFlag(revision, false);
                setRevisionInputCheckedFlag(revision, false);
            });

            var r1 = queue[0];
            var r2 = queue[1];
            var isQueueLengthValid = queue.length === 2;
            var title = isQueueLengthValid ?
                'View the diff between ' + r1 + ' and ' + r2 :
                'Select two revisions to view their diff';

            $diffButton
                .data('href', baseUrl + '?r1=' + r1 + '&r2=' + r2)
                .attr('aria-disabled', !isQueueLengthValid)
                .prop('disabled', !isQueueLengthValid)
                .prop('title', title);
        };

        $diffButton.on('click', diffButtonClickHandler);
        $revisions.on('change', '.revision-check', revisionSelectionChangeHandler);
    };

    var getRevisionFromUrl = function () {
        var hash = document.location.hash;
        if (!hash) {
            return null;
        }
        // remove the '#r' from the anchor
        return hash.substring(2);
    };

    var getRevisionIdFromUrl = function () {
        return historyTableData.revisionToRevisionIdMap[getRevisionFromUrl()];
    };

    var setupFocusedRevision = function (revisionIdToFocus) {
        if (!revisionIdToFocus) {
            return;
        }

        var revisionData = revisionDataForId(revisionIdToFocus);
        setRowFocus(revisionData);
    };

    var currentPage = null;
    var REVISIONS_PER_PAGE = 30;

    var getTotalPageCount = function () {
        return Math.ceil(historyTableData.revisions.length / REVISIONS_PER_PAGE);
    };

    /**
     * Ensures that the navigation buttons (next/previous revisions) are synchronised -- ie, that they are enabled/disabled
     * when required.
     */
    var syncPaginationButtons = function () {
        var $firstLink = $("a.pagination-first");
        var $prevLink = $("a.pagination-back");
        var $nextLink = $("a.pagination-next");
        var $lastLink = $("a.pagination-last");

        var totalPageCount = getTotalPageCount();
        var show = true;

        if (totalPageCount <= 1) {
            $firstLink.addClass("disabled");
            $prevLink.addClass("disabled");
            $nextLink.addClass("disabled");
            $lastLink.addClass("disabled");
            show = false;
        } else if (currentPage <= 1) {
            $firstLink.removeClass("disabled");
            $prevLink.removeClass("disabled");
            $nextLink.addClass("disabled");
            $lastLink.addClass("disabled");
        } else if (currentPage >= totalPageCount) {
            $firstLink.addClass("disabled");
            $prevLink.addClass("disabled");
            $nextLink.removeClass("disabled");
            $lastLink.removeClass("disabled");
        } else {
            $firstLink.removeClass("disabled");
            $prevLink.removeClass("disabled");
            $nextLink.removeClass("disabled");
            $lastLink.removeClass("disabled");
        }
        var topPagination = $("span.taskbar-pagination");
        var bottomPagination = $("div.toolbar-bottom");
        if (show) {
            topPagination.show();
            bottomPagination.show();
        } else {
            topPagination.hide();
            bottomPagination.hide();
        }

        $(".pagination-text").text(currentPage + "/" + totalPageCount);
    };

    function getPageForRevisionId(revisionId) {
        var revIndex = null;
        Array.first(historyTableData.revisions, function (revision, index) {
            if (revision.id === revisionId) {
                revIndex = index;
                return true;
            }
            return false;
        });
        if (revIndex === null) {
            return null;
        }

        return Math.floor(revIndex / REVISIONS_PER_PAGE) + 1;
    }

    /**
     * Loads revisions into the history table for display. This method will lazily load extra information from the server
     * if it isn't available locally (such as rendered commit messages).
     * @param pageNumber page to render
     */
    var loadRevisionsPage = function (pageNumber, revisionIdToFocus) {
        if (currentPage === pageNumber) {
            throw new Error("Trying to load current page.");
        }

        pageNumber = pageNumber || 1;
        currentPage = pageNumber;

        var totalPageCount = getTotalPageCount();
        if (pageNumber > totalPageCount) {
            pageNumber = totalPageCount;
        }

        blurFocusedRow();

        var allRevisions = historyTableData.revisions;
        var $historyTable = $("#history-revisions");
        $historyTable
            .removeClass("data-loaded")
            .addClass("data-loading")
            .empty();

        var startPoint = (pageNumber - 1) * REVISIONS_PER_PAGE;
        if (startPoint < 0) {
            startPoint = 0;
        }
        var endPoint = startPoint + REVISIONS_PER_PAGE;

        var revIdsToLoad = [];

        for (var c = 0, i = startPoint, revCount = allRevisions.length;
             i < endPoint && i < revCount;
             i++, c++) {
            var rev = allRevisions[i];

            if (!rev.loaded) {
                revIdsToLoad.push(rev.id);
            }
        }

        syncPaginationButtons();

        loadRevisionData(revIdsToLoad, function () {
            var $revisionElementsToInsert = $([]);

            for (var c = 0, i = startPoint, revCount = allRevisions.length;
                 i < endPoint && i < revCount;
                 i++, c++) {
                var rev = allRevisions[i];

                var $revElem = $(rev.htmlContent).data(revIdKey, rev.id);
                article.setExpanded($revElem, !rev.collapsed);

                if (rev.focussed) {
                    $revElem.addClass("history-focus");
                }

                $revisionElementsToInsert.push($revElem.get(0));
            }
            $historyTable.append($revisionElementsToInsert);

            $historyTable
                .addClass("data-loaded")
                .removeClass("data-loading");


            if (revisionIdToFocus) {
                setupFocusedRevision(revisionIdToFocus);
            }
        });
    };

    /**
     * Sends a list of revisions to the server for extra data which is required for displaying complete data for each
     * revision to the user. Upon completion of the ajax call, retrieved data is stored locally for later reuse.
     * @param revisions array of revisions to complete
     * @param onDone callback function
     */
    var loadRevisionData = function (revisions, onDone) {
        if (!revisions || revisions.length === 0) {
            onDone && onDone();
            return;
        }
        var done = function (resp) {
            var revs = resp.revisions;
            if (!revs) {
                return;
            }
            for (var i = 0, len = revs.length; i < len; i++) {
                var rev = revs[i];

                // Update the data structure
                var revisionData = revisionDataForId(rev.id);

                revisionData.htmlContent = rev.htmlContent;
                revisionData.loaded = !!revisionData.htmlContent;
            }
            onDone && onDone();
        };

        var params = {
            content: true,
            rev: revisions
        };

        var pathName = window.location.pathname;
        pathName = pathName.substring(FECRU.pageContext.length);
        var url = FECRU.pageContext + "/json" + pathName;

        FECRU.AJAX.ajaxDo(url, params, done);
    };

    /**
     * Create a map between each revision id and it's index in the {{historyTableData.revisions}} array. This method
     * should be called whenever the order of the revisions data is changed, such as on sorting.
     *
     * This data structure is used so that we can efficiently look up revisionData objects and update them as we lazily
     * retrieve expensive data from the server.
     */
    var createRevisionKeyIndexMap = function () {
        var allRevisions = historyTableData.revisions;
        var revisionKeyValueIndexMap = {};
        var revisionToRevisionIdMap = {};
        if (allRevisions) {
            for (var i = 0, len = allRevisions.length; i < len; i++) {
                var rev = allRevisions[i];
                revisionKeyValueIndexMap[rev.id] = i;
                revisionToRevisionIdMap[rev.revision] = rev.id;
            }
        }
        historyTableData.revisionKeyValueIndexMap = revisionKeyValueIndexMap;
        historyTableData.revisionToRevisionIdMap = revisionToRevisionIdMap;
    };

    var expandCollapseCallback = function ($article, isExpanded) {
        var revData = revisionDataForId(revisionIdForRow($article));
        revData.collapsed = !isExpanded;
    };

    FE.setupFileHistoryPage = function () {
        createRevisionKeyIndexMap();

        setupDiffCheckboxes();

        var pageNumberToLoad = 1; // We start at 1
        var revisionIdToLoad = getRevisionIdFromUrl();
        if (revisionIdToLoad) {
            var arrayIndexToLoad = historyTableData.revisionKeyValueIndexMap[revisionIdToLoad];
            pageNumberToLoad = Math.floor(arrayIndexToLoad / REVISIONS_PER_PAGE) + 1; // pageNumber starts at 1
        }

        loadRevisionsPage(pageNumberToLoad, revisionIdToLoad);

        $("a.pagination-back").click(function () {
            if ($(this).hasClass("disabled")) {
                return false;
            }
            loadRevisionsPage(currentPage + 1); // next page
            return false;
        });
        $("a.pagination-next").click(function () {
            if ($(this).hasClass("disabled")) {
                return false;
            }
            loadRevisionsPage(currentPage - 1); // previous page
            return false;
        });
        $("a.pagination-last").click(function () {
            if ($(this).hasClass("disabled")) {
                return false;
            }
            loadRevisionsPage(1); // last page
            return false;
        });
        $("a.pagination-first").click(function () {
            if ($(this).hasClass("disabled")) {
                return false;
            }
            var totalPageCount = getTotalPageCount();
            loadRevisionsPage(totalPageCount); // first page
            return false;
        });

        var $historyRevisions = $("#history-revisions");

        article.initialize($historyRevisions, {
            expandCollapseCallback: expandCollapseCallback
        });

        $('#expand-all-revisions').on('click', function () {
            var allRevisions = historyTableData.revisions;
            var hasRevisions = allRevisions && allRevisions.length > 0;
            if (hasRevisions) {
                var $el = $(this).toggleClass('active');
                var shouldExpand = $el.hasClass('active');
                var $articles = $historyRevisions.children('.article');

                article.setAllExpanded($articles, shouldExpand);

                allRevisions.forEach(function (rev) {
                    rev.collapsed = shouldExpand;
                });
            }
        });

        FECRU.UI.registerMetadataExpanders($historyRevisions);

        // handle IE7 z-index bug for positioned elements
        if ($.browser.msie && $.browser.version < 8) {
            $(".aui-dd-link").live("click", function () {
                var $this = $(this);

                // restore the default z-index for all the revisions
                $(".revision.ie-zindex-fixer").removeClass("ie-zindex-fixer");

                // make the parent revision above the subsequent one(s) and the dropdown above the subsequent "more" button
                var $revision = $this.closest(".revision").addClass("ie-zindex-fixer");

                // bind a IE-specific event handler to know when the dropdown becomes hidden
                // (unbind first to prevent duplicate bindings if the user clicks several times)
                var dropdown = $(this).siblings(".aui-dropdown");
                dropdown.unbind("propertychange").bind("propertychange", function () { // does not work with .live()
                    var $dropdown = $(this);
                    if ($dropdown.is(":not(:visible)")) {
                        $revision.removeClass("ie-zindex-fixer");
                        $dropdown.unbind("propertychange");
                    }
                });
            });
        }

        // setup local-link links
        $(document).delegate("a.local-link", 'click', function (e) {
            var href = $(this).attr("href");
            var revision = href.substring(href.lastIndexOf("#") + 2);
            var revisionId = historyTableData.revisionToRevisionIdMap[revision];
            var revisionData = revisionDataForId(revisionId);
            if (revisionId) {
                setRowFocus(revisionData);
            }
            e.preventDefault();
        });
    };

    // Following methods not used for the file history - move them elsewhere
    FE.fileHistoryPathLinkFn = function (event) {
        event.stopPropagation();
        return true;
    };
})(AJS.$);
/*[{!fisheye_history_js_rk6s52b!}]*/