FE.VIS.ProjectVisualiser = (function ($, templateFactory) {

    // Imports
    var VIS = FE.VIS;
    var DRAWING = VIS.DRAWING;
    var Configuration = FE.VIS.Configuration;
    var PERMALINK_THROTTLE = Configuration.PERMALINK_THROTTLE;
    var AJAX = FECRU.AJAX;
    var EventProducer = FECRU.MIXINS.EventProducer;
    var createGraphView = DRAWING.createGraphView;
    var EdgeProximityEvent = DRAWING.EdgeProximityEvent;
    var ChangesetDAG = VIS.ChangesetDAG;
    var ChangesetDialog = FE.VIS.ChangesetDialog;
    var BranchSet = VIS.BranchSet;
    var BranchHeader = VIS.BranchHeader;
    var Vector = Shapes.Vector;
    var HighlightContext = FE.VIS.HIGHLIGHTS.HighlightContext;
    var defaultPositioner = FE.VIS.POSITIONING.defaultPositioner;
    var Fragment = VIS.Fragment;

    // Private


    var defaultSettings = {
        "maxSliceSize": 50, // when loading data, it will be split into multiple slices
                           // if more than this many changesets are given.
        "orientation": 'vertical',
        "dataUrl": "",
        "metadataUrl": "",
        "scrollToCsid": null,
        "focusCsid": null,
        "preloadBuffer": 250, // if we get with this many changesets of The Unknown, load some more.
        "initialPreload": 100, // load this many changesets initially.
        "changesetsPerLoad": 250 // every request after the first, load this many changesets
    };

    // Public
    /*
     * @param target - the drawing target on which to render the changeset graph.
     * @param settings - object containing any settings you want to override.
     */
    var ProjectVisualiser = function (target, $metadataContainer, positioner, settings) {
        if (settings && settings.repositoryType === "SVN") {
            var self = this;
            this.getPrimaryBranches = function (branches) {
                return getPrimaryBranchesForSvn(self.branchSet, branches);
            };
        }
        this.target = target;
        this.positioner = positioner;
        this.$metadataContainer = $metadataContainer && $metadataContainer.length === 1 ? $metadataContainer : null;
        this.setSettings(settings);
    };

    $.extend(ProjectVisualiser.prototype, EventProducer);
    ProjectVisualiser.HighlightRegistered = "HighlightRegistered";
    ProjectVisualiser.DataLoaded = "DataLoaded";
    ProjectVisualiser.DataUnloaded = "DataUnloaded";

    ProjectVisualiser.prototype.setSettings = function (settings) {
        this.settings = settings = $.extend({}, defaultSettings, settings || {});
    };

    ProjectVisualiser.prototype.initialize = function (settings) {
        //reset the settings so that
        settings && this.setSettings(settings);

        this.initializeHighlighters();


        this.initializeData();

        if (this.branchSet.isAllInOne() || this.branchSet.getBranchNames().length) {

            this.initializeUi();
            this.doInitialLoad();
        } else {
            var self = this;
            $("#visualiser")
                .children('.dag, .metadata-wrapper')
                .addClass('hidden')
                .end()
                .append(templateFactory.load("branch-mode").toString())
                .delegate('.option-select-branches', 'click', function () {
                    $("#branch-modifier").click();
                })
                .delegate('.option-all-branches', 'click', function () {
                    self.branchSet.saveBranches(self.branchSet.getBranchNames(), true);
                })
                .delegate('.option-recently-active', 'click', function () {
                    self.branchSet.saveBranches(null, false);
                });
        }
    };

    ProjectVisualiser.prototype.initializeData = function () {
        var settings = this.settings;

        var branchSet = new BranchSet(settings.branches || [], settings.branchMode === 'ALL', settings);
        this.setBranchSet(branchSet);

        this.dag = new ChangesetDAG(this, settings, this.positioner);

        this.dataLoadingTracker = {
            reachedBeginning: false,
            reachedEnd: false,
            alreadyLoadingFuture: false,
            alreadyLoadingPast: false,
            canLoadFuture: function () {
                return !this.reachedEnd && !this.alreadyLoadingFuture;
            },
            canLoadPast: function () {
                return !this.reachedBeginning && !this.alreadyLoadingPast;
            }
        };
    };

    /**
     * Method that updates the permalink. This is debounced so that it can be
     * repeatedly invoked in an event handler without causing choking.
     */
    var updatePermalink = $.debounce(PERMALINK_THROTTLE, function (visualiser) {
        visualiser.$permalink && visualiser.$permalink.attr('href', visualiser.getPermalinkUrl());
    });

    ProjectVisualiser.prototype.initializeUi = function () {

        if (this.dagView) {
            this.dagView.destroy();
            this.dagView = undefined;
        }

        // positive time is either toward +x or -y.
        var isPositiveTimeDirection = this.settings.orientation === 'horizontal';

        var $metadataContainer = this.$metadataContainer;
        var target = this.target;
        var self = this;

        this.dagView = createGraphView(
            this.dag,
            this.target,
            this.$metadataContainer,
            this.settings.orientation,
            isPositiveTimeDirection
        );


        // don't rebind events if this is our second time through.
        if (!this.initializedUi) {
            // The number of pixels to scroll per mousewheel delta (should match native browser scroll).
            // Firefox likes it slower than others.
            var mouseWheelPx = $.browser.mozilla ? 120 * 2 / 5 : 120;

            /* Safari + SVG + mousewheel events don't work, so also trigger on the HTML wrapper element. */
            var $mouseWheelElements = $(target.getElement()).add('.dag');

            // tools actions need to stop the click from propagating as to not click on underlying metadata
            $(self.$metadataContainer).delegate(".aui-dropdown", "click", function (e) {
                e.stopPropagation();
            });

            $("#goto")
                .placeholder("Enter changeset ID or tag")
                .bind('keydown', function (event) {
                    if (event.which === $.ui.keyCode.ENTER) { //the enter key
                        event.preventDefault();
                        event.stopPropagation();

                        var tagOrId = $.trim($(this).val());

                        if (tagOrId.length) {
                            var changeset = self.dag.findChangesetByIdOrTag(tagOrId);

                            if (changeset) {
                                self.dagView.translateTo(changeset);
                                self.dagView.setFocus(self.dagView.context.highlightContext.getChangeset(changeset));
                            } else {
                                window.location.href = FECRU.NAVBUILDER.graphAtChangeset(self.settings.repositoryName, tagOrId);
                            }

                        }
                    }
                });

            var onMouseWheel;
            if (!$metadataContainer) {
                onMouseWheel = function (e, delta) {
                    self.dagView.translate(new Vector(0, delta * mouseWheelPx));
                    updatePermalink(self);

                    return false;
                };
            } else {
                onMouseWheel = function (e, delta) {
                    $metadataContainer.scrollTop($metadataContainer.scrollTop() - delta * mouseWheelPx);

                    return false;
                };
            }

            $mouseWheelElements.mousewheel(onMouseWheel);

            if ($metadataContainer) {

                var lastScrollPos = {
                    x: $metadataContainer.scrollLeft(),
                    y: $metadataContainer.scrollTop()
                };

                $metadataContainer.bind('scroll', function () {

                    var scrollPos = {
                        x: $metadataContainer.scrollLeft(),
                        y: $metadataContainer.scrollTop()
                    };

                    // We want an inverse diff to prev. If we scroll down, the graph should go up
                    var delta = new Vector(lastScrollPos.x - scrollPos.x, lastScrollPos.y - scrollPos.y);
                    if (delta.x || delta.y) {
                        self.dagView.translate(delta);
                        updatePermalink(self);

                        lastScrollPos = scrollPos;
                    }
                });
            }
            this.changesetDialog = ChangesetDialog.create(self, self.settings.repositoryName);
        }

        this.initializedUi = true;
    };

    ProjectVisualiser.prototype.doInitialLoad = function () {
        var settings = this.settings;
        var self = this;

        // do initial data load
        var params = {
            direction: "around",
            size: settings.initialPreload
        };
        if (settings.scrollToCsid) {
            params.id = settings.scrollToCsid;
        }
        this.requestData(params, function () {
            self.dagView.init(params.id, settings.focusCsid);
            self.updatePermalink();
        });

        self.dagView.bind(EdgeProximityEvent, function (evt) {
            var pivot;
            var id;
            var params;

            if (evt.proximityToFutureHorizon < self.settings.preloadBuffer) {
                pivot = self.dag.getLastChangeset();
                id = pivot && pivot.id;
                params = {
                    direction: "after"
                };
                if (id) {
                    params.id = id;
                }
                self.requestData(params, null);
            }
            if (evt.proximityToPastHorizon < self.settings.preloadBuffer) {
                pivot = self.dag.getFirstChangeset();
                id = pivot && pivot.id;
                params = {
                    direction: "before"
                };
                if (id) {
                    params.id = id;
                }
                self.requestData(params, null);
            }
        });
    };

    ProjectVisualiser.prototype.initializeHighlighters = function () {
        var highlighterContextsById = []; // auto-incrementing 1-based id, AKA index == id + 1.
        var activeHighlightContext = null;
        var self = this;

        var getChangesetsForIds = function (csids) {
            return Array.map(csids, function (csid) {
                return activeHighlightContext.getChangeset(self.dag.findChangesetById(csid));
            })
        };

        this.bind(ProjectVisualiser.DataLoaded, function (event) {
            var changesetIds = event.changesetIds;
            if (activeHighlightContext && changesetIds.length) {
                activeHighlightContext.onDataLoaded(getChangesetsForIds(changesetIds));
            }
        });

        this.bind(ProjectVisualiser.DataUnloaded, function (event) {
            var changesetIds = event.changesetIds;
            if (activeHighlightContext && changesetIds.length) {
                activeHighlightContext.onDataUnloaded(getChangesetsForIds(changesetIds));
            }
        });

        this.registerHighlighters = function (highlighters) {
            var self = this;
            Array.each(highlighters, function (highlighter) {
                self.registerHighlighter(highlighter);
            });
        };

        this.registerHighlighter = function (highlighter) {
            var self = this;
            var highlightContext = new HighlightContext(highlighter);

            highlightContext.id = highlighterContextsById.push(highlightContext);
            this.trigger(ProjectVisualiser.HighlightRegistered, {
                name: highlighter.name,
                id: highlightContext.id
            });

            highlightContext.bind(HighlightContext.LoadingStarted, function () {
                $("#highlight-spinner").addClass("visible");
            });

            highlightContext.bind(HighlightContext.LoadingFinished, function () {
                $("#highlight-spinner").removeClass("visible");
            });

            highlightContext.bind(HighlightContext.RedrawRequested, function () {
                self.dagView.reHighlight();
            });

            //check for highlighters in the permalink, and set it if found. Otherwise,
            //sets up the initial selected highlighter from cookie. Defaults to Lineage
            var preferredHighlighterName = self.settings.hashFragment.getHighlighterName() || self.settings.highlighterName;
            var currentHighlighterName = highlightContext.highlighter.name;

            if (preferredHighlighterName === currentHighlighterName
                || currentHighlighterName === "Lineage") {
                this.$highlightSelector.val(highlightContext.id);
                this.setActiveHighlighter(highlightContext.id);
            }
            return highlightContext.id;
        };

        this.getHighlighterContext = function (id) {
            return highlighterContextsById[id - 1];
        };

        this.getHighlighter = function (id) {
            var context = this.getHighlighterContext(id);
            return context && context.highlighter;
        };

        /**
         * Returns an array of structs, each containing the highlighter id and the highlighter obj
         * e.g., [{id:1,{name:"highlighter1",...}},{id:2,{name:"highlighter2",...}},...}
         */
        this.getHighlighters = function () {
            return Array.map(highlighterContextsById, function (context) {
                return {id: context.id, highlighter: context.highlighter};
            });
        };

        this.setActiveHighlighter = function (id) {
            var newActive = this.getHighlighterContext(id);
            if (newActive && newActive !== activeHighlightContext) {
                var oldContext = activeHighlightContext;
                var focusedObject = oldContext && oldContext.focusedObject;

                oldContext && oldContext.onDeactivation();

                activeHighlightContext = newActive;

                var allChangesets;
                if (this.dag) {
                    allChangesets = Array.map(this.dag.getAllChangesets(), function (cs) {
                        return activeHighlightContext.getChangeset(cs);
                    })
                } else {
                    allChangesets = [];
                }

                // Convert focused object to new highlight context.
                if (focusedObject) {
                    focusedObject = focusedObject.isChangeset() ?
                        activeHighlightContext.getChangeset(focusedObject._model) :
                        activeHighlightContext.getEdge(focusedObject._model);
                }

                activeHighlightContext.onActivation(allChangesets, focusedObject);

                this.dagView && this.dagView.setHighlightContext(activeHighlightContext);
                updatePermalink(this);
                return activeHighlightContext.highlighter;
            } else {
                return undefined;
            }
        };
        this.getActiveHighlighter = function () {
            return activeHighlightContext && activeHighlightContext.highlighter;
        };

        //after the first initialize, reuse all the functions and just clear out the data.
        this.initializeHighlighters = function () {
            // This is a shortcut - we should really be marking all data as unloaded and reloaded.
            this.dagView && this.dagView.setHighlightContext(activeHighlightContext);
        };
    };

    /**
     * Saves to the cookie the given highlighter.
     * @param visualiser the visualiser object
     * @param highlighter
     */
    function doSaveHighlighterCookie(visualiser, highlighter) {
        if (highlighter) {
            AJAX.ajaxDo(visualiser.settings.saveBranchesUrl, {
                repoName: visualiser.settings.repositoryName,
                'vhl': '{"' +
                visualiser.settings.repositoryName + '":{' +
                'hl:"' + FECRU.quoteString(highlighter.name) + '"' +
                '}' +
                '}'
            });

            if (window.localStorage) {
                window.localStorage[FE.VIS.localStorageKeys(visualiser.settings.repositoryName).highlighter] = highlighter.name;
            }
        }
    }

    ProjectVisualiser.prototype.setHighlighterSelector = function ($select) {
        var self = this;
        var $oldSelector = this.$highlightSelector;

        this.$highlightSelector = $select;

        var addHighlight = function (highlightData) {
            $("<option>")
                .text(highlightData.name)
                .attr('value', highlightData.id)
                .appendTo($select);
        };
        $select.data('add-highlight-handler', addHighlight);
        this.bind(ProjectVisualiser.HighlightRegistered, addHighlight);
        this.getHighlighters && Array.each(this.getHighlighters(), function (highlighterData) {
            addHighlight({id: highlighterData.id, name: highlighterData.highlighter.name});
        });

        $select.change(function (e) {
            self.setActiveHighlighter && doSaveHighlighterCookie(self, self.setActiveHighlighter(e.target.value));
        });

        if ($oldSelector) {
            this.unbind(ProjectVisualiser.HighlightRegistered, $oldSelector.data('add-highlight-handler'))
        }
    };

    ProjectVisualiser.prototype.resize = function () {
        this.dagView && this.dagView.resize();
    };

    ProjectVisualiser.prototype.setLoading = function (loading) {
        var $container = this.$metadataContainer;

        var $spinner = $container.data("metadata-loading");
        if (loading) {
            // don't do anything if there is already a spinner running
            if (!$spinner) {
                $spinner = $("<div class='metadata-loading'>Loading...</div>");
                $container.prepend($spinner);
                $container.data("metadata-loading", $spinner);
            } else {
                $spinner.show();
            }
        } else {
            $spinner && $spinner.hide();
        }
    };


    ProjectVisualiser.prototype.loadMetadata = function (loadParams) {
        var self = this;
        self.setLoading(true);

        $.ajax({
            url: self.settings.metadataUrl,
            type: "POST",
            contentType: 'application/json',
            data: JSON.stringify(loadParams),
            success: function (data) {
                self.setLoading(false);
                self.dag.updateMetadata(data.changesets);
                self.dagView.reloadMetadata();
                self.trigger(ProjectVisualiser.DataLoaded, {
                    "changesetIds": Array.map(data.changesets, function (cs) {
                        return cs.csid;
                    })
                });
            },
            error: function (req) {
                displayError(req);
            }
        });
    };

    ProjectVisualiser.prototype.requestData = function (params, callback) {
        var self = this;
        var dataLoadingTracker = self.dataLoadingTracker;
        var onLoadSuccess = function (params, callback) {
            return function (data) {
                if (data.revisions && data.revisions.length) {

                    var oldBreadth = self.dag.getBreadth();
                    var newIds = self.dag.loadSlices(data, params.direction === 'after');
                    var breadthChanged = oldBreadth !== self.dag.getBreadth();

                    //only load new metadata if there are new ids loaded from this ajax call
                    newIds && newIds.length > 0 && self.loadMetadata({
                        id: newIds
                    });
                }

                if (data.revisions && data.revisions.length < params.size) {
                    if (params.direction === 'before') {
                        dataLoadingTracker.reachedBeginning = true;
                    } else if (params.direction === 'after') {
                        dataLoadingTracker.reachedEnd = true;
                    } else { // around
                        dataLoadingTracker.reachedBeginning = true;
                        dataLoadingTracker.reachedEnd = true;
                    }
                }

                if (params.direction === 'before') {
                    dataLoadingTracker.alreadyLoadingPast = false;
                } else if (params.direction === 'after') {
                    dataLoadingTracker.alreadyLoadingFuture = false;
                } else { // around
                    dataLoadingTracker.alreadyLoadingFuture = false;
                    dataLoadingTracker.alreadyLoadingPast = false;
                }

                var dagView = self.dagView;
                dagView.sync(breadthChanged);
                breadthChanged && dagView.resize();

                callback && callback();
            };
        };

        var defaults = {
            size: self.settings.changesetsPerLoad
        };

        if (!self.branchSet.isAllInOne()) {
            defaults.branch = self.branchSet.getBranchNames();
        }

        var ajaxParams = $.extend({}, defaults, params);

        if ((!dataLoadingTracker.canLoadPast() && ajaxParams.direction === 'before') ||
            (!dataLoadingTracker.canLoadFuture() && ajaxParams.direction === 'after') ||
            (!dataLoadingTracker.canLoadPast() && !dataLoadingTracker.canLoadFuture())) {
            return;
        }
        if (ajaxParams.direction === 'before' || ajaxParams.direction === 'around') {
            dataLoadingTracker.alreadyLoadingPast = true;
        }
        if (ajaxParams.direction === 'after' || ajaxParams.direction === 'around') {
            dataLoadingTracker.alreadyLoadingFuture = true;
        }

        $.ajax({
            url: self.settings.dataUrl,
            type: "GET",
            contentType: 'application/json',
            data: ajaxParams,
            success: onLoadSuccess(ajaxParams, callback),
            error: function (req, textStatus, errorThrown) {
                displayError(req);
            }
        });
    };

    /**
     * Displays an error dialog box. This method only handles errors that come from jersey's error response,
     * which contains a stacktrace property that we display.
     * @param req the request object from jquery's ajax 'onerror' callback handler. It should be a raw string, and should evaluate to
     * a json object with a stacktrace property
     */
    function displayError(req) {
        var resp = eval('(' + req.responseText + ')');
        if (resp.stacktrace) {
            FECRU.AJAX.appendErrorResponse(resp.stacktrace, false);
            FECRU.AJAX.showNotificationBox(resp.code + ": " + resp.message);
        }
    }

    /**
     * Reload the graph by reinitialising with the given newBranches
     * @param newBranches the set of branches to reload the graph with.
     */
    ProjectVisualiser.prototype.reload = function (newBranches) {
        //todo: this needs to reload at the same position/changeset
        var newSettings = $.extend(this.settings, {branches: newBranches});
        this.initialize(newSettings);
    };

    ProjectVisualiser.prototype.setBranchSet = function (branchSet) {
        var self = this;
        this.branchSet = branchSet;
        if (this.branchHeader) {
            this.branchHeader.setBranchSet(branchSet);
        }

        self.branchSet.bind(BranchSet.BranchesRemoved, function (event) {
            Array.each(event.removedBranches, function (branch) {
                self.branchSet.removeBranch(branch);
            });
            //persist the branchset after branch(es) are removed
            self.branchSet.saveBranches(event.visibleBranches, false);
        });

        self.branchSet.bind(BranchSet.BranchesAdded, function (event) {
            Array.each(event.addedBranches, function (branch) {
                self.branchSet.addBranch(branch);
            });
            //persist the branchset after branch(es) are added
            self.branchSet.saveBranches(event.visibleBranches, false);
        });

        self.branchSet.bind(BranchSet.BranchModified, function (event) {
            window.location.href = window.location.href.replace(/[\?#].*$/, ""); // page pop
        });

        self.branchSet.bind(BranchSet.BranchesReordered, function (event) {
            self.branchSet.saveBranches(event.branches, false);
        });
    };

    ProjectVisualiser.prototype.getBranchSet = function () {
        return this.branchSet;
    };

    ProjectVisualiser.prototype.setBranchHeaderContainer = function ($branchHeaderContainer) {
        this.branchHeader = new BranchHeader($branchHeaderContainer, this.branchSet, this.positioner);
    };

    ProjectVisualiser.prototype.getPrimaryBranches = function (branches) {
        if (this.branchSet.isAllInOne()) {
            return [this.branchSet.getAllInOne()];
        }
        var branch = Array.first(this.branchSet.getAllBranches(), function (branchInSet) {
            var name = branchInSet.name;
            return Array.any(branches, function (branchNameToCheck) {
                return name === branchNameToCheck;
            });
        });
        return branch ? [branch] : [];
    };

    ProjectVisualiser.prototype.getTopChangeset = function () {
        return this.positioner.getChangesetAtTimePosition(this.dag, Math.round(this.dagView.getScrolledToTimePosition()));
    };

    ProjectVisualiser.prototype.getBaseUrl = function () {
        return window.location.pathname;
    };

    ProjectVisualiser.prototype.updatePermalink = function () {
        this.$permalink && this.$permalink.attr('href', this.getPermalinkUrl());
    };

    ProjectVisualiser.prototype.getPermalinkUrl = function () {

        var topChangeset = this.getTopChangeset();
        var branches = this.branchSet.isAllInOne() ? undefined : this.branchSet.getBranchNames();
        var scrollToChangesetId = topChangeset ? topChangeset.id : undefined;
        var highlighter = this.getActiveHighlighter();
        var hashFragment = Fragment.create(branches, this.branchSet.isAllInOne(), undefined, highlighter && highlighter.name, scrollToChangesetId).asString();

        return hashFragment.length ? "#" + hashFragment : "";
    };

    ProjectVisualiser.prototype.setPermalinkElement = function (anchorSelector) {
        this.$permalink = $(anchorSelector);
    };

    /**
     * Create a new ProjectVisualiser
     * @param selector the selector of the visualisation
     * @param settings object of settings to override
     * @return a new instance of the ProjectVisualiser object
     */
    ProjectVisualiser.create = function (selector, settings) {
        var $projectVisualiser = $(selector);
        var $branchHeader = $projectVisualiser.children(".branch-header");
        var $highlightSelector = $('#highlight');
        var $dag = $projectVisualiser.children(".dag");
        var $metadataContainer = $projectVisualiser.children(".metadata-wrapper");

        settings = settings ? $.extend({}, settings) : {};

        var TargetClass = $.browser.msie && $.browser.version < 9 ? Targets.VML : Targets.SVG;
        var target = new TargetClass($dag[0]);

        if (/horizontal/.test(window.location.hash)) {
            settings.orientation = 'horizontal';
        } else {
            settings.orientation = 'vertical';
        }

        var pv = new ProjectVisualiser(target, $metadataContainer, defaultPositioner, settings);
        pv.initialize();
        pv.setBranchHeaderContainer($branchHeader);
        pv.setHighlighterSelector($highlightSelector);
        pv.setPermalinkElement("#visualisation-permalink");

        function setVisualisationHeight() {
            var $visualisationContent = $dag.add($metadataContainer);
            var $toolbar = $("#toolbar");

            var visHeight = parseInt(document.getElementById("columns").style.height, 10) // We want the actual style height not the computed height
                - $toolbar.outerHeight(true)
                - 2; // Some border somewhere which is fucking up Chrome and Safari.
            var contentHeight = visHeight - $branchHeader.outerHeight(true);

            $projectVisualiser.height(visHeight);
            $visualisationContent.height(contentHeight);
            pv.resize();
        }

        setVisualisationHeight();
        FECRU.UI.setCompletedResizeTimeout(window, setVisualisationHeight);

        return pv;
    };

    function getPrimaryBranchesForSvn(branchSet, branches) {
        if (branchSet.isAllInOne()) {
            return [branchSet.getAllInOne()];
        }
        var primaryBranches = [];
        Array.each(branchSet.getAllBranches(), function (branchInSet) {
            var name = branchInSet.name;
            if (Array.any(branches, function (branchNameToCheck) {
                    return name === branchNameToCheck;
                })) {
                primaryBranches.push(branchInSet);
            }
        });
        return primaryBranches;
    }

    return ProjectVisualiser;
})(AJS.$, AJS.template);


/*[{!project_visualiser_js_4bfv52n!}]*/