FE.VIS.Changeset = (function ($) {

    var State = {
        UNLOADED: {
            toString: function () {
                return "Unloaded";
            }
        },
        LOADED: {
            toString: function () {
                return "Loaded";
            }
        },
        LIVE: {
            toString: function () {
                return "Live";
            }
        },
        SPARSE: {
            toString: function () {
                return "Sparse";
            }
        }
    };

    // Imports
    var BranchRevision = FE.VIS.BranchRevision;

    // Public

    /*
     * @param id - a unique identifier for this changeset.
     */
    var Changeset = function (id) {
        this.id = id;
        this.parentChangesets = undefined;
        this.childChangesets = undefined;
        this.branchRevisions = undefined;
        this.timePosition = NaN;
        this.state = State.UNLOADED;
        this.metadataState = State.UNLOADED;
    };

    /* Static Methods */

    /* Prototype Methods */

    Changeset.prototype.compareTo = function (other) {
        // order is a string, so we can't just take the difference between orders.
        return other && other.order ? (
            this.order < other.order ? -1 :
                this.order > other.order ? 1 :
                    0
        ) :
            -Infinity;
    };

    /*
     * A READ-ONLY array of this changeset's direct parents.
     */
    Changeset.prototype.parents = function () {
        return this.parentChangesets ? this.parentChangesets.slice(0) : [];
    };

    /*
     * A READ-ONLY array of this changeset's direct children.
     */
    Changeset.prototype.children = function () {
        return this.childChangesets ? this.childChangesets.slice(0) : [];
    };

    Changeset.prototype.isUnloaded = function () {
        return this.state === State.UNLOADED;
    };

    Changeset.prototype.isMetadataLoaded = function () {
        return this.metadataState === State.LOADED;
    };

    Changeset.prototype.isLive = function () {
        return this.state === State.LIVE;
    };

    Changeset.prototype.isSparse = function () {
        return this.state === State.SPARSE;
    };

    Changeset.prototype.isSparseAncestor = function () {
        return this.isSparse() && this.primaryBranches && this.primaryBranches.length > 0 && this.hasLiveChild();
    };

    Changeset.prototype.isSparseDescendant = function () {
        return this.isSparse() && this.primaryBranches && this.primaryBranches.length > 0 && this.hasLiveParent();
    };

    Changeset.prototype.isSparseOnBranch = function () {
        return this.isSparse() && (!this.primaryBranches || this.primaryBranches.length === 0);
    };

    Changeset.prototype.load = function (data, branchManager) {
        data = data || {};
        //this.date = data.date && (Date.parse(data.date) || typeof data.date === "number") ? new Date(data.date) : undefined;
        this.order = data.position;
        this.branches = data.branches || [];
        this.tags = data.tags || [];

        this.primaryBranches = this.branches && branchManager.getPrimaryBranches(this.branches);

        this.branchRevisions = [];
        var self = this;
        Array.each(this.primaryBranches, function (primaryBranch) {
            self.branchRevisions.push(new BranchRevision(self, primaryBranch));
        });

        this.author = {};

        this.state = State.LOADED;
    };

    Changeset.prototype.loadMetadata = function (data) {
        this.date = data.date;
        this.author = data.user;
        this.comment = data.comment;

        this.metadataState = State.LOADED;
    };

    Changeset.prototype.setSlice = function (slice) {
        this.slice = slice;
    };

    Changeset.prototype.makeSparse = function () {
        if (this.isUnloaded()) {
            throw "changeset must have data loaded before making it sparse"
        }

        this.state = State.SPARSE;
    };

    Changeset.prototype.makeLive = function () {
        if (this.isUnloaded()) {
            throw "changeset must have data loaded before making it live"
        }
        this.state = State.LIVE;
    };

    Changeset.prototype.dispose = function () {
        if (!this.isSparse()) {
            throw "Cannot dispose non-sparse changeset";
        }

        delete this.parentEdges;
        delete this.childEdges;
    };

    /*
     * returns whether this changeset is linked to any loaded parents
     */
    Changeset.prototype.hasLiveParent = function () {
        return this.parentChangesets && Array.any(this.parentChangesets, function (parent) {
                return parent.isLive();
            });
    };

    // returns whether this changeset is linked to any loaded children
    Changeset.prototype.hasLiveChild = function () {
        return this.childChangesets && Array.any(this.childChangesets, function (child) {
                return child.isLive();
            });
    };

    Changeset.prototype.isDisposable = function () {
        return this.isSparse() && !(this.hasLiveParent() || this.hasLiveChild());
    };

    Changeset.prototype.addParent = function (newParent) {
        if (this.parentChangesets) {
            if ($.inArray(newParent, this.parentChangesets) !== -1) {
                return this;
            }
        } else {
            this.parentChangesets = [];
        }

        if (!newParent.childChangesets) {
            newParent.childChangesets = [];
        }

        this.parentChangesets.push(newParent);
        newParent.childChangesets.push(this);

        return this;
    };

    Changeset.prototype.getBranchRevision = function (branch) {
        return Array.first(this.branchRevisions, function (brev) {
                return brev.primaryBranch === branch;
            }) || null;
    };

    function findMatchingBranchRevision(parentChangeset, primaryBranch) {
        // If there is only one parentBranchRevision, it must link to that. Otherwise we try and find the one
        // on the same branch as our branchRevision
        return parentChangeset.branchRevisions.length === 1 ?
            parentChangeset.branchRevisions[0] :
            parentChangeset.getBranchRevision(primaryBranch);
    }

    Changeset.prototype.resolveParents = function () {
        var branchRevision;
        var primaryBranch;
        var parents = this.parentChangesets;

        if (!parents || !parents.length) {
            return;
        }

        var unresolved = parents.slice(0);
        var branchRevisions = this.branchRevisions;

        // We want to return quickly for SCMs other than SVN. The will have a 1-1 relationship betweem changeset->
        if (branchRevisions.length === 1) {
            branchRevision = branchRevisions[0];
            primaryBranch = branchRevision.primaryBranch;
            // We know that all the parents should link to a branchRevision and because there is only one
            // We try and associate all the parents with it
            Array.each(parents, function (parentChangeset) {

                var parentBranchRevision = findMatchingBranchRevision(parentChangeset, primaryBranch);

                // Only add the edge if we could figure out the parentBranchRevision
                // This will only be falsy in SVN
                if (parentBranchRevision) {
                    branchRevision.addParent(parentChangeset, parentBranchRevision);
                }
            });
        } else {
            // We first want to loop over every comment
            Array.each(branchRevisions, function (branchRevision) {
                var parentChangeset;
                var parentBranchRevision;
                var primaryBranch = branchRevision.primaryBranch;

                // Use simple logic if we can
                if (parents.length === 1) {

                    parentChangeset = parents[0];
                    parentBranchRevision = findMatchingBranchRevision(parentChangeset, primaryBranch);

                    if (parentBranchRevision) {
                        branchRevision.addParent(parentChangeset, parentBranchRevision);
                    }

                } else {
                    // If we have multiple branchRevisions and multiple parents, we want to try and match
                    // branchRevisions with the changeset with the highest order with a matching primaryBranch
                    var parentsOnBranch = $.grep(parents, function (parent) {
                        return !!parent.getBranchRevision(primaryBranch)
                    });

                    // If we have a match
                    if (parentsOnBranch.length) {

                        // Sort them in reverse order. the highest changeset will be the first
                        parentsOnBranch.sort(function (a, b) {
                            return -a.compareTo(b);
                        });
                        parentChangeset = parentsOnBranch[0];

                        // We already know that there will be a matching bRev for the primary branch
                        // Due to the .grep above
                        parentBranchRevision = parentChangeset.getBranchRevision(primaryBranch);
                        branchRevision.addParent(parentChangeset, parentBranchRevision);

                        // Remove it from the unresolved list.
                        var i = $.inArray(parentChangeset, unresolved);
                        if (i !== -1) {
                            Array.remove(unresolved, i);
                        }
                    }
                }
            });

            // There are some things we just can't match. This is a logical deduction to try and match the
            // last parent with the last branchRevision
            if (unresolved.length === 1) {
                var branchRevisionsWithoutParent = $.grep(branchRevisions, function (brev) {
                    return brev.parents().length === 0
                });
                if (branchRevisionsWithoutParent.length === 1) {
                    // I haz a match
                    branchRevision = branchRevisionsWithoutParent[0];
                    primaryBranch = branchRevision.primaryBranch;
                    var parentChangeset = unresolved[0];
                    var parentBranchRevision = findMatchingBranchRevision(parentChangeset, primaryBranch);

                    if (parentBranchRevision) { // Only add it if the parent exists
                        branchRevision.addParent(parentChangeset, parentBranchRevision);
                    }

                }
            }
        }
    };

    Changeset.prototype.addChild = function (child) {
        child.addParent(this);
        return this;
    };

    return Changeset;
})(AJS.$);
/*[{!changeset_js_p74b539!}]*/