/*
 * A ChangesetGroup represents a named branch within a given ChangesetDAGSlice. All commits
 * for that branch within the current slice's range of "commit order" should be contained here.
 * */
FE.VIS.ChangesetGroup = (function ($) {

    /*
     * @param branch - the branch that this group is part of
     * @param slice - the slice object that this group is part of.
     */
    var ChangesetGroup = function (branch, slice) {
        this.branch = branch;
        this.element = null;
        this.branchRevisions = [];
        this.internalEdges = undefined;
        this.externalEdges = undefined;

        this.slice = slice;

        this.dirty = true;
    };
    ChangesetGroup.prototype.addBranchRevision = function (branchRevision) {
        this.branchRevisions.push(branchRevision);
        branchRevision.setGroup(this);
    };
    /*
     * sort an edge into the internalEdges array or the externalEdges array based on whether
     * otherChangeset is contained in internalBranchRevisions.
     * @param edge - the edge to be sorted.  One end should point to a changeset within this
     *          Group, and the other end should point to otherChangeset
     * @param otherChangeset - check this changeset and see if it's also within this Group
     * @param internalBranchRevisions - a list of branch revisions in this group
     * @param externalEdges - a list of edges that link branch revisions in this Group to ones in other Groups.
     * @param internalEdges - a list of edges that link branch revisions in this Group to other branch revisions in this Group.
     */
    function classifyEdgeType(edge, otherBranchRevision, internalBranchRevisions, externalEdges, internalEdges) {
        if ($.inArray(otherBranchRevision, internalBranchRevisions) === -1) { // if the other end is outside, add to external edges
            if ($.inArray(edge, externalEdges) === -1) {
                externalEdges.push(edge);
            }
        } else { // otherwise add it internally
            if ($.inArray(edge, internalEdges) === -1) {
                internalEdges.push(edge);
            }
        }
    }

    ChangesetGroup.prototype.updateEdgeLevels = function () {
        var branchRevisions = this.branchRevisions;
        var externalEdges = this.externalEdges = [];
        var internalEdges = this.internalEdges = [];

        Array.each(branchRevisions, function (branchRevision) {
            if (branchRevision.parentEdges) {
                Array.each(branchRevision.parentEdges, function (edge) {
                    classifyEdgeType(edge, edge.parentBranchRevision, branchRevisions, externalEdges, internalEdges);
                });
            }
            if (branchRevision.childEdges) {
                Array.each(branchRevision.childEdges, function (edge) {
                    classifyEdgeType(edge, edge.childBranchRevision, branchRevisions, externalEdges, internalEdges);
                });
            }
        });
    };

    ChangesetGroup.prototype.getType = function () {
        return "Group";
    };

    ChangesetGroup.prototype.getAllBranchRevisions = function () {
        return this.branchRevisions.slice(0);
    };

    ChangesetGroup.prototype.dispose = function () {
        Array.each(this.branchRevisions, function (branchRevision) {
            branchRevision.dispose(); // necessary?
        });
        delete this.branchRevisions;
        delete this.internalEdges;
        delete this.externalEdges;
    };

    return ChangesetGroup;
})(AJS.$);
/*[{!changeset_group_js_tvdw538!}]*/