FE.VIS.ChangesetDAG = (function ($) {

    // Imports
    var VIS = FE.VIS;
    var ChangesetDAGSlice = VIS.ChangesetDAGSlice;
    var Changeset = VIS.Changeset;
    var Vector = Shapes.Vector;
    var Map = FECRU.DATA_STRUCTURES.Map;
    var memoize = FECRU.DECORATORS.memoize;
    var eraseMemoized = FECRU.DECORATORS.eraseMemoized;

    var decimalPattern = /^[0-9]+$/;

    function isDecimalString(str) {
        return decimalPattern.test(str);
    }

    // Public

    /*
     * @param branchManager - an object with the methods getBranchSet and getPrimaryBranch.
     * @param settings - an object containing relevant settings.
     */
    var ChangesetDAG = function (branchManager, settings, positioner) {
        this.changesetsById = new Map();
        this.dagSlices = [];
        this.interSliceEdges = [];
        this.sparseDescendants = new Map();
        this.sparseAncestors = new Map();
        this.sparseOnBranch = new Map();
        this.branchManager = branchManager;
        this.positioner = positioner;
        this.positioningData = {};

        this.element = null;
        this.translation = new Vector(0, 0);
        this.settings = settings;
    };
    ChangesetDAG.prototype.addInterSliceEdges = function (edges) {
        var interSliceEdges = this.interSliceEdges;
        Array.each(edges, function (edge) {
            if ($.inArray(edge, interSliceEdges) === -1) {
                interSliceEdges.push(edge);
            }
        });
    };

    ChangesetDAG.prototype.createChangesets = function (data) {
        // Cache values
        var changesetsById = this.changesetsById;
        var sparseAncestors = this.sparseAncestors;
        var sparseDescendants = this.sparseDescendants;
        var sparseOnBranch = this.sparseOnBranch;
        var dag = this;

        var newLiveChangesets = [];
        var newSparse = [];
        var newSparseAncestors = [];
        var newSparseDescendants = [];
        var newSparseOnBranch = [];
        var changesetData = data.changesets || data.revisions;
        var sparseData = data.sparseChangesets || data.sparseRevisions;

        if (!changesetData && !sparseData) {
            throw "Invalid data format.  Must include 'revisions' and 'sparseRevisions' properties.";
        }

        if (changesetData) {
            Array.each(changesetData, function (csData) {
                var csId = csData.id || csData.csid;
                var cs = null;

                if (changesetsById.has(csId)) {
                    cs = changesetsById.get(csId);
                    if (cs.isSparse()) {
                        cs.makeLive();
                        sparseAncestors.remove(csId);
                        sparseDescendants.remove(csId);
                        newLiveChangesets.push(cs);
                    }
                } else {
                    cs = new Changeset(csId);
                    cs.load(csData, dag.branchManager);
                    cs.makeLive();

                    changesetsById.set(csId, cs);
                    newLiveChangesets.push(cs);
                }
            });
        }

        if (sparseData) {
            Array.each(sparseData, function (csData) {
                var csId = csData.id || csData.csid;
                if (!changesetsById.has(csId)) {
                    var cs = new Changeset(csId);
                    cs.load(csData, dag.branchManager);
                    cs.makeSparse();

                    changesetsById.set(csId, cs);
                    newSparse.push(cs);
                }
            });
        }

        //link them to parent/child changesets.
        if (changesetData) {
            Array.each(changesetData, function (csData) {
                var cs = changesetsById.get(csData.id || csData.csid);
                Array.each(csData.parents, function (parentId) {
                    if (changesetsById.has(parentId)) {
                        cs.addParent(changesetsById.get(parentId));
                    }
                });
                Array.each(csData.children, function (childId) {
                    if (changesetsById.has(childId)) {
                        changesetsById.get(childId).addParent(cs);
                    }
                });
            });
        }

        Array.each(newSparse, function (cs) {
            // if we're showing this changeset's branch, it's a descendent or ancestor
            if (cs.primaryBranches && cs.primaryBranches.length > 0) {
                if (cs.hasLiveParent()) {
                    sparseDescendants.set(cs.id, cs);
                    newSparseDescendants.push(cs);
                } else if (cs.hasLiveChild()) {
                    sparseAncestors.set(cs.id, cs);
                    newSparseAncestors.push(cs);
                }
            } else { // otherwise, it's on a skipped branch.
                sparseOnBranch.set(cs.id, cs);
                newSparseOnBranch.push(cs);
            }
        });
        function resolveParents(cs) {
            cs.resolveParents();
        }

        Array.each(newLiveChangesets, resolveParents);
        Array.each(newSparse, resolveParents);

        return {
            newLiveChangesets: newLiveChangesets,
            newSparseAncestors: newSparseAncestors,
            newSparseDescendants: newSparseDescendants,
            newSparseOnBranch: newSparseOnBranch
        };
    };

    ChangesetDAG.prototype.getType = function () {
        return "Graph";
    };


    ChangesetDAG.prototype.getLength = function () {
        return Array.reduce(this.dagSlices, 0, function (base, slice) {
            return base + slice.length;
        });
    };
    memoize(ChangesetDAG.prototype, 'getLength');

    ChangesetDAG.prototype.getBreadth = function () {
        return Array.reduce(this.branchManager.getBranchSet().getAllBranches(), 0, function (base, branch) {
            return base + branch.breadth;
        });
    };
    memoize(ChangesetDAG.prototype, 'getBreadth');

    /**
     * Load multiple slices of changesets across multiple branches.
     * @param data data in the format { changesets: [changesetData...], sparseChangesets: [changesetData...]  }
     * @param atEnd load at the end of time. Positioning based off ancestors
     */
    ChangesetDAG.prototype.loadSlices = function (data, atEnd) {
        var ret = this.createChangesets(data);
        var newLiveChangesets = ret.newLiveChangesets;
        var newSparseAncestors = ret.newSparseAncestors;
        var newSparseDescendants = ret.newSparseDescendants;
        var newSparseOnBranch = ret.newSparseOnBranch;
        var newSlices = [];
        var dag = this;

        atEnd = !!atEnd;

        var sign = atEnd ? 1 : -1;
        newLiveChangesets.sort(function (a, b) {
            return sign * a.compareTo(b);
        });

        var slice;
        var sliceChangesets;
        var existingSlices = this.dagSlices;
        var neighbor;

        if (existingSlices.length) {
            neighbor = atEnd ?
                existingSlices[existingSlices.length - 1] :
                existingSlices[0];
        } else {
            neighbor = null;
        }

        for (var i = 0, length = newLiveChangesets.length, sliceSize = this.settings.maxSliceSize;
             i < length;
             i += sliceSize) {
            sliceChangesets = newLiveChangesets.slice(i, i + sliceSize);

            slice = new ChangesetDAGSlice(dag);
            slice.load(dag.branchManager.getBranchSet(), sliceChangesets, atEnd);

            if (atEnd) {
                this.dagSlices.push(slice);
            } else {
                this.dagSlices.unshift(slice);
            }

            newSlices.push(slice);
            this.addInterSliceEdges(slice.externalEdges);
        }

        this.positioner.positionNewSlices(
            newSlices,
            newSparseAncestors,
            newSparseDescendants,
            newSparseOnBranch,
            this.positioningData,
            neighbor,
            atEnd);

        var newIds = [];
        Array.each(newLiveChangesets, function (cs) {
            newIds.push(cs.id);
        });
        return newIds;

    };
    eraseMemoized(ChangesetDAG.prototype, 'loadSlices', ['getLength', 'getBreadth']);

    ChangesetDAG.prototype.updateMetadata = function (changesets) {
        var changesetsById = this.changesetsById;
        Array.each(changesets, function (changesetData) {
            if (changesetsById.has(changesetData.csid)) {
                var cs = changesetsById.get(changesetData.csid);
                cs.loadMetadata(changesetData);
            }
        });

    };

    ChangesetDAG.prototype.getFirstChangeset = function () {
        var slices = this.dagSlices;
        if (slices.length) {
            var changesets = slices[0].getAllChangesets();
            if (changesets.length) {
                return changesets[0];
            }
        }
        return null;
    };
    ChangesetDAG.prototype.getLastChangeset = function () {
        var slices = this.dagSlices;
        if (slices.length) {
            var changesets = slices[slices.length - 1].getAllChangesets();
            if (changesets.length) {
                return changesets[changesets.length - 1];
            }
        }
        return null;
    };

    ChangesetDAG.prototype.getAllChangesets = function () {
        return this.changesetsById.getValues();
    };

    ChangesetDAG.prototype.findChangesetById = function (id) {
        return this.changesetsById.get(id);
    };

    ChangesetDAG.prototype.findChangesetByIdOrTag = function (idOrTag) {
        // If we have an exact match, return it
        var changeset = this.findChangesetById(idOrTag);
        if (changeset && changeset.isLive()) {
            return changeset;
        }

        // Otherwise try and find a partial match of an id or match a tag
        var changesets = this.getAllChangesets();
        return Array.first(changesets, function (cs) {
                return cs.isLive() &&
                    ($.inArray(idOrTag, cs.tags) !== -1 ||
                    (!isDecimalString(idOrTag) && cs.id.substr(0, idOrTag.length) === idOrTag));
            }) || null;
    };

    return ChangesetDAG;
})(AJS.$);
/*[{!changeset_dag_js_7w1f537!}]*/