/*
 * A ChangesetDAGSlice represents commits within an arbitrary range of "commit order".
 * "commit order" is measured generally through the commit date, with modifications
 * by the server for consistency. It contains a list of ChangesetGroups that represent
 * all displayed branches.
 */
FE.VIS.ChangesetDAGSlice = (function ($) {

    // Imports
    var VIS = FE.VIS;
    var ChangesetGroup = VIS.ChangesetGroup;
    var Map = FECRU.DATA_STRUCTURES.Map;
    var memoize = FECRU.DECORATORS.memoize;
    var eraseMemoized = FECRU.DECORATORS.eraseMemoized;

    // Public

    /*
     * @param dag - the ChangesetDAG object that this slice is part of.
     */
    var ChangesetDAGSlice = function (dag) {
        this.changesetGroupsByBranch = new Map();
        this.changesetGroups = [];
        this.changesets = [];
        this.dirtyPositions = true;
        this.interGroupEdges = undefined;
        this.externalEdges = undefined;
        this.element = undefined;
        this.timePosition = 0;
        this.length = undefined;

        this.dirty = true;

        this.dag = dag;
    };

    function setGroups(slice, branchSet) {
        Array.each(branchSet.getAllBranches(), function (branch) {
            var newGroup = new ChangesetGroup(branch, slice);
            slice.changesetGroups.push(newGroup);
            slice.changesetGroupsByBranch.set(branch.name, newGroup);
        });
    }

    ChangesetDAGSlice.prototype.getGroup = function (branch) {
        return this.changesetGroupsByBranch.get(branch.name);
    };
    ChangesetDAGSlice.prototype.updateEdgeLevels = function () {
        var externalEdges = this.externalEdges = [];
        var interGroupEdges = this.interGroupEdges = [];

        function eachGroupExternalEdge(edge) {
            var interSliceIndex = $.inArray(edge, externalEdges);
            if (interSliceIndex === -1) { // first group where this edge was external to the group
                externalEdges.push(edge); //assume external to the slice also.
            } else { // we've found the other endpoint of the edge within the slice.
                interGroupEdges.push(edge);
                Array.remove(externalEdges, interSliceIndex);
            }
        }

        Array.each(this.changesetGroups, function (group) {
            group.updateEdgeLevels();
            Array.each(group.externalEdges, eachGroupExternalEdge);
        });
    };
    ChangesetDAGSlice.prototype.getAllChangesets = function (reverseOrder) {
        var ret = this.changesets.slice(0);
        var sign = reverseOrder ? -1 : 1;

        ret.sort(function (a, b) {
            return sign * a.compareTo(b);
        });
        return ret;
    };

    ChangesetDAGSlice.prototype.dispose = function () {
        if (this.element) {
            throw "Hide the slice before disposing it.";
        }

        for (var i = this.changesetGroups.length - 1; i >= 0; i--) {
            this.changesetGroups[i].dispose();
        }
        delete this.interGroupEdges;
        delete this.changesetGroups;
        delete this.changesetGroupsByBranch;
        delete this.externalEdges;
    };

    ChangesetDAGSlice.prototype.getType = function () {
        return "Slice";
    };

    ChangesetDAGSlice.prototype.getLength = function () {
        return this.length;
    };
    memoize(ChangesetDAGSlice.prototype, 'getLength');

    ChangesetDAGSlice.prototype.getBreadth = function () {
        return Array.reduce(this.changesetGroups, 0, function (base, group) {
            return base + group.branch.breadth;
        });
    };
    memoize(ChangesetDAGSlice.prototype, 'getBreadth');

    ChangesetDAGSlice.prototype.load = function (branchSet, changesets) {
        setGroups(this, branchSet);
        var slice = this;
        Array.each(changesets, function (cs) {
            Array.each(cs.branchRevisions, function (brev) {
                slice.getGroup(brev.primaryBranch).addBranchRevision(brev);
            });
            slice.changesets.push(cs);
            cs.setSlice(slice);
        });
        this.length = changesets.length;

        this.updateEdgeLevels();
    };
    eraseMemoized(ChangesetDAGSlice.prototype, 'load', ['getLength', 'getBreadth']);

    return ChangesetDAGSlice;
})(AJS.$);
/*[{!changeset_dag_slice_js_bjah536!}]*/