FE.VIS.BranchRevision = (function ($) {

    // Imports
    var Edge = FE.VIS.Edge;
    var decorators = FECRU.DECORATORS;
    var memoize = decorators.memoize;
    var eraseMemoized = decorators.eraseMemoized;

    var BranchRevision = function (cs, branch) {
        this.id = branch.name + cs.id;
        this.changeset = cs;
        this.primaryBranch = branch;
        this.parentEdges = undefined;
        this.childEdges = undefined;
        this.branchPosition = NaN;
    };

    BranchRevision.prototype.setGroup = function (group) {
        this.group = group;
    };

    /*
     * A READ-ONLY array of this branch revision's direct parents.
     */
    BranchRevision.prototype.parents = function () {
        return this.parentEdges ?
            $.map(this.parentEdges, function (edge) {
                return edge.parentBranchRevision;
            }) :
            [];
    };
    memoize(BranchRevision.prototype, "parents");

    /*
     * A READ-ONLY array of this branch revision's direct children.
     */
    BranchRevision.prototype.children = function () {
        return this.childEdges ?
            $.map(this.childEdges, function (edge) {
                return edge.childBranchRevision;
            }) :
            [];
    };
    memoize(BranchRevision.prototype, "children");

    /*
     * A READ-ONLY array of this branch revision's direct parents.
     */
    BranchRevision.prototype.parentsOnBranch = function () {
        var primaryBranch = this.primaryBranch;
        return $.grep(this.parents(), function (parent) {
            return parent.primaryBranch === primaryBranch;
        });
    };
    memoize(BranchRevision.prototype, "parentsOnBranch");

    /*
     * A READ-ONLY array of this branch revision's direct children.
     */
    BranchRevision.prototype.childrenOnBranch = function () {
        var primaryBranch = this.primaryBranch;
        return $.grep(this.children(), function (child) {
            return child.primaryBranch === primaryBranch;
        });
    };
    memoize(BranchRevision.prototype, "childrenOnBranch");

    BranchRevision.prototype.isSparse = function () {
        return this.changeset.isSparse();
    };


    BranchRevision.prototype.isSparseAncestor = function () {
        return this.changeset.isSparseAncestor();
    };

    BranchRevision.prototype.isSparseDescendant = function () {
        return this.changeset.isSparseDescendant();
    };

    BranchRevision.prototype.isLive = function () {
        return this.changeset.isLive();
    };

    BranchRevision.prototype.addParentEdge = function (newEdge) {
        if (newEdge.childBranchRevision !== this) {
            throw "Cannot add this edge as parent.  The edge's child is not this changeset.";
        }

        if (this.parentEdges) {
            //if it already has its parent set, do nothing.
            if (Array.any(this.parentEdges, function (edge) {
                    return edge.parentChangeset === newEdge.parentChangeset
                        && edge.parentBranchRevision === newEdge.parentBranchRevision;
                })) {
                return this;
            }
        } else {
            this.parentEdges = [];
        }

        // add the parent to this changeset's list of parents.
        this.parentEdges.push(newEdge);

        return this;
    };
    eraseMemoized(BranchRevision.prototype, "addParentEdge", ["parents", "parentsOnBranch"]);

    BranchRevision.prototype.addChildEdge = function (newEdge) {
        if (newEdge.parentBranchRevision !== this) {
            throw "Cannot add this edge as child.  The edge's parent is not this changeset.";
        }

        if (this.childEdges) {
            //if it already has its parent set, do nothing.
            if (Array.any(this.childEdges, function (edge) {
                    return edge.childChangeset === newEdge.childChangeset
                        && edge.childBranchRevision === newEdge.childBranchRevision;
                })) {
                return this;
            }
        } else {
            this.childEdges = [];
        }

        this.childEdges.push(newEdge);

        return this;
    };
    eraseMemoized(BranchRevision.prototype, "addChildEdge", ["children", "childrenOnBranch"]);

    BranchRevision.prototype.addParent = function (parentChangeset, parentBranchRevision) {
        var edge = new Edge(parentChangeset, this.changeset, parentBranchRevision, this);

        this.addParentEdge(edge);
        if (parentBranchRevision) {
            parentBranchRevision.addChildEdge(edge);
        }

        return this;
    };

    BranchRevision.prototype.addChild = function (childChangeset, childBranchRevision) {
        var edge = new Edge(this.changeset, childChangeset, this, childBranchRevision);

        this.addChildEdge(edge);
        if (childBranchRevision) {
            childBranchRevision.addParentEdge(edge);
        }

        return this;
    };

    return BranchRevision;
})(AJS.$);
/*[{!branch_revision_js_ml1n535!}]*/