window.Targets = window.Targets || {};

/* Drawing Targets
 * Must implement the following interface:
 * getElement() - returns the root DOM element of the drawing area.
 * getWidth() - returns the width of the root element of the drawing area.
 * getHeight() - returns the height of the root element of the drawing area.
 * drawGroup(parent, translation, scale, width, height) - draw a group element, used for doing bulk transforms on
 *       elements. Returns an object representing the group.
 * drawPoint(parent, position, radius, color) - draw a circle
 * drawPath(parent, start, \* optional midpoints, *\ end, color, moveToBack) - draw a path through the given points.
 * erase(element) - destroy the given element and remove it from the drawing area.
 * clear() - destroy all elements and reset the drawing area.
 */
Targets.VML = (function ($) {
    // Imports
    var Point = Shapes.Point;
    var ElementData = Targets.ElementData;
    var EventProducer = FECRU.MIXINS.EventProducer;

    var vmlns = "vml";
    var tagStart = '<' + vmlns + ':';
    var tagEnd = ' class="' + vmlns + '">';

    function createElement(tagName, extraClasses) {
        var el;
        var classes = extraClasses ? vmlns + " " + extraClasses : vmlns;

        try {
            el = document.createElement(tagStart + tagName + ' class="' + classes + '">');
        } catch (e) {
            el = document.createElement('<group xmlns="urn:schemas-microsoft.com:vml" class="' + classes + '">');
        }
        return el;
    }

    var VMLTarget = function (containerElement, settings) {
        this.settings = $.extend({}, Targets.defaultSettings, settings);

        if (document.namespaces && !document.namespaces[vmlns]) {
            if ($.browser.version >= 8) {
                document.namespaces.add(vmlns, "urn:schemas-microsoft-com:vml");
                document.createStyleSheet().addRule(vmlns + "\\:group," + vmlns + "\\:polyline," + vmlns + "\\:oval," + vmlns + "\\:rect", "behavior:url(#default#VML); display:inline-block; position: absolute;");
            } else {
                document.namespaces.add(vmlns, "urn:schemas-microsoft-com:vml");
                document.createStyleSheet().addRule(vmlns + "\\:*", "behavior:url(#default#VML); display:inline-block; position: absolute;");
            }

        }

        var width = containerElement.offsetWidth;
        var height = containerElement.offsetHeight;
        var root = document.createElement("div");
        root.style.width = width;
        root.style.height = height;
        root.style.overflow = "hidden";
        containerElement.appendChild(root);

        var target = this;
        this.resize = function () {
            if (width !== containerElement.offsetWidth || height !== containerElement.offsetHeight) {
                width = containerElement.offsetWidth;
                height = containerElement.offsetHeight;
                root.style.width = width;
                root.style.height = height;

                target.trigger(Targets.TargetResizeEvent, {
                    width: width,
                    height: height
                });
            }
        };

        this.getElement = function () {
            return root;
        };
        this.getContainer = function () {
            return containerElement;
        };
        this.getWidth = function () {
            return width;
        };
        this.getHeight = function () {
            return height;
        };
    };
    $.extend(VMLTarget.prototype, EventProducer);

    function getRootElementData(target) {
        return target.rootData || (
                target.rootData = new ElementData(target.getElement(), {
                    position: new Point(0, 0),
                    width: target.getWidth(),
                    height: target.getHeight()
                })
            );
    }

    VMLTarget.prototype.drawGroup = function (parent, position, attributes) {
        //if no parent, assume root.
        parent = parent || getRootElementData(this);

        var position = new Point(position.x, position.y);

        attributes = $.extend({
            scale: 1,
            position: position,
            parent: parent
        }, attributes);

        var group = createElement('group');
        var elData = new ElementData(group, attributes);

        group.coordorigin = "0,0";
        this.setGroupAttributes(elData, attributes);

        parent.element.appendChild(group);

        return elData;
    };

    VMLTarget.prototype.setGroupAttributes = function (elData, attributes) {
        var element = elData.element;
        var data = elData.data;

        if (attributes) {
            data !== attributes && $.extend(elData.data, attributes);

            var position = data.position;
            var width = data.width;
            var height = data.height;
            var scale = data.scale;

            element.coordsize = Math.floor(width / scale) + "," + Math.floor(height / scale);
            element.style.left = position.x + 'px';
            element.style.top = position.y + 'px';
            element.style.width = width + 'px';
            element.style.height = height + 'px';
        }
    };

    VMLTarget.prototype.drawRect = function (parent, position, attributes, prepend) {
        //if no parent, assume root.
        parent = parent || getRootElementData(this);

        attributes = $.extend({}, attributes);

        var rect = createElement('rect', 'rect');
        var elData = new ElementData(rect, attributes);

        rect.stroked = false;
        rect.filled = true;
        this.setRectAttributes(elData, attributes);

        if (prepend) {
            parent.element.insertBefore(rect, parent.element.firstChild);
        } else {
            parent.element.appendChild(rect);
        }

        return elData;
    };

    VMLTarget.prototype.setRectAttributes = function (elData, attributes) {
        var element = elData.element;

        if (attributes) {
            element.style.left = attributes.x + "px";
            element.style.top = attributes.y + "px";
            element.style.width = attributes.width;
            element.style.height = attributes.height;
            element.fillcolor = attributes.fill;

            elData.data !== attributes && $.extend(elData.data, attributes);
        }
    };

    VMLTarget.prototype.drawPoint = function (parent, position, attributes) {
        //if no parent, assume root.
        parent = parent || getRootElementData(this);

        var position = new Point(position.x, position.y);

        attributes = $.extend({
            position: position,
            moveToBack: false
        }, this.settings, attributes);

        var circle = createElement('oval', 'point');
        var elementData = new ElementData(circle, attributes);

        circle.filled = true;
        circle.stroked = true;
        this.setPointAttributes(elementData, attributes);

        parent.element.appendChild(circle);

        return elementData;
    };

    VMLTarget.prototype.setPointAttributes = function (elData, attributes) {
        var element = elData.element;
        if (attributes) {
            if (attributes.radius) {
                var position = elData.data.position;
                var radius = attributes.radius;
                var diameter = 2 * radius;

                element.style.left = Math.floor(position.x - radius) + "px";
                element.style.top = Math.floor(position.y - radius) + "px";
                element.style.width = diameter + "px";
                element.style.height = diameter + "px";
            }
            attributes.fillColor && (element.fillcolor = attributes.fillColor);
            attributes.strokeColor && (element.strokecolor = attributes.strokeColor);
            attributes.strokeWidth && (element.strokeweight = attributes.strokeWidth + "px");

            elData.data !== attributes && $.extend(elData.data, attributes);
        }
    };

    function getRetracedPolylineStr(points) {
        var pointStrs = Array.map(points, function (point) {
            return point.x + "," + point.y;
        });

        // bit of a hack.  To avoid overlays, we retrace our steps.  VML will auto close the last segment,
        // so we do: point[0], point[1], ... point[n-1], point[n], point[n-1], ... point[2], point[1]
        var i;
        var len;
        var pointsStr = pointStrs.join(" ");

        for (i = pointStrs.length - 1; i > 0; i--) {
            pointsStr += " " + pointStrs[i];
        }
        return pointsStr;
    }

    VMLTarget.prototype.drawPath = function (parent, points, attributes) {
        //if no parent, assume root.
        parent = parent || getRootElementData(this);

        var pointsStr = getRetracedPolylineStr(points);
        var pointsCopy = Array.map(points, function (point) {
            return new Point(point.x, point.y);
        });

        attributes = $.extend({
            positions: pointsCopy
        }, this.settings, attributes);

        var line = createElement('polyline', 'path');
        var elementData = new ElementData(line, attributes);

        line.stroke = "true";
        line.fill = "false";
        line.points = pointsStr;
        setPathAttributesInternal(elementData, attributes, parent.element);

        return elementData;
    };

    VMLTarget.prototype.setPathAttributes = function (elData, attributes) {
        setPathAttributesInternal(elData, attributes);
        $.extend(elData.data, attributes);
    };
    function setPathAttributesInternal(elData, attributes, parentInternalUse) {
        var element = elData.element;
        if (attributes) {
            attributes.strokeColor && (element.strokecolor = attributes.strokeColor);
            attributes.strokeWidth && (element.strokeweight = attributes.strokeWidth);

            //don't move anything if they don't specify these attributes.
            var parentNode = parentInternalUse || element.parentNode;
            if (attributes.insertBeforeElement) {
                parentNode.insertBefore(element, attributes.insertBeforeElement.element);
            } else if (attributes.moveToBack !== undefined) {
                if (attributes.moveToBack) {
                    parentNode.insertBefore(element, parentNode.firstChild);
                } else {
                    parentNode.appendChild(element);
                }
            }
        }
    }

    VMLTarget.prototype.moveGroup = function (elData, position) {
        var el = elData.element;
        el.style.left = position.x + "px";
        el.style.top = position.y + "px";
        elData.data.position = new Point(position.x, position.y);
    };

    VMLTarget.prototype.movePoint = function (elData, position) {
        var el = elData.element;
        el.style.left = Math.floor(position.x - elData.data.radius) + "px";
        el.style.top = Math.floor(position.y - elData.data.radius) + "px";
        elData.data.position = new Point(position.x, position.y);
    };

    VMLTarget.prototype.movePath = function (elData, points) {
        var el = elData.element;
        var nextSibling;
        var parent = el.parentNode;

        // IE fuckup: points attribute cannot be changed while element is attached to the DOM.
        if (parent) {
            nextSibling = el.nextSibling;
            parent.removeChild(el);
        }

        el.points = getRetracedPolylineStr(points);
        elData.data.positions = points;

        if (parent) {
            if (nextSibling) {
                parent.insertBefore(el, nextSibling);
            } else {
                parent.appendChild(el);
            }
        }
    };

    VMLTarget.prototype.moveRectangle = function (elData, position) {
        var el = elData.element;
        el.style.left = position.x + "px";
        el.style.top = position.y + "px";
        elData.data.position = new Point(position.x, position.y);
    };

    VMLTarget.prototype.clear = function () {
        $(this.getElement()).children().remove();
    };
    VMLTarget.prototype.erase = function (el) {
        var element = el.element;
        if (element && element.parentNode) {
            element.parentNode.removeChild(element);
            el.element = null;
            return true;
        }
        return false;
    };

    return VMLTarget;
})(AJS.$);
/*[{!vml_target_js_4h6s534!}]*/