window.Targets = window.Targets || {};

/* Drawing Targets
 * Must implement the following interface:
 * getElement() - returns the root DOM element of the drawing area.
 * getWidth() - returns the width of the root element of the drawing area.
 * getHeight() - returns the height of the root element of the drawing area.
 * drawGroup(parent, translation, scale, width, height) - draw a group element, used for doing bulk transforms on
 *       elements. Returns an object representing the group.
 * drawPoint(parent, position, radius, color) - draw a circle
 * drawPath(parent, start, \* optional midpoints, *\ end, color, moveToBack) - draw a path through the given points.
 * erase(element) - destroy the given element and remove it from the drawing area.
 * clear() - destroy all elements and reset the drawing area.
 */
Targets.SVG = (function ($) {
    // Imports
    var Point = Shapes.Point;
    var Vector = Shapes.Vector;
    var ElementData = Targets.ElementData;
    var EventProducer = FECRU.MIXINS.EventProducer;

    function toTransform(scale, translation) {
        var svgTranslate = "translate(" + translation.x + "," + translation.y + ")";
        var svgScale = typeof(scale) === 'number' && scale !== 1 ?
        "scale(" + scale + "," + scale + ")" :
            "";

        return svgTranslate || svgScale ? [svgScale, svgTranslate].join(' ') : "";
    }

    var SVGTarget = function (containerElement, settings) {
        this.settings = $.extend({}, Targets.defaultSettings, settings);
        var width = containerElement.offsetWidth;
        var height = containerElement.offsetHeight;
        var root = document.createElementNS(this.svgns, "svg");
        root.style.width = width;
        root.style.height = height;
        root.style.overflow = "hidden";

        var target = this;
        this.resize = function () {
            if (width !== containerElement.offsetWidth || height !== containerElement.offsetHeight) {
                width = containerElement.offsetWidth;
                height = containerElement.offsetHeight;
                root.style.width = width + 'px';
                root.style.height = height + 'px';

                target.trigger(Targets.TargetResizeEvent, {
                    width: width,
                    height: height
                });
            }
        };

        containerElement.appendChild(root);

        this.getElement = function () {
            return root;
        };
        this.getContainer = function () {
            return containerElement;
        };
        this.getWidth = function () {
            return width;
        };
        this.getHeight = function () {
            return height;
        };
    };
    $.extend(SVGTarget.prototype, EventProducer);
    SVGTarget.prototype.svgns = "http://www.w3.org/2000/svg";

    function getRootElementData(target) {
        return new ElementData(target.getElement(), {
            position: new Point(0, 0),
            width: target.getWidth(),
            height: target.getHeight()
        });
    }

    SVGTarget.prototype.drawGroup = function (parent, position, attributes) {
        //if no parent, assume root.
        parent = parent || getRootElementData(this);

        var position = new Point(position.x, position.y);

        attributes = $.extend({
            scale: 1,
            position: position
        }, attributes);

        var group = document.createElementNS(this.svgns, 'g');
        var elData = new ElementData(group, attributes);

        this.setGroupAttributes(elData, attributes);

        parent.element.appendChild(group);

        return elData;
    };

    SVGTarget.prototype.setGroupAttributes = function (elData, attributes) {
        var element = elData.element;

        if (attributes) {
            var data = elData.data;
            if (attributes.scale) {
                var position = data.position;
                var transform = toTransform(attributes.scale, position);

                if (transform) {
                    element.setAttribute('transform', transform);
                }
            }

            elData.data !== attributes && $.extend(elData.data, attributes);
        }
    };

    SVGTarget.prototype.drawRect = function (parent, position, attributes, prepend) {
        //if no parent, assume root.
        parent = parent || getRootElementData(this);

        attributes = $.extend({}, attributes);

        var rect = document.createElementNS(this.svgns, 'rect');
        var elData = new ElementData(rect, attributes);

        this.setRectAttributes(elData, attributes);

        if (prepend) {
            parent.element.insertBefore(rect, parent.element.firstChild);
        } else {
            parent.element.appendChild(rect);
        }

        return elData;
    };

    SVGTarget.prototype.setRectAttributes = function (elData, attributes) {
        var element = elData.element;

        if (attributes) {
            element.setAttribute('x', attributes.x + "px");
            element.setAttribute('y', attributes.y + "px");
            element.setAttribute('width', attributes.width + "px");
            element.setAttribute('height', attributes.height + "px");
            element.setAttribute('fill', attributes.fill);

            elData.data !== attributes && $.extend(elData.data, attributes);
        }
    };

    SVGTarget.prototype.drawPoint = function (parent, position, attributes) {
        //if no parent, assume root.
        parent = parent || getRootElementData(this);

        attributes = $.extend({
            radius: 9,
            position: new Point(position.x, position.y)
        }, this.settings, attributes);

        var circle = document.createElementNS(this.svgns, 'circle');
        var elementData = new ElementData(circle, attributes);

        circle.setAttribute('cy', position.y + 'px');
        circle.setAttribute('cx', position.x + 'px');
        circle.setAttribute('class', 'point');
        this.setPointAttributes(elementData, attributes);

        parent.element.appendChild(circle);

        return elementData;
    };
    SVGTarget.prototype.setPointAttributes = function (elData, attributes) {
        var element = elData.element;
        if (attributes) {
            attributes.radius && element.setAttribute('r', attributes.radius + 'px');
            attributes.fillColor && element.setAttribute('fill', attributes.fillColor);
            attributes.strokeColor && element.setAttribute('stroke', attributes.strokeColor);
            attributes.strokeWidth && element.setAttribute('stroke-width', attributes.strokeWidth + "px");

            elData.data !== attributes && $.extend(elData.data, attributes);
        }
    };

    function getPathStr(points) {

        if (points.length % 3 === 0) {
            var pointsCopy = points.slice(0);
            var startCmd = "M";
            var str = "";

            while (pointsCopy.length) {
                var p1 = pointsCopy.shift();
                var p2 = pointsCopy.shift();
                var p3 = pointsCopy.shift();

                str += startCmd + p1.x + "," + p1.y +
                    " Q" + p2.x + "," + p2.y + " " + p3.x + "," + p3.y;
                startCmd = " L";
            }
            return str;
        } else { // fallback to straight lines.
            return Array.reduce(points, "", function (str, point) {
                return (str ? str + " L" : "M") + point.x + "," + point.y;
            });
        }
    }

    SVGTarget.prototype.drawPath = function (parent, points, attributes) {
        //if no parent, assume root.
        parent = parent || getRootElementData(this);

        var pointsCopy = Array.map(points, function (point) {
            return new Point(point.x, point.y);
        });

        attributes = $.extend({
            positions: pointsCopy
        }, this.settings, attributes);

        var line = document.createElementNS(this.svgns, 'path');
        var elementData = new ElementData(line, attributes);

        line.setAttribute('d', getPathStr(points));
        line.setAttribute('fill', "none");
        line.setAttribute('class', 'path');
        setPathAttributesInternal(elementData, attributes, parent.element);

        return elementData;
    };

    SVGTarget.prototype.setPathAttributes = function (elData, attributes) {
        setPathAttributesInternal(elData, attributes);
        $.extend(elData.data, attributes);
    };
    function setPathAttributesInternal(elData, attributes, parentInternalUse) {
        var element = elData.element;
        if (attributes) {
            attributes.strokeColor && element.setAttribute('stroke', attributes.strokeColor);
            attributes.strokeWidth && element.setAttribute('stroke-width', attributes.strokeWidth);

            //don't move anything if they don't specify DOM placement attributes.
            var parentNode = parentInternalUse || element.parentNode;
            if (attributes.insertBeforeElement) {
                parentNode.insertBefore(element, attributes.insertBeforeElement.element);
            } else if (attributes.moveToBack !== undefined) {
                if (attributes.moveToBack) {
                    parentNode.insertBefore(element, parentNode.firstChild);
                } else {
                    parentNode.appendChild(element);
                }
            }
        }
    };

    SVGTarget.prototype.moveGroup = function (elData, position) {
        var el = elData.element;
        el.setAttribute('transform', toTransform(elData.scale, position));
        elData.data.position = new Point(position.x, position.y);
    };
    SVGTarget.prototype.movePoint = function (elData, position) {
        var el = elData.element;
        el.setAttribute('cx', position.x + "px");
        el.setAttribute('cy', position.y + "px");
    };
    SVGTarget.prototype.movePath = function (elData, points) {
        var el = elData.element;
        var pointsStr = getPathStr(points);
        el.setAttribute('d', pointsStr);
    };
    SVGTarget.prototype.moveRectangle = function (elData, position) {
        var el = elData.element;
        el.setAttribute('x', position.x + "px");
        el.setAttribute('y', position.y + "px");
    };

    SVGTarget.prototype.clear = function () {
        $(this.getElement()).children().remove();
    };
    SVGTarget.prototype.erase = function (el) {
        var element = el.element;
        if (element && element.parentNode) {
            element.parentNode.removeChild(element);
            el.element = null;
            return true;
        }
        return false;
    };
    return SVGTarget;
})(AJS.$);
/*[{!svg_target_js_9bu2533!}]*/