Targets.Raphael = (function ($, Raphael) {

    var EventProducer = FECRU.MIXINS.EventProducer;
    var Vector = Shapes.Vector;
    var Point = Shapes.Point;
    var ElementData = Targets.ElementData;

    function RaphaelTarget(containerElement, settings) {
        this.settings = $.extend({}, Targets.defaultSettings, settings);
        var width = containerElement.offsetWidth;
        var height = containerElement.offsetHeight;
        var paper = Raphael(containerElement, width, height);
        var root = paper.group();

        var target = this;
        this.resize = function () {
            if (width !== containerElement.offsetWidth || height !== containerElement.offsetHeight) {
                width = containerElement.offsetWidth;
                height = containerElement.offsetHeight;
                paper.setSize(width, height);

                target.trigger(Targets.TargetResizeEvent, {
                    width: width,
                    height: height
                });
            }
        };

        this.getPaper = function () {
            return paper;
        };
        this.getElement = function () {
            return root;
        };
        this.getContainer = function () {
            return containerElement;
        };
        this.getWidth = function () {
            return width;
        };
        this.getHeight = function () {
            return height;
        };
    }

    $.extend(RaphaelTarget.prototype, EventProducer);

    function getRootElementData(target) {
        return new ElementData(target.getElement(), {
            position: new Point(0, 0),
            width: target.getWidth(),
            height: target.getHeight()
        });
    }

    var resetTransform;
    if (Raphael.svg) {
        resetTransform = function (group) {
            var transform = "";
            group.attrs.translation && (transform += "translate(" + group.attrs.translation + ")");
            group.attrs.scale && (transform += "scale(" + group.attrs.scale + ")");
            group.attrs.rotation && (transform += "rotation(" + group.attrs.rotation + ")");
            group.node.setAttribute("transform", transform);
        };
        Raphael.prototype.group = function () {
            var res;
            var el = document.createElementNS(this.svgns, 'g');
            el.style.webkitTapHighlightColor = "rgba(0,0,0,0)";
            res = new Raphael.el.constructor(el, this);
            res.type = 'group';
            res.push = function (el) {
                this.node.appendChild(el.node);
            };
            res.translate = function (x, y) {
                this.attrs.translation = Math.floor(x) + " " + Math.floor(y);
                resetTransform(this);
                return this;
            };
            res.scale = function (x, y) {
                this.attrs.scale = Math.floor(x) + " " + Math.floor(y);
                resetTransform(this);
                return this;
            };

            this.canvas.appendChild(res.node);

            return res;
        };
    } else {
        resetTransform = function (group) {
            if (group.attrs.translation) {
                var xy = group.attrs.translation.split(/\s+/);
                group.node.style.left = xy[0];
                group.node.style.top = xy[1];
            }
        };
        Raphael.prototype.group = function (width, height) {
            var res;
            var el;
            var groupingGroup;

            try {
                !document.namespaces.rvml && document.namespaces.add("rvml", "urn:schemas-microsoft-com:vml");
                el = document.createElement('<rvml:group class="rvml">');
                groupingGroup = document.createElement('<rvml:group class="rvml">');
            } catch (e) {
                el = document.createElement('<group xmlns="urn:schemas-microsoft.com:vml" class="rvml">');
                groupingGroup = document.createElement('<group xmlns="urn:schemas-microsoft.com:vml" class="rvml">');
            }
            el.style.position = 'absolute';
            groupingGroup.appendChild(el);
            res = new Raphael.el.constructor(el, groupingGroup, this);

            res.type = 'group';
            res.push = function (el) {
                this.node.appendChild(el.Group);
            };
            res.translate = function (x, y) {
                this.attrs.translation = Math.floor(x) + " " + Math.floor(y);
                resetTransform(this);
                return this;
            };
            res.scale = function (x, y) {
                this.attrs.scale = Math.floor(x) + " " + Math.floor(y);
                resetTransform(this);
                return this;
            };
            res.resize = function (width, height) {
                this.node.setAttribute('coordsize', width + "px," + height + "px");
            };

            res.resize(width, height);

            this.canvas.appendChild(groupingGroup);

            return res;
        };
    }

    RaphaelTarget.prototype.drawGroup = function (parent, translation, scale, width, height) {
        //if no parent, assume root.
        parent = parent || getRootElementData(this);
        translation = translation || new Vector(0, 0);

        var group = this.getPaper().group(width, height).translate(translation.x, translation.y).scale(scale, scale);

        parent.element.push(group);

        return new ElementData(group, {
            position: new Point(translation.x, translation.y),
            width: width,
            height: height,
            scale: scale
        });
    };

    RaphaelTarget.prototype.drawPoint = function (parent, position, attributes) {
        //if no parent, assume root.
        parent = parent || getRootElementData(this);

        attributes = $.extend({radius: 9}, this.settings, attributes);

        var circle = this.getPaper().circle(position.x, position.y, attributes.radius);
        var data = $.extend({
            position: new Point(position.x, position.y)
        }, attributes);
        var elementData = new ElementData(circle, data);

        this.setPointAttributes(elementData, attributes);

        parent.element.push(circle);

        return elementData;
    };
    RaphaelTarget.prototype.setPointAttributes = function (elData, attributes) {
        var element = elData.element;
        if (attributes) {
            attributes.radius && element.attr('r', attributes.radius);
            attributes.fillColor && element.attr('fill', attributes.fillColor);
            attributes.strokeColor && element.attr('stroke', attributes.strokeColor);
            attributes.strokeWidth && element.attr('stroke-width', attributes.strokeWidth);
            $.extend(elData.data, attributes);
        }
    };

    RaphaelTarget.prototype.drawPath = function (parent, points, attributes) {
        //if no parent, assume root.
        parent = parent || getRootElementData(this);

        attributes = $.extend({}, this.settings, attributes);

        var pointsCopy = Array.map(points, function (point) {
            return new Point(point.x, point.y);
        });
        var data = $.extend({positions: pointsCopy}, attributes);
        var line = this.getPaper().path("M" + Array.reduce(points, "", function (str, point) {
                return (str ? str + "L" : "") + point.x + " " + point.y;
            }));
        var elementData = new ElementData(line, data);

        line.attr('fill', "none");
        setPathAttributesInternal(elementData, data, parent.element.node);

        return elementData;
    };

    RaphaelTarget.prototype.setPathAttributes = function (elData, attributes) {
        setPathAttributesInternal(elData, attributes);
    };
    function setPathAttributesInternal(elData, attributes, parentInternalUse) {
        var element = elData.element;
        if (attributes) {
            attributes.strokeColor && element.attr('stroke', attributes.strokeColor);
            attributes.strokeWidth && element.attr('stroke-width', attributes.strokeWidth);

            //don't move anything if they don't specify DOM placement attributes.
            var node = Raphael.svg ? element.node : element.Group;
            var parentNode = parentInternalUse || node.parentNode;

            if (attributes.insertBeforeElement) {
                var insertBeforeRaph = attributes.insertBeforeElement.element;
                var insertBeforeElement = Raphael.svg ? insertBeforeRaph.node : insertBeforeRaph.Group;

                parentNode.insertBefore(node, insertBeforeElement);
            } else if (attributes.moveToBack !== undefined) {
                if (attributes.moveToBack) {
                    parentNode.insertBefore(node, parentNode.firstChild);
                } else {
                    parentNode.appendChild(node);
                }
            }
            $.extend(elData.data, attributes);
        }
    }

    RaphaelTarget.prototype.moveGroup = function (elData, position) {
        elData.element.translate(position.x, position.y);

        elData.data.position = new Point(position.x, position.y);
    };
    RaphaelTarget.prototype.movePoint = function (elData, position) {
        var el = elData.element;
        var oldPosition = elData.data.position;

        el.translate(position.x - oldPosition.x, position.y - oldPosition.y);
        elData.data.position = position;
    };
    RaphaelTarget.prototype.movePath = function (elData, points) {
        var el = elData.element;
        var pointsStr = "";

        Array.each(points, function (pos, i) {
            pointsStr += (i === 0 ? "M" : "L") + pos.x + " " + pos.y;
        });
        el.attr('path', pointsStr);
    };

    RaphaelTarget.prototype.clear = function () {
        this.getPaper().clear();
    };
    RaphaelTarget.prototype.erase = function (el) {
        var element = el.element;
        if (element) {
            element.remove();
            el.element = null;
            return true;
        }
        return false;
    };

    return RaphaelTarget;
})(AJS.$, Raphael);
/*[{!raphael_target_js_eb98532!}]*/