var Shapes = (function () {
    var Point = function (x, y) {
        this.x = Number(x);
        this.y = Number(y);
    };
    Point.prototype.isValid = function () {
        return !isNaN(this.x) && !isNaN(this.y);
    };
    Point.prototype.toString = function () {
        return '(' + this.x.toFixed(2) + ',' + this.y.toFixed(2) + ')';
    };
    Point.prototype.isOrigin = function () {
        return this.x === 0 && this.y === 0;
    };

    var Vector = function (x, y) {
        this.x = Number(x);
        this.y = Number(y);
    };
    Vector.prototype.isValid = function () {
        return !isNaN(this.x) && !isNaN(this.y) && Number(this.x) === this.x && Number(this.y) === this.y;
    };
    Vector.prototype.add = function (vector) {
        if (!this.isValid() || !(vector instanceof Vector) || !vector.isValid()) {
            return undefined;
        }
        return new Vector(this.x + vector.x, this.y + vector.y);
    };
    Vector.prototype.equals = function (vector) {
        return vector && (vector instanceof Vector) && this.x === vector.x && this.y === vector.y;
    };
    Vector.prototype.getMagnitude = function () {
        return Math.sqrt(this.x * this.x + this.y * this.y);
    };
    Vector.prototype.multiply = function (scalar) {
        return new Vector(this.x * scalar, this.y * scalar);
    };
    Vector.prototype.toString = function () {
        return '(' + this.x.toFixed(2) + ',' + this.y.toFixed(2) + ')';
    };

    /*
     * Takes in two Point instances.  The points will be reordered
     * such that point1 is left of point2 (or below point2 if the line is perfectly vertical).
     */
    var LineSegment = function (point1, point2) {
        if (!point1 || !point2) {
            throw "You must specify two Point objects";
        }
        if (point1.x < point2.x ||
            (point1.x === point2.x && point1.y <= point2.y)) {
            this.point1 = point1;
            this.point2 = point2;
        } else {
            this.point1 = point2;
            this.point2 = point1;
        }
    };
    LineSegment.hasIntersection = function (lineSegmentA, lineSegmentB) {
        //cache EVERYTHING
        var pA1 = lineSegmentA.point1;
        var pA2 = lineSegmentA.point2;
        var pB1 = lineSegmentB.point1;
        var pB2 = lineSegmentB.point2;
        var xA1 = pA1.x;
        var yA1 = pA1.y;
        var xA2 = pA2.x;
        var yA2 = pA2.y;
        var xB1 = pB1.x;
        var yB1 = pB1.y;
        var xB2 = pB2.x;
        var yB2 = pB2.y;

        /*
         The math is from: http://local.wasp.uwa.edu.au/~pbourke/geometry/lineline2d/

         Line segment A has endpoints P1(x1, y1) and P2(x2, y2)
         Line segment B has endpoints P3(x3, y3) and P4(x4, y4)

         Point on line segment A = Pa = P1 + ua ( P2 - P1 ); ua is between 0 and 1
         Point on line segment B = Pb = P3 + ub ( P4 - P3 ); ub is between 0 and 1

         Intersection of A and B is where Pa = Pb
         In terms of x and y, this is:
         x1 + ua (x2 - x1) = x3 + ub (x4 - x3)
         y1 + ua (y2 - y1) = y3 + ub (y4 - y3)

         Solving for ua and ua gives:
         ua = [(x4 - x3)(y1 - y3) - (y4 - y3)(x1 - x3)] / [(y4 - y3)(x2 - x1) - (x4 - x3)(y2 - y1)]
         ub = [(x2 - x1)(y1 - y3) - (y2 - y1)(x1 - x3)] / [(y4 - y3)(x2 - x1) - (x4 - x3)(y2 - y1)]
         Notice the denominators are the same.

         Notes I've taken for granted and can't explain:
         If the denominator for the equations for ua and ub is 0 then the two lines are parallel.
         If the denominator and numerator for the equations for ua and ub are 0 then the two lines are coincident.

         Since we're checking the line segments, ua and ub have to remain between 0 and 1 for their to be a collision.
         Because coincident lines have a denominator of 0, uA and uB are undefined, and checking 0 <= u <= 1 fails.
         Instead I rotate the coincident line segments parallel to x.
         */
        var yB2mYB1 = yB2 - yB1;
        var xA2mXA1 = xA2 - xA1;
        var xB2mXB1 = xB2 - xB1;
        var yA2mYA1 = yA2 - yA1;
        var yA1mYB1 = yA1 - yB1;
        var xA1mXB1 = xA1 - xB1;

        var uDenominator = yB2mYB1 * xA2mXA1 - xB2mXB1 * yA2mYA1;
        var uANumerator = xB2mXB1 * yA1mYB1 - yB2mYB1 * xA1mXB1;
        var uBNumerator = xA2mXA1 * yA1mYB1 - yA2mYA1 * xA1mXB1;
        var parallel = uDenominator === 0;

        if (parallel) {
            var coincident = uANumerator === 0 && uBNumerator === 0;
            if (coincident) { // They are both the same line, but possibly different segments of it.
                // First I rotate the points around the origin such that they are all parallel to the x-axis.
                // This lets me use the x coordinate alone to determine overlap (all y's should be the same)
                var angleToRotate = -Math.atan2(yA2mYA1, xA2mXA1);
                var cosAngle = Math.cos(angleToRotate);
                var sinAngle = Math.sin(angleToRotate);
                var xA1rotated = xA1 * cosAngle - yA1 * sinAngle;
                var xA2rotated = xA2 * cosAngle - yA2 * sinAngle;
                var xB1rotated = xB1 * cosAngle - yB1 * sinAngle;
                var xB2rotated = xB2 * cosAngle - yB2 * sinAngle;

                // the following line is only sufficient for intersection with the following assumptions:
                // 1) xA1rotated <= xA2rotated && xB1rotated <= xB2rotated - should be met by LineSegment ctor logic
                // 2) yA1rotated == yA2rotated == yB1rotated == yB2rotated - should be met by the rotation we did.
                return xA1rotated <= xB2rotated && xB1rotated <= xA2rotated;
            } else {
                return false;
            }
        } else {
            var uA = uANumerator / uDenominator;
            var uB = uBNumerator / uDenominator;

            return uA <= 1 && uA >= 0 && uB <= 1 && uB >= 0;
        }
    };
    LineSegment.prototype.intersects = function (other) {
        if (other instanceof LineSegment) {
            return LineSegment.hasIntersection(this, other);
        } else {
            throw "Not implemented."
        }
    };

    var Rectangle = function (x, y, width, height) {
        this.x = x;
        this.y = y;
        this.width = width;
        this.height = height;
        this.topLeft = new Point(x, y);
        this.topRight = new Point(x + width, y);
        this.bottomLeft = new Point(x, y + height);
        this.bottomRight = new Point(x + width, y + height);
        this.top = new LineSegment(this.topLeft, this.topRight);
        this.bottom = new LineSegment(this.bottomLeft, this.bottomRight);
        this.left = new LineSegment(this.topLeft, this.bottomLeft);
        this.right = new LineSegment(this.topRight, this.bottomRight);
    };
    /*
     * Determines whether the line segment passed in intersects with this Rectangle.
     * the line segment is considered to intersect if it intersects with any edge of
     * the Rectangle or is completely contained within the Rectangle
     * */
    Rectangle.prototype.intersectsLineSegment = function (ls) {
        return ( // is contained in Rectangle
                ls.point1.x > this.x &&
                ls.point1.y > this.y &&
                ls.point1.x < this.x + this.width &&
                ls.point1.y < this.y + this.height
            ) ||
            ls.intersects(this.left) || // intersects the left edge
            ls.intersects(this.right) || // intersects the right edge
            ls.intersects(this.top) || // intersects the top edge
            ls.intersects(this.bottom); // intersects the bottom edge
    };
    Rectangle.prototype.intersectsRectangle = function (rect) {
        return this.x <= rect.x + rect.width &&
            rect.x <= this.x + this.width &&
            this.y <= rect.y + rect.height &&
            rect.y <= this.y + this.height;
    };

    Rectangle.prototype.toString = function () {
        return '(' + this.x.toFixed(2) + ',' + this.y.toFixed(2) + ',' +
            this.width.toFixed(2) + ',' + this.height.toFixed(2) + ')';
    };

    return {
        Point: Point,
        Vector: Vector,
        LineSegment: LineSegment,
        Rectangle: Rectangle
    };
})();
/*[{!shapes_js_q4r352s!}]*/