/**
 * Implements View objects for Changesets, Edges, Groups, Slices, and the Graph, and builds up a hierarchy of the
 * views. When you call the draw method on the presenter, it updates these views and redraws them.
 *
 *  - DefaultPresenter (ctor; takes in a target to draw to, and has:
 *       .draw(view)
 *       .translate(vector)
 *       .createGraphView(ChangesetDAG) returns a view.
 *
 * The purpose of splitting this out is to
 * 1. simplify the model files a bit more.
 * 2. allow someone to implement a new kind of view (e.g., HTML Tables like Sharkie thought up) that wouldn't even use
 *    x and y positions like SVG and VML do. They can implement their own GraphView and use that instead
 *    of this.
 */

FE.VIS.DRAWING = (function ($, templateFactory) {

    /* Imports */
    var Point = Shapes.Point;
    var Vector = Shapes.Vector;
    var Rectangle = Shapes.Rectangle;
    var VIS = FE.VIS;
    var Configuration = VIS.Configuration;
    var CHANGESET_SPACING = Configuration.CHANGESET_SPACING;
    var CHANGESET_RADIUS = Configuration.CHANGESET_RADIUS;
    var BRANCH_SPACING = Configuration.BRANCH_SPACING;
    var METADATA_PADDING_LEFT = Configuration.METADATA_PADDING_LEFT;
    var EDGE_HOVER_WIDTH = Configuration.EDGE_HOVER_WIDTH;
    var NODE_HOVER_RADIUS = Configuration.NODE_HOVER_RADIUS;
    var EventProducer = FECRU.MIXINS.EventProducer;
    var HIGHLIGHTS = VIS.HIGHLIGHTS;
    var HighlightContext = HIGHLIGHTS.HighlightContext;
    var COLOR_BY_BRANCH = HIGHLIGHTS.COLOR_BY_BRANCH;
    var HighlightPopup = VIS.HIGHLIGHT_POPUP;
    var HighlightLozenge = VIS.HIGHLIGHT_LOZENGE;
    var fallbackBranchColor = FE.VIS.fallbackBranchColor;
    var Map = FECRU.DATA_STRUCTURES.Map;
    var pad = FECRU.pad;

    var createBackgroundColor = (function (isIE, opacity) {
        var cache = new Map();
        var threeHexPattern = /#[a-fA-F0-9]{3}/;
        var sixHexPattern = /#[a-fA-F0-9]{6}/;
        var rgbPattern = /rgb[^\(]*\([^\d]*(\d{1,3}),[^\d]*(\d{1,3}),[^\d]*(\d{1,3})[^\)]*\)/;
        var opacityString;
        var stylePrefix;
        var createColor;
        var minValue = (1 - opacity) * 255;

        function padLeft(str) {
            return pad(str, 2, false, "0");
        }

        function translateToRGB(color) {
            var match;
            var ret = null;

            if (!color) {
                ret = null;
            } else if (match = color.match(sixHexPattern)) {
                ret = [
                    parseInt(color.substring(1, 3), 16),
                    parseInt(color.substring(3, 5), 16),
                    parseInt(color.substring(5, 7), 16)
                ];
            } else if (match = color.match(threeHexPattern)) {
                ret = [
                    parseInt(color[1] + color[1], 16),
                    parseInt(color[2] + color[2], 16),
                    parseInt(color[3] + color[3], 16)
                ];
            } else if (match = color.match(rgbPattern)) {
                ret = match.slice(1);
            }
            return ret;
        }

        if (!!isIE) {
            opacityString = (Math.floor(opacity * 255)).toString(16);
            stylePrefix = "progid:DXImageTransform.Microsoft.gradient";
            createColor = function (color) {
                if (!color) {
                    return {filter: color};
                }
                if (!cache.has(color)) {
                    var rgb = translateToRGB(color);
                    var newColor = "#"
                        + opacityString
                        + padLeft(rgb[0].toString(16))
                        + padLeft(rgb[1].toString(16))
                        + padLeft(rgb[2].toString(16));
                    var style = stylePrefix + "(startColorStr=" + newColor + ", endColorStr=" + newColor + ")";

                    cache.set(color, style);
                }
                return {filter: cache.get(color)};
            };
        } else {
            opacityString = opacity.toString();
            stylePrefix = "rgba";
            createColor = function (color) {
                if (!color) {
                    return {backgroundColor: color};
                }
                if (!cache.has(color)) {
                    var rgb = translateToRGB(color);
                    var newColor = rgb[0] + ", "
                        + rgb[1] + ", "
                        + rgb[2] + ", "
                        + opacityString;
                    var style = stylePrefix + "(" + newColor + ")";

                    cache.set(color, style);
                }
                return {backgroundColor: cache.get(color)};
            };
        }
        return createColor;
    })($.browser.msie && $.browser.version < 9, 0.1);

    var EdgeProximityEvent = "EdgeProximityEvent";

    function abstractFunction() {
        throw "Cannot call abstract method";
    }

    function removeDisposedViews(views) {
        var newViews = [];
        views && Array.each(views, function (view) {
            if (!view.model.disposed) {
                newViews.push(view);
            }
        });
        return newViews;
    }

    function addNewViews(thisView, currentViews, newModels, viewFactory) {
        newModels && Array.each(newModels, function (model) {
            var viewAlreadyCreated = Array.any(currentViews, function (view) {
                return view.model === model;
            });
            if (!viewAlreadyCreated) {
                currentViews.push(viewFactory(model, thisView));
            }
        });
    }

    function createContext(horizontal, flippedTimewise, spacing, changesetRadius) {
        var pathNubX = spacing / 2;
        var pathNubY = spacing / 2;

        var context = {
            getSpacing: function () {
                return spacing;
            },
            getChangesetRadius: function () {
                return changesetRadius;
            },
            setHighlightContext: function (highlightContext) {
                this.highlightContext = highlightContext;
            },
            highlightContext: new HighlightContext(null)
        };

        /**
         These functions allow us to encapsulate the direction of the drawn graph.
         They map (time, branch) coordinates to (x, y) coordinates depending on whether you are displaying the graph
         vertically or horizontally.

         The "flipped time" transformation we do throughout this file should be fairly simple:
             (time, breadth) => (-time, breadth)
            It's complicated by the fact that we use svg:g elements with their own coordinate systems. To avoid putting
            a negative width/height on these groups, we have to switch the "first" and "last" points.
            We flip the sign on Slice positions and then subtract their time-width.  This means a Rectangle:
                top-left: (20, 0);
                bottom-right: (45, 50);
                width: 25;
                height: 50;
            becomes:
                top-left: (-(20) - 25, 0);
                bottom-right: (-25, 50);
                width: 25;
                height: 50;
              AKA:
                top-left: (-45, 0);                 <= we must switch the left and right
                bottom-right: (-25, 50);            <= sides so we don't have negative width.
                width: 25;
                height: 50;
        */
        context.createPoint = horizontal ?
            function (timewise, branchwise) {
                return new Point(timewise, branchwise);
            } :
            function (timewise, branchwise) {
                return new Point(branchwise, timewise);
            };
        context.createVector = horizontal ?
            function (timewise, branchwise) {
                return new Vector(timewise, branchwise);
            } :
            function (timewise, branchwise) {
                return new Vector(branchwise, timewise);
            };
        context.createRectangle = horizontal ?
            function (timewiseStart, breadthwiseStart, length, breadth) {
                return new Rectangle(timewiseStart, breadthwiseStart, length, breadth);
            } :
            function (timewiseStart, breadthwiseStart, length, breadth) {
                return new Rectangle(breadthwiseStart, timewiseStart, breadth, length);
            };
        context.getChangesetTimewise = flippedTimewise ?
            function (timePosition, parentLength) {
                return (parentLength - timePosition - 1);
            } :
            function (timePosition, parentLength) {
                return timePosition;
            };
        context.getSliceTimewise = flippedTimewise ?
            function (timePosition, length) {
                return ( -timePosition - length);
            } :
            function (timePosition) {
                return timePosition;
            };
        context.getGraphTimewise = context.getSliceTimewise;
        context.getGraphTimePosition = flippedTimewise ?
            function (dag) {
                var slices = dag.dagSlices;
                var slicesLength = slices.length;
                var length = dag.getLength();

                if (slicesLength) {
                    return slices[slicesLength - 1].timePosition + slices[slicesLength - 1].length - length;
                } else {
                    return 0;
                }
            } :
            function (dag) {
                var slices = dag.dagSlices;
                return slices.length ? slices[0].timePosition : 0
            };
        context.createGraphSpaceRectangle = flippedTimewise ?
            function (x, y, width, height) {
                var rect = context.createRectangle(x, y, width, height);
                rect.x = -rect.x - rect.width;
                return rect;
            } :
            function (x, y, width, height) {
                return context.createRectangle(x, y, width, height);
            };

        context.translateBy = horizontal ?
            function (translation, vector) {
                translation.x += vector.x;
            } :
            function (translation, vector) {
                translation.y += vector.y;
            };

        context.translateTo = horizontal ?
            function (translation, position) {
                translation.x = -position.x + (spacing / 2);
            } :
            function (translation, position) {
                translation.y = -position.y + (spacing / 2);
            };

        /**
         * Given one of the changeset's containers (the group, slice, or dag),
         * this function will convert the changeset's local coordinates into that container's
         * coordinate system.
         * If a changeset is Sparse, then its local coordinates are unknown and assumed infinitely far.  In that case,
         * this function will position it at the further of the appropriate edge of the container or the viewport.
         */
        context.getPositionInContainerCoordinates = function (changeset, branchRevision, containerView, viewport) {
            var position;
            var timewise;
            var breadthwise;
            var containerModel = containerView.model;

            if (changeset.isLive()) {
                var group = branchRevision.group;
                var slice = group.slice;


                timewise = context.getChangesetTimewise(changeset.timePosition, slice.getLength());
                breadthwise = branchRevision.branchPosition;
                if (containerModel !== group) {
                    breadthwise += group.branch.branchPosition;

                    if (containerModel !== slice) {
                        timewise += context.getSliceTimewise(slice.timePosition, slice.length);
                    }
                }
                position = context.createPoint(
                    timewise * spacing,
                    breadthwise * spacing
                );
            } else if (changeset.isSparse()) {
                var branch = branchRevision ? branchRevision.primaryBranch : null;

                // time positions should be Infinity or NaN, so translation/scale is moot. Only the sign matters.
                timewise = (flippedTimewise ? -1 : 1) * (changeset.timePosition);

                // branch positions, on the other hand, are only Infinity for sparseOnBranch.
                breadthwise = (branchRevision ? branchRevision.branchPosition : Infinity) + (branch ? branch.branchPosition : 0);

                position = context.createPoint(
                    timewise * spacing,
                    breadthwise * spacing
                );
                var graphBounds = containerView.boundingBox;

                if (position.x === -Infinity) {
                    position.x = Math.min(graphBounds.x, viewport.x);
                } else if (position.x === Infinity) {
                    var lastGraphX = graphBounds.x + graphBounds.width;
                    var lastViewportX = viewport.x + viewport.width;

                    position.x = Math.max(lastGraphX, lastViewportX);
                }
                if (position.y === -Infinity) {
                    position.y = Math.min(graphBounds.y, viewport.y);
                } else if (position.y === Infinity) {
                    var lastGraphY = graphBounds.y + graphBounds.height;
                    var lastViewportY = viewport.y + viewport.height;

                    position.y = Math.max(lastGraphY, lastViewportY);
                }
            }
            return position;
        };

        /**
         * An edge can be at one of a few levels, so it must calculate the parent/child changeset positions in a common
         * coordinate space.
         * It also must calculate a few midpoints between the changesets for a prettier line.
         */
        /*eslint-disable complexity*/
        context.calculateEdgePositions = function (parentChangeset, parentBranchRevision, childChangeset, childBranchRevision, commonContainerView) {
            var parentPosition = context.getPositionInContainerCoordinates(parentChangeset, parentBranchRevision, commonContainerView, commonContainerView.viewport);
            var childPosition = context.getPositionInContainerCoordinates(childChangeset, childBranchRevision, commonContainerView, commonContainerView.viewport);

            //if one of these is SparseOnBranch, we draw it directly inline with the Live one.
            if (isNaN(parentPosition.x)) {
                parentPosition.x = childPosition.x;
            } else if (isNaN(childPosition.x)) {
                childPosition.x = parentPosition.x;
            } else if (isNaN(parentPosition.y)) {
                parentPosition.y = childPosition.y;
            } else if (isNaN(childPosition.y)) {
                childPosition.y = parentPosition.y;
            } else {
                var heightDifference = parentPosition.y - childPosition.y;
                var widthDifference = parentPosition.x - childPosition.x;
                var parentAbove = heightDifference > 0;
                var parentBefore = widthDifference < 0;
                var notInLine = heightDifference !== 0 && widthDifference !== 0;
                var spacing = context.getSpacing();
                var absHeightDiff = Math.abs(heightDifference);
                var absWidthDiff = Math.abs(widthDifference);
                var notDiagonallyOneAway = !(absHeightDiff === absWidthDiff && absHeightDiff === spacing);
                var requiresCenter = !(absHeightDiff === spacing || absWidthDiff === spacing);

                var curveX = pathNubX / 2;
                var curveY = pathNubY / 2;

                if (notInLine) {//} && notDiagonallyOneAway) {
                    var parentCurveEnd = new Point(
                        parentPosition.x + (parentBefore ? pathNubX : -pathNubX ),
                        parentPosition.y + (parentAbove ? -pathNubY : pathNubY )
                    );
                    var childCurveEnd = new Point(
                        childPosition.x + (parentBefore ? -pathNubX : pathNubX),
                        childPosition.y + (parentAbove ? pathNubY : -pathNubY )
                    );

                    var center;
                    if (requiresCenter) {
                        center = new Point(parentCurveEnd.x, childCurveEnd.y);
                    }

                    // HACK: sharp edges - we should make edges curvy in IE8 also eventually.
                    if ($.browser.msie && $.browser.version < 9) {
                        if (requiresCenter) {
                            return [parentPosition, parentCurveEnd, center, childCurveEnd, childPosition];
                        } else {
                            return [parentPosition, parentCurveEnd, childCurveEnd, childPosition];
                        }
                    }

                    var parentCurveMiddle;
                    var childCurveMiddle;

                    if (absWidthDiff === spacing && notDiagonallyOneAway) {
                        childCurveMiddle = new Point(childCurveEnd.x, childPosition.y);
                    } else {
                        childCurveMiddle = new Point(childPosition.x, childCurveEnd.y);
                    }

                    if (absHeightDiff === spacing) {
                        parentCurveMiddle = new Point(parentPosition.x, parentCurveEnd.y);
                    } else {
                        parentCurveMiddle = new Point(parentCurveEnd.x, parentPosition.y);
                    }

                    if (requiresCenter) {
                        var centerCurveStart = new Point(
                            center.x,
                            center.y + (parentAbove ? curveY : -curveY)
                        );
                        var centerCurveEnd = new Point(
                            center.x + (parentBefore ? curveX : -curveX),
                            center.y
                        );
                        return [parentPosition, parentCurveMiddle, parentCurveEnd,
                            centerCurveStart, center, centerCurveEnd,
                            childCurveEnd, childCurveMiddle, childPosition];
                    } else {
                        return [parentPosition, parentCurveMiddle, parentCurveEnd,
                            childCurveEnd, childCurveMiddle, childPosition];
                    }
                }
            }

            return [parentPosition, childPosition];
        };
        /*eslint-enable*/

        return context;
    }

    /**
     * A Generic view that allows subViews and a model. Mixin or extend this class. never create a new view
     */
    var View = (function () {
        var View = function () {
            this.model = undefined;
            this.parentView = undefined;
            this.subViews = undefined;

            this._isSyncedWithModel = false;
            this._isVisible = false;
            this._requiresHighlight = true;

            throw new Error("View is abstract and should not be instantiated.");
        };

        /**
         * Updates only the currently existing DOM elements to reflect any changes we've made.
         */
        View.prototype.reDraw = function () {
            if (this.isVisible()) {
                this.drawSelf();
                this.subViews && Array.each(this.subViews, function (subView) {
                    subView.reDraw();
                });
            }
        };

        /**
         * Similar to draw but doesn't draw the subviews as well
         */
        View.prototype.drawSelf = abstractFunction;

        /**
         * Updates positioning calculations ready for this view to be redrawn and its subViews. Usually called before draw()
         */
        View.prototype.sync = function (forceFullRefresh) {
            if (forceFullRefresh || this.requiresSync()) {
                this.syncSelf();
                this.setSyncRequired(false);
            }

            this.subViews && Array.each(this.subViews, function (subView) {
                subView.sync(forceFullRefresh);
            });

            return this;
        };

        /**
         * Similar to sync but ignores subViews
         */
        View.prototype.syncSelf = abstractFunction;

        /**
         * Updates visibility to reflect changes in viewport. Usually called before draw()
         */
        View.prototype.updateVisibility = function () {
            this.updateVisibilitySelf(); // recursion is handled in setVisible

            this.subViews && Array.each(this.subViews, function (subView) {
                subView.updateVisibility();
            });

            return this;
        };

        /**
         * Similar to updateVisibility but ignores subViews
         */
        View.prototype.updateVisibilitySelf = function () {
            this.setVisible(this.parentView.isVisible());
            return this;
        };

        /**
         * Permanently remove this view.  The view is unusuable after this method is called.
         */
        View.prototype.destroy = function () {
            this.eraseSelf();

            this.subViews && Array.each(this.subViews, function (view) {
                view.destroy();
            });
        };

        /**
         * If visible, updates appearance data to reflect changes in highlighting and redraws.
         */
        View.prototype.reHighlight = function () {
            if (this.isVisible()) {
                this.highlightSelf();
                this.drawSelf();
            } else {
                this.setHighlightRequired(true);
            }
            this.subViews && Array.each(this.subViews, function (view) {
                view.reHighlight();
            });

            return this;
        };

        /**
         * Similar to highlight but ignores subViews
         */
        View.prototype.highlightSelf = abstractFunction;

        /**
         * Similar to erase() but ignores subViews
         */
        View.prototype.eraseSelf = abstractFunction;

        /**
         * Removes any subViews from the subView array whos model has been disposed
         */
        View.prototype.removeDisposedSubViews = function () {
            var views = this.subViews;
            var newViews = [];

            views && Array.each(views, function (view) {
                if (!view.model.disposed) {
                    newViews.push(view);
                }
            });
            this.subViews = newViews;
        };

        /**
         * Recursively find a subView associated
         * @param model
         */
        View.prototype.findViewByModel = function (model) {
            var foundView = null;
            this.forEachView(function (view) {
                if (view.model === model) {
                    foundView = view;
                    return View.END_ALL;
                }
            });
            return foundView;
        };


        View.END_PATH = {};
        View.END_ALL = false;
        /**
         * Call a function on each view in this view tree.
         * If the function returns false, stop traversing.
         * @param func function that will be called with each view as the first parameter. If
         * your function returns View.END_PATH on a node, then forEachView will not traverse that node's subtree.
         * If your function returns View.END_ALL at any point, forEachView will stop traversal of the entire View tree.
         */
        View.prototype.forEachView = function (func) {
            var result = func(this);
            if (result === View.END_ALL) {
                return false;
            }
            if (result !== View.END_PATH && this.subViews) {
                var lastResult;
                Array.each(this.subViews, function (subView) {
                    return lastResult = subView.forEachView(func);
                });
                return lastResult;
            }
        };

        View.prototype.isVisible = function () {
            return this._isVisible;
        };

        View.prototype.setVisible = function (isVisible) {
            if (!this._isVisible && isVisible) {
                this._isVisible = true;
                if (this.requiresHighlight()) {
                    this.highlightSelf();
                }
                this.drawSelf();
            } else if (this._isVisible && !isVisible) {
                this._isVisible = false;
                this.eraseSelf(!this.parentView || !this.parentView.element);
            }
        };


        View.prototype.requiresHighlight = function () {
            return this._requiresHighlight;
        };

        View.prototype.setHighlightRequired = function (requiresHighlight) {
            this._requiresHighlight = requiresHighlight;
        };


        View.prototype.requiresSync = function () {
            return !this._isSyncedWithModel;
        };

        View.prototype.setSyncRequired = function (requiresSync) {
            this._isSyncedWithModel = !requiresSync;
        };

        return View;
    })();

    var TopLevelView = (function () {
        var TopLevelView = function () {
            this.model = undefined;
            this.context = undefined;
            throw new Error("TopLevelView is abstract and should not be instantiated");
        };
        $.extend(TopLevelView.prototype, View.prototype, EventProducer);


        TopLevelView.prototype.setHighlightContext = function (highlightContext) {
            if (this.parentView) {
                this.parentView.setHighlightContext(highlightContext);
                return;
            }

            this.context.setHighlightContext(highlightContext);
            this.reHighlight();
        };

        TopLevelView.prototype.setFocus = function (highlightable) {
            if (this.parentView) {
                this.parentView.setFocus(highlightable);
                return;
            }

            var highlightContext = this.context.highlightContext;
            if (highlightContext.focusedObject === highlightable) {
                // Blur
                highlightable = null;
            }
            highlightContext.onFocusChanged(highlightable);
            this.reHighlight();
        };

        TopLevelView.prototype.translate = abstractFunction;

        TopLevelView.prototype.translateTo = abstractFunction;

        TopLevelView.prototype.resize = abstractFunction;

        TopLevelView.prototype.getScrolledToTimePosition = abstractFunction;

        return TopLevelView;
    })();

    var SliceView = (function () {
        var SliceView = function () {
            this.model = undefined;
            this.context = undefined;
            throw new Error("SliceView is abstract and should not be instantiated.");
        };
        $.extend(SliceView.prototype, View.prototype);

        SliceView.prototype.syncSelf = function () {
            var model = this.model;
            var length = model.getLength();
            var breadth = model.getBreadth();
            var context = this.context;
            var spacing = context.getSpacing();

            var boundingBox = this.boundingBox = context.createRectangle(
                spacing * context.getSliceTimewise(model.timePosition, length),
                0,
                spacing * length,
                spacing * breadth
            );
            this.boundingBoxInGraphSpace = new Rectangle(
                model.timePosition,
                0,
                length,
                breadth
            );

            this.translation = new Vector(boundingBox.x, boundingBox.y);

            this.attributes = {
                scale: 1,
                width: boundingBox.width,
                height: boundingBox.height
            };

            this.removeDisposedSubViews();
            this.addNewSubViews();

            return this;
        };

        return SliceView;
    })();

    var GraphEdgeView = (function () {
        var GraphEdgeView = function (edge, parentView) {
            this.model = edge;
            this.parentView = parentView;
            this.target = parentView.target;
            this.context = parentView.context;

            this._requiresHighlight = true;
            this.highlightable = undefined;
        };
        $.extend(GraphEdgeView.prototype, View.prototype);
        GraphEdgeView.create = function (edge, parentView) {
            var view = new GraphEdgeView(edge, parentView);

            view.syncSelf();
            view.updateVisibilitySelf();
            view.resetHighlights();

            return view;
        };

        GraphEdgeView.prototype.sync = function (forceFullRefresh) {
            if (forceFullRefresh || this.requiresSync()) {
                var wasSparse = this.model.hasSparseEndpoint();
                this.syncSelf();
                //if it was sparse,
                if (wasSparse) {
                    this.updateVisibilitySelf();
                    this.drawSelf();
                }
                !this.model.hasSparseEndpoint() && this.setSyncRequired(false); // only set it clean once it's not sparse.
            }

            return this;
        };

        GraphEdgeView.prototype.syncSelf = function () {
            var edge = this.model;
            this.points = this.context.calculateEdgePositions(edge.parentChangeset, edge.parentBranchRevision, edge.childChangeset, edge.childBranchRevision, this.parentView);
        };

        GraphEdgeView.prototype.setVisible = function (isVisible) {
            // HACK : Sparse edges are always visible, and redraw every time.
            if (this.parentView instanceof GraphView || (!this._isVisible && isVisible)) {
                this._isVisible = true;
                if (this.requiresHighlight()) {
                    this.highlightSelf();
                }
                this.drawSelf();
            } else if (this._isVisible && !isVisible) {
                this._isVisible = false;
                this.eraseSelf();
            }
        };

        GraphEdgeView.prototype.updateVisibilitySelf = function () {
            // TODO : do edge visibility calculations in slices for better perf.
            // set sparse edges visible always. Yes, it's a hack.
            this.setVisible(this.parentView instanceof GraphView || this.parentView.isVisible());
        };

        GraphEdgeView.prototype.resetHighlights = function () {
            var context = this.highlightContext = this.context.highlightContext;
            var highlightable = this.highlightable = context.getEdge(this.model);
        };

        GraphEdgeView.prototype.highlightSelf = function () {
            if (this.highlightContext !== this.context.highlightContext) {
                this.resetHighlights();
            }

            var highlightable = this.highlightable;

            this.highlightContext.highlightEdge(highlightable);

            var color = highlightable.color();
            var parent;

            if (color === COLOR_BY_BRANCH) {
                parent = this.model.parentBranchRevision;
                if (parent) {
                    color = parent.primaryBranch ? parent.primaryBranch.color : fallbackBranchColor;
                }
            }

            this.attributes = {
                strokeWidth: 1,
                strokeColor: color
            };

            if (highlightable.hasBorder()) {
                var borderColor = highlightable.borderColor();
                if (borderColor === COLOR_BY_BRANCH) {
                    parent = this.model.parentBranchRevision;
                    if (parent) {
                        color = parent.primaryBranch ? parent.primaryBranch.color : fallbackBranchColor;
                    }
                }

                this.glowAttributes = {
                    strokeWidth: highlightable.borderWidth() * 2 + 1, //width of 2 borders + actual path width
                    strokeColor: borderColor
                };
            } else {
                this.glowAttributes = null;
            }

            return this;
        };

        GraphEdgeView.prototype.eraseSelf = function () {
            if (this.element) {
                this.target.erase(this.element);
                this.element = null;
            }
            return this;
        };

        GraphEdgeView.prototype.drawSelf = function () {
            var target = this.target;

            if (this.element) {
                this.target.movePath(this.element, this.points);
                target.setPathAttributes(this.element, this.attributes);
            } else {
                this.attributes.moveToBack = true;
                this.element = target.drawPath(this.parentView.element, this.points, this.attributes);
                $(this.element.element).data('model', this.model);
            }

            /*
             There's no good way to put an aura/glow around a path. "stroke" is already being used to draw the edge,
             and "fill" will do weird things (try to fill the shape defined by the path). So we need a second element
             to represent the outer color.
             */
            if (this.glowAttributes) {
                if (this.glowElement) {
                    target.movePath(this.glowElement, this.points);
                    target.setPathAttributes(this.glowElement, this.glowAttributes);
                } else {
                    this.glowAttributes.insertBeforeElement = this.element;
                    this.glowElement = target.drawPath(this.parentView.element, this.points, this.glowAttributes);
                }
            } else if (this.glowElement) {
                target.erase(this.glowElement);
                this.glowElement = null;
            }
        };

        return GraphEdgeView;
    })();

    var GraphBranchRevisionView = (function () {
        var GraphBranchRevisionView = function (changeset, groupView) {
            this.model = changeset;
            this.parentView = groupView;
            this.target = groupView.target;
            this.context = groupView.context;

            this._requiresHighlight = true;
            this.highlightable = undefined;
        };
        $.extend(GraphBranchRevisionView.prototype, View.prototype);

        GraphBranchRevisionView.create = function (changeset, groupView) {
            var view = new GraphBranchRevisionView(changeset, groupView);

            view.syncSelf();
            view.updateVisibilitySelf();
            view.resetHighlights();

            return view;
        };

        GraphBranchRevisionView.prototype.highlightSelf = function () {
            if (this.highlightContext !== this.context.highlightContext) {
                this.resetHighlights();
            }

            var highlightable = this.highlightable;

            this.highlightContext.highlightChangeset(highlightable);

            var strokeWidth;
            var strokeColor;
            var fillColor;

            fillColor = highlightable.color();

            if (fillColor === COLOR_BY_BRANCH) {
                fillColor = this.model.primaryBranch ? this.model.primaryBranch.color : fallbackBranchColor;
            }

            if (highlightable.hasBorder()) {
                strokeWidth = highlightable.borderWidth();
                strokeColor = highlightable.borderColor();

                if (strokeColor === COLOR_BY_BRANCH) {
                    strokeColor = this.model.primaryBranch ? this.model.primaryBranch.color : fallbackBranchColor;
                }
            } else {
                strokeWidth = 1;
                strokeColor = fillColor;
            }

            this.attributes = {
                // stroke is centered on the edge, so we need to shift it outward by half its width.
                // this ensures the visible "fill" area is a constant size, unaffected by strokeWidth.
                radius: this.context.getChangesetRadius() + (strokeWidth / 2),
                strokeWidth: strokeWidth,
                strokeColor: strokeColor,
                fillColor: fillColor
            };

            return this;
        };

        GraphBranchRevisionView.prototype.syncSelf = function () {
            var branchRevision = this.model;
            var changeset = branchRevision.changeset;
            var context = this.context;
            var spacing = context.getSpacing();

            this.position = context.createPoint(
                context.getChangesetTimewise(changeset.timePosition, changeset.slice.length) * spacing,
                branchRevision.branchPosition * spacing
            );

            return this;
        };

        GraphBranchRevisionView.prototype.drawSelf = function () {
            var target = this.target;

            if (this.element) {
                target.movePoint(this.element, this.position);
                target.setPointAttributes(this.element, this.attributes);
            } else {
                this.element = target.drawPoint(this.parentView.element, this.position, this.attributes);
                $(this.element.element).data('model', this.model.changeset);
            }
            return this;
        };

        GraphBranchRevisionView.prototype.eraseSelf = function () {
            if (this.element) {
                this.target.erase(this.element);
                this.element = null;
            }
            return this;
        };

        GraphBranchRevisionView.prototype.resetHighlights = function () {
            var context = this.highlightContext = this.context.highlightContext;
            this.highlightable = context.getChangeset(this.model.changeset);
        };

        return GraphBranchRevisionView;
    })();

    var GraphGroupView = (function () {
        var GraphGroupView = function (group, sliceView) {
            this.model = group;
            this.parentView = sliceView;
            this.target = sliceView.target;
            this.context = sliceView.context;
        };
        $.extend(GraphGroupView.prototype, View.prototype);
        GraphGroupView.create = function (group, sliceView) {
            var view = new GraphGroupView(group, sliceView);

            view.syncSelf();
            view.updateVisibilitySelf();

            return view;
        };

        GraphGroupView.prototype.highlightSelf = function () {
            // Do Nothing
            return this;
        };

        GraphGroupView.prototype.syncSelf = function () {
            var branch = this.model.branch;
            var slice = this.parentView.model;
            var context = this.context;
            var spacing = context.getSpacing();

            var boundingBox = this.boundingBox = this.context.createRectangle(
                0,
                spacing * branch.branchPosition,
                spacing * slice.length,
                spacing * branch.breadth
            );

            this.translation = new Vector(boundingBox.x, boundingBox.y);

            this.attributes = {
                scale: 1,
                width: boundingBox.width,
                height: boundingBox.height
            };

            this.removeDisposedSubViews();
            this.addNewSubViews();
        };

        GraphGroupView.prototype.addNewSubViews = function () {
            var model = this.model;
            var subViews = this.subViews;

            addNewViews(this, subViews, model.branchRevisions, GraphBranchRevisionView.create);
            addNewViews(this, subViews, model.internalEdges, GraphEdgeView.create);
        };

        GraphGroupView.prototype.drawSelf = function () {
            if (this.element) {
                this.target.moveGroup(this.element, this.translation);
                this.target.setGroupAttributes(this.element, this.attributes);
            } else {
                this.element = this.target.drawGroup(this.parentView.element, this.translation, this.attributes);
            }

            return this;
        };

        GraphGroupView.prototype.eraseSelf = function () {
            if (this.element) {
                this.target.erase(this.element);
                this.element = null;
            }
            return this;
        };

        return GraphGroupView;
    })();

    var GraphSliceView = (function () {
        var GraphSliceView = function (slice, graphView) {
            this.model = slice;
            this.parentView = graphView;
            this.target = graphView.target;
            this.context = graphView.context;
        };
        $.extend(GraphSliceView.prototype, SliceView.prototype);
        GraphSliceView.create = function (slice, graphView) {
            var view = new GraphSliceView(slice, graphView);

            view.syncSelf();
            view.updateVisibilitySelf();

            return view;
        };

        GraphSliceView.prototype.addNewSubViews = function () {
            var model = this.model;
            var subViews = this.subViews;

            addNewViews(this, subViews, model.changesetGroups, GraphGroupView.create);
            addNewViews(this, subViews, model.interGroupEdges, GraphEdgeView.create);
        };

        GraphSliceView.prototype.updateVisibilitySelf = function () {
            var isVisible = this.parentView.isVisible() &&
                this.boundingBoxInGraphSpace.intersectsRectangle(this.parentView.visibleAreaInGraphSpace);
            this.setVisible(isVisible);

            return this;
        };

        GraphSliceView.prototype.highlightSelf = function () {
            // Do Nothing
            return this;
        };

        GraphSliceView.prototype.drawSelf = function () {
            if (this.element) {
                this.target.moveGroup(this.element, this.translation);
                this.target.setGroupAttributes(this.element, this.attributes);
            } else {
                this.element = this.target.drawGroup(this.parentView.element, this.translation, this.attributes);
            }

            return this;
        };

        GraphSliceView.prototype.eraseSelf = function () {
            if (this.element) {
                this.target.erase(this.element);
                this.element = null;
            }
            return this;
        };

        return GraphSliceView;
    })();

    var GraphBranchView = (function () {
        var GraphBranchView = function (branch, graphView) {
            this.model = branch;
            this.parentView = graphView;
            this.context = graphView.context;

            this.element = $("<div></div>").attr("id", "lane-" + FECRU.zEscape(branch.name));
            $(graphView.target.getContainer()).siblings(".branch-lanes").append(this.element);
        };

        $.extend(GraphBranchView.prototype, View.prototype);

        GraphBranchView.create = function (branch, graphView) {
            var view = new GraphBranchView(branch, graphView);

            view.syncSelf();
            view.updateVisibilitySelf();

            return view;
        };

        GraphBranchView.prototype.highlightSelf = function () {
            // Do Nothing
            return this;
        };

        GraphBranchView.prototype.syncSelf = function () {
            var branch = this.model;
            var $element = this.element;
            var context = this.context;
            var spacing = context.getSpacing();

            var boundingBox = this.boundingBox = context.createRectangle(
                0,
                spacing * branch.branchPosition + (BRANCH_SPACING / 2),
                spacing * context.createVector($element.width(), $element.height()).x,
                spacing * Math.max(branch.breadth, 1) - BRANCH_SPACING
            );

            this.css = {
                left: boundingBox.x,
                top: boundingBox.y,
                width: boundingBox.width,
                height: "100%",
                position: "absolute",
                backgroundColor: "#f0f0f0"
            };

            return this;
        };

        GraphBranchView.prototype.drawSelf = function () {
            if (this.element) {
                this.element.css(this.css)
            }

            return this;
        };

        GraphBranchView.prototype.eraseSelf = function () {
            if (this.element) {
                this.element.remove();
            }
            return this;
        };

        return GraphBranchView;
    })();

    var GraphView = (function () {
        var GraphView = function (dag, target, initialViewport, initialTranslation, context) {
            this.model = dag;
            this.target = target;
            this.context = context;
            this.viewport = initialViewport;

            this.visibilityBuffer = context.getSpacing() * 20;

            this.translation = initialTranslation;
            this.visibleAreaInGraphSpace = undefined;
            this.element = undefined;

            var self = this;
            var $targetElement = $(target.getElement());

            var getChangesetFromPoint = function (point) {
                return $(point).data('model');
            };

            var getEdgeFromPath = function (path) {
                return $(path).data('model');
            };

            $targetElement.delegate('[class~="path"]', 'click', function () {
                var edge = getEdgeFromPath(this);
                var highlightable = edge ? self.context.highlightContext.getEdge(edge) : null;

                highlightable && self.setFocus(highlightable);
            }).delegate('[class~="point"]', 'click', function () {
                var cs = getChangesetFromPoint(this);
                var highlightable = cs ? self.context.highlightContext.getChangeset(cs) : null;

                highlightable && self.setFocus(highlightable);
            });

            if ($.browser.msie && ($.browser.version < 9)) {
                //ie8 handling of hover paths and nodes
                $targetElement.delegate('[class~="path"]', 'mouseenter', function () {
                    $(this).data("strokeweight", this.strokeweight);
                    this.strokeweight = EDGE_HOVER_WIDTH;
                }).delegate('[class~="path"]', 'mouseleave', function () {
                    if (Number(this.strokeweight) === EDGE_HOVER_WIDTH) {
                        this.strokeweight = $(this).data("strokeweight");
                    }
                }).delegate('[class~="point"]', 'mouseenter', function () {
                    var elemStyle = this.style;
                    var radius = parseFloat(elemStyle.width, 10) / 2;
                    var position = {x: parseFloat(elemStyle.left), y: parseFloat(elemStyle.top), radius: radius};
                    var newRadius = radius + (NODE_HOVER_RADIUS);
                    var newDiameter = newRadius * 2;

                    $(this).data('vml-position', position);
                    elemStyle.width = (newDiameter) + "px";
                    elemStyle.height = (newDiameter) + "px";
                    //shift the centre of the circle a bit due to increased radiu, so it stays in the original place
                    elemStyle.left = Math.floor(position.x - NODE_HOVER_RADIUS) + "px";
                    elemStyle.top = Math.floor(position.y - NODE_HOVER_RADIUS) + "px";
                }).delegate('[class~="point"]', 'mouseleave', function () {
                    var elemStyle = this.style;
                    var oldPosition = $(this).data('vml-position');
                    var oldRadius = oldPosition.radius;
                    var oldDiameter = oldRadius * 2;

                    if (Number(parseFloat(elemStyle.width, 10)) === (oldDiameter + (NODE_HOVER_RADIUS * 2))) {
                        elemStyle.width = (oldDiameter) + "px";
                        elemStyle.height = (oldDiameter) + "px";
                        elemStyle.left = Math.floor(oldPosition.x) + "px";
                        elemStyle.top = Math.floor(oldPosition.y) + "px";
                    }
                });
            } else {
                // jquery 1.8.3 does not delegate events fired by svg properly
                $targetElement.delegate('[class~="path"]', 'mouseenter', function () {
                    $(this).data('stroke-width', this.getAttribute('stroke-width'));
                    this.setAttribute('stroke-width', EDGE_HOVER_WIDTH);
                }).delegate('[class~="path"]', 'mouseleave', function () {
                    if (Number(this.getAttribute('stroke-width')) === EDGE_HOVER_WIDTH) {
                        this.setAttribute('stroke-width', $(this).data('stroke-width'));
                    }
                }).delegate('[class~="point"]', 'mouseenter', function () {
                    var oldRadius = parseFloat(this.getAttribute('r'), 10);
                    $(this).data('radius', oldRadius);
                    this.setAttribute('r', oldRadius + NODE_HOVER_RADIUS);
                }).delegate('[class~="point"]', 'mouseleave', function () {
                    var oldRadius = $(this).data('radius');
                    if (parseFloat(this.getAttribute('r'), 10) === (oldRadius + NODE_HOVER_RADIUS)) {
                        this.setAttribute('r', oldRadius);
                    }
                });
            }

            this.highlightPopup = HighlightPopup.create('[class~="point"]', "changeset-popup", {
                getHighlightContext: function () {
                    return self.context.highlightContext;
                },
                getDisplayCsId: function (csid) {
                    return FECRU.getDisplayCsId(self.model.settings.repositoryType, csid);
                },
                getHighlightableChangeset: getChangesetFromPoint
            });

            this.syncSelf();
            this.updateVisibilitySelf();
        };
        $.extend(GraphView.prototype, TopLevelView.prototype);

        GraphView.prototype.updateVisibilitySelf = function () {
            var context = this.context;
            var spacing = context.getSpacing();
            var buffer = this.visibilityBuffer;
            var length = this.model.getLength();
            var viewportInDrawSpace = this.viewport;
            var translationInDrawSpace = this.translation;
            var timePosition = context.getGraphTimePosition(this.model);

            //the visible portion of the bounding box is based on the viewport and how translated the dag is.
            var trueVisibleAreaInGraphSpace = context.createGraphSpaceRectangle(
                (viewportInDrawSpace.x - translationInDrawSpace.x) / spacing,
                (viewportInDrawSpace.y - translationInDrawSpace.y) / spacing,
                (viewportInDrawSpace.width) / spacing,
                (viewportInDrawSpace.height) / spacing
            );
            var graphSpaceBuffer = buffer / spacing;
            var visibleAreaInGraphSpaceWithBuffer = this.visibleAreaInGraphSpace = new Rectangle(
                trueVisibleAreaInGraphSpace.x - graphSpaceBuffer,
                trueVisibleAreaInGraphSpace.y - graphSpaceBuffer,
                trueVisibleAreaInGraphSpace.width + (2 * graphSpaceBuffer),
                trueVisibleAreaInGraphSpace.height + (2 * graphSpaceBuffer)
            );

            var visibleAreaPastEdge = visibleAreaInGraphSpaceWithBuffer.x; // x is really timePosition
            var visibleAreaFutureEdge = visibleAreaInGraphSpaceWithBuffer.x + visibleAreaInGraphSpaceWithBuffer.width; // width is really length

            // BUG: additional space needed. Not sure why.  Thought it was the branch header, but that seems unrelated.
            var buggyExtraSpace = 1.5;
            this.scrolledToTimePosition =
                trueVisibleAreaInGraphSpace.x + trueVisibleAreaInGraphSpace.width - buggyExtraSpace;

            this.setVisible(true);

            this.trigger(EdgeProximityEvent, {
                proximityToPastHorizon: visibleAreaPastEdge - timePosition,
                proximityToFutureHorizon: timePosition + length - visibleAreaFutureEdge
            });
        };

        GraphView.prototype.highlightSelf = function () {
            // Do Nothing
            return this;
        };

        GraphView.prototype.syncSelf = function () {

            var model = this.model;
            var context = this.context;
            var spacing = context.getSpacing();
            var length = model.getLength();
            var breadth = model.getBreadth();
            var timePosition = context.getGraphTimePosition(model);

            this.boundingBox = context.createRectangle(
                context.getGraphTimewise(timePosition, length) * spacing,
                0,
                length * spacing,
                breadth * spacing
            );

            this.attributes = {
                scale: 1,
                width: this.boundingBox.width,
                height: this.boundingBox.height
            };

            this.removeDisposedSubViews();
            this.addNewSubViews();
            return this;
        };

        GraphView.prototype.addNewSubViews = function () {
            var model = this.model;
            var subViews = this.subViews;
            /*DEBUG*/
            var nonSparseOnBranch = $.grep(model.interSliceEdges, function (edge) {
                return !edge.parentChangeset.isSparseOnBranch() && !edge.childChangeset.isSparseOnBranch();
            });
            addNewViews(this, subViews, model.dagSlices, GraphSliceView.create);
            addNewViews(this, subViews, nonSparseOnBranch, GraphEdgeView.create);
            addNewViews(this, subViews, model.branchManager.getBranchSet().getAllBranches(), GraphBranchView.create);
        };

        GraphView.prototype.drawSelf = function () {

            if (this.element) {
                this.target.moveGroup(this.element, this.translation);
                this.target.setGroupAttributes(this.element, this.attributes);
            } else {
                this.element = this.target.drawGroup(null, this.translation, this.attributes);
            }
            return this;
        };

        GraphView.prototype.eraseSelf = function () {
            if (this.element) {
                this.target.erase(this.element);
                this.element = null;
            }
            if (this.highlightPopup) {
                this.highlightPopup.erase();
                this.highlightPopup = null;
            }
            return this;
        };

        GraphView.prototype.translate = function (vector) {
            this.context.translateBy(this.translation, vector);
            this.updateVisibility();
            //need to update with translation changes
            this.drawSelf();
            return this;
        };

        GraphView.prototype.translateTo = function (changesetOrPxFromTop) {
            var context = this.context;

            if (typeof changesetOrPxFromTop === 'number') {
                var pixels = changesetOrPxFromTop;
                var firstChangeset = this.model.getLastChangeset();

                if (!firstChangeset.branchRevisions || !firstChangeset.branchRevisions.length) {
                    return;
                }

                var branchRevision = firstChangeset.branchRevisions[0];
                var firstCsPosition = context.getPositionInContainerCoordinates(firstChangeset, branchRevision, this, this.viewport);
                var fullTranslation = new Vector(firstCsPosition.x + pixels, firstCsPosition.y + pixels);

                context.translateTo(this.translation, fullTranslation);
            } else {
                var changeset = changesetOrPxFromTop;
                if (!changeset.branchRevisions || !changeset.branchRevisions.length) {
                    return;
                }

                var branchRevision = changeset.branchRevisions[0]; // It doesn't matter which one we scroll to. We only use the timewise position anyway
                var csPosition = context.getPositionInContainerCoordinates(changeset, branchRevision, this, this.viewport);

                context.translateTo(this.translation, csPosition);
            }

            this.updateVisibility();
            //need to update with translation changes
            this.drawSelf();
        };

        GraphView.prototype.resize = function () {
            this.target.resize();
            var oldViewport = this.viewport;
            var oldCenter = new Point(
                oldViewport.x + oldViewport.width / 2,
                oldViewport.y + oldViewport.height / 2
            );
            var newViewport = new Rectangle(
                0,
                0,
                this.target.getWidth(),
                this.target.getHeight()
            );
            var newCenter = new Point(
                oldViewport.x + oldViewport.width / 2,
                oldViewport.y + oldViewport.height / 2
            );
            this.viewport = newViewport;
            this.translate(new Vector(newCenter.x - oldCenter.x, newCenter.y - oldCenter.y));
        };

        GraphView.prototype.getScrolledToTimePosition = function () {
            return this.scrolledToTimePosition;
        };

        return GraphView;
    })();

    var MetadataChangesetView = (function () {
        var MetadataChangesetView = function (changeset, sliceView) {
            this.model = changeset;
            this.parentView = sliceView;
            this.context = sliceView.context;

            this.element = undefined;

            this._requiresHighlight = true;
            this.highlightable = undefined;
            this.highlightLozenges = [];

            this.attributes = {
                backgroundColor: '',
                color: ''
            };
            this.isFocused = false;

            this.width = sliceView.width;
        };
        $.extend(MetadataChangesetView.prototype, View.prototype);

        MetadataChangesetView.create = function (changeset, sliceView) {
            var view = new MetadataChangesetView(changeset, sliceView);

            view.syncSelf();
            view.updateVisibilitySelf();
            view.resetHighlights();

            return view;
        };

        MetadataChangesetView.prototype.highlightSelf = function () {
            if (this.highlightContext !== this.context.highlightContext) {
                this.resetHighlights();
            }

            var highlightable = this.highlightable;

            this.attributes = createBackgroundColor(highlightable.backgroundColor());
            this.attributes.color = highlightable.textColor();
            this.attributes.metadataCssClass = highlightable.metadataCssClass();
            this.isFocused = highlightable === this.highlightContext.focusedObject;

            return this;
        };

        MetadataChangesetView.prototype.drawSelf = function () {

            var getAvatarUrl = function (author, Configuration) {
                var url = (author.avatarUrl ? author.avatarUrl : Configuration.defaultAvatarUrl);
                return FECRU.getAvatarUrlAtSize(url, "16");
            };

            var isFocused = this.isFocused;
            var attributes = this.attributes;
            var cs = this.model;
            var classes = "metadata-changeset" +
                (isFocused ? " focused" : "") +
                (attributes.metadataCssClass ? " " + attributes.metadataCssClass : "") +
                (cs.isMetadataLoaded() ? " loaded" : "");

            if (cs.isMetadataLoaded()) {
                if (!this.element) {
                    var author = cs.author;
                    var html = templateFactory.load('metadata-changeset').fill({
                        'id': cs.id,
                        //a zEncoded id suitable for use in the 'id' attr of an element
                        'IdEscaped': FECRU.zEscape(cs.id),
                        'date': cs.date || '',
                        'comment:html': cs.comment || '',
                        'userDisplayName': author.displayName || '',
                        'userUrl': author.url || '',
                        'userAvatarUrl': getAvatarUrl(author, Configuration),
                        'classes': classes,
                        'width': this.width
                    });
                    var $metadataElement = this.element = $(html.toString());

                    $metadataElement.css('top', this.position.y);
                    attributes.backgroundColor && $metadataElement.css('backgroundColor', attributes.backgroundColor);
                    attributes.filter && $metadataElement.css('filter', attributes.filter);
                    attributes.color && $metadataElement.css('color', attributes.color);

                    this.parentView.element.append($metadataElement);
                    $metadataElement.data('model', this.model);
                } else {
                    this.element.css('backgroundColor', attributes.backgroundColor)
                        .css('filter', attributes.filter)
                        .css('color', attributes.color)
                        .attr('class', classes)
                        .children()
                        .first()
                        .css('paddingLeft', this.width);
                }
                this.drawLozenges();
            }
        };

        MetadataChangesetView.prototype.eraseSelf = function (parentErased) {
            if (this.element) {
                //.remove is heavy.  Don't do it if the parent element has already been removed.
                !parentErased && this.element.remove();
                this.element = null;
            }
            if (this.highlightLozenges) {
                Array.each(this.highlightLozenges, function (lozenge) {
                    lozenge.erase();
                });
                this.highlightLozenges = [];
            }
        };

        MetadataChangesetView.prototype.reloadMetadata = function () {
            this.eraseSelf();
            this.drawSelf();
            return this;
        };

        MetadataChangesetView.prototype.resize = function () {
            if (this.width !== this.parentView.width) {
                this.width = this.parentView.width;
                if (this.element) {
                    this.element.children().first().css('paddingLeft', this.width);
                }
            }
        };

        MetadataChangesetView.prototype.syncSelf = function () {
            var changeset = this.model;
            var context = this.context;
            var spacing = context.getSpacing();

            this.position = context.createPoint(
                context.getChangesetTimewise(changeset.timePosition, changeset.slice.length) * spacing,
                0
            );

            return this;
        };

        MetadataChangesetView.prototype.resetHighlights = function () {
            var context = this.highlightContext = this.context.highlightContext;
            this.highlightable = context.getChangeset(this.model);
        };

        MetadataChangesetView.prototype.drawLozenges = function () {
            if (!this.element) {
                return;
            }
            var cs = this.model;
            var $lozengeContainer = this.element.find(".lozenge-annotations").empty();
            var tags = cs.tags;

            var context = this.highlightContext;
            var highlightable = context.getChangeset(cs);
            var lozenge = context.getLozenges(highlightable);

            if (lozenge) {
                this.highlightLozenges.push(lozenge);
                Array.each(lozenge.getLozenges(), function (l) {
                    $lozengeContainer.append(l);
                })
            }

            if (tags && tags.length > 0) {
                var tagLozenge = HighlightLozenge.create(tags, {
                    collapsedLozengeText: function (itemCount) {
                        return itemCount + " tags";
                    },
                    getItemText: function (item) {
                        return FECRU.truncateText(item, {
                            maxLength: 50
                        })
                    },
                    getItemTitle: function (item) {
                        return item.length > 53 ? item : null; // only show title if truncated
                    },
                    sort: true,
                    lozengeClass: "tag-lozenge"
                });
                this.highlightLozenges.push(tagLozenge);
                Array.each(tagLozenge.getLozenges(), function (l) {
                    $lozengeContainer.append(l);
                })
            }
        };

        return MetadataChangesetView;
    })();

    var MetadataSliceView = (function () {
        var MetadataSliceView = function (slice, graphView) {
            this.model = slice;
            this.parentView = graphView;
            this.context = graphView.context;
            this.width = graphView.width;
        };
        $.extend(MetadataSliceView.prototype, SliceView.prototype);

        MetadataSliceView.create = function (slice, graphView) {
            var view = new MetadataSliceView(slice, graphView);

            view.syncSelf();
            view.updateVisibilitySelf();

            return view;
        };

        MetadataSliceView.prototype.addNewSubViews = function () {
            var model = this.model;
            var subViews = this.subViews;

            addNewViews(this, subViews, model.getAllChangesets(), MetadataChangesetView.create);

            return this;
        };

        MetadataSliceView.prototype.updateVisibilitySelf = function () {
            var isVisible = this.boundingBoxInGraphSpace.intersectsRectangle(this.parentView.visibleAreaInGraphSpace);
            this.setVisible(isVisible);
        };

        MetadataSliceView.prototype.highlightSelf = function () {
            // Do Nothing
            return this;
        };

        MetadataSliceView.prototype.drawSelf = function () {
            var boundingBox = this.boundingBox;
            var parentView = this.parentView;

            if (!this.element) {
                var $metadataElement = this.element = $('<div class="metadata-slice"/>');
                $metadataElement.height(boundingBox.height)
                    .css('top', boundingBox.y - parentView.boundingBox.y);
                parentView.element.append($metadataElement);
            } else {
                this.element.height(boundingBox.height)
                    .css('top', boundingBox.y - parentView.boundingBox.y);
            }

            return this;
        };

        MetadataSliceView.prototype.eraseSelf = function () {
            if (this.element) {
                this.element.remove();
                this.element = null;
            }
            return this;
        };

        MetadataSliceView.prototype.reloadMetadata = function () {
            if (this.subViews) {
                if (this.isVisible()) {
                    Array.each(this.subViews, function (subView) {
                        subView.highlightable.init(subView.model);
                        subView.reloadMetadata();
                    });
                } else {
                    Array.each(this.subViews, function (subView) {
                        subView.highlightable.init(subView.model);
                    });
                }
            }
            return this;
        };

        MetadataSliceView.prototype.resize = function () {
            this.width = this.parentView.width;
            this.subViews && Array.each(this.subViews, function (subView) {
                subView.resize();
            });
        };

        return MetadataSliceView;
    })();

    var MetadataView = (function () {
        var MetadataView = function (dag, $container, initialViewport, initialTranslation, context) {
            this.model = dag;
            this.$container = $container;
            this.context = context;
            this.viewport = initialViewport;

            this._events = {};

            this._isVisible = false;
            this.visibilityBuffer = context.getSpacing() * 20;

            this.translation = initialTranslation;
            this.visibleAreaInGraphSpace = undefined;
            this.element = undefined;

            this.pendingRescrollPx = 0;

            bindEvents(this);

            this.syncSelf();
            this.updateVisibilitySelf();
        };
        $.extend(MetadataView.prototype, TopLevelView.prototype);

        function bindEvents(self) {
            var $container = self.$container;
            var events = self._events;

            $container.scroll(events.onScroll = function () {
                self.scrollTop = $container.scrollTop();
            });

            $container.delegate('.metadata-changeset', 'click', events.onChangesetClick = function (e) {
                if ($(e.target).closest(".navigation", this).length > 0) {
                    return;
                }
                var cs = $(this).data('model');
                var highlightable = cs ? self.context.highlightContext.getChangeset(cs) : null;

                highlightable && self.setFocus(highlightable);
            });

            $container.delegate('.csid-copy input', 'focus mouseup', events.onCsidCopyFocus = function (e) {
                this.select();
            });

        }

        function unbindEvents(self) {
            var $container = self.$container;
            var events = self._events;

            events.onScroll && $container.unbind('scroll', events.onScroll);
            events.onChangesetClick && $('.metadata-changeset', $container).undelegate('click', events.onChangesetClick);
            events.onCsidCopyFocus && $('.csid-copy input', $container).undelegate('focus mouseup', events.onCsidCopyFocus);
        }

        MetadataView.prototype.addNewSubViews = function () {
            var model = this.model;
            var subViews = this.subViews;

            addNewViews(this, subViews, model.dagSlices, MetadataSliceView.create);
        };

        MetadataView.prototype.updateVisibilitySelf = function () {
            var context = this.context;
            var spacing = context.getSpacing();
            var buffer = this.visibilityBuffer;
            var length = this.model.getLength();
            var viewportInDrawSpace = this.viewport;
            var translationInDrawSpace = this.translation;
            var timePosition = context.getGraphTimePosition(this.model);

            //the visible portion of the bounding box is based on the viewport and how translated the dag is.
            var trueVisibleAreaInGraphSpace = context.createGraphSpaceRectangle(
                (viewportInDrawSpace.x - translationInDrawSpace.x) / spacing,
                (viewportInDrawSpace.y - translationInDrawSpace.y) / spacing,
                (viewportInDrawSpace.width) / spacing,
                (viewportInDrawSpace.height) / spacing
            );
            var graphSpaceBuffer = buffer / spacing;
            var visibleAreaInGraphSpaceWithBuffer = this.visibleAreaInGraphSpace = new Rectangle(
                trueVisibleAreaInGraphSpace.x - graphSpaceBuffer,
                trueVisibleAreaInGraphSpace.y - graphSpaceBuffer,
                trueVisibleAreaInGraphSpace.width + (2 * graphSpaceBuffer),
                trueVisibleAreaInGraphSpace.height + (2 * graphSpaceBuffer)
            );

            var visibleAreaPastEdge = visibleAreaInGraphSpaceWithBuffer.x; // x is really timePosition
            var visibleAreaFutureEdge = visibleAreaInGraphSpaceWithBuffer.x + visibleAreaInGraphSpaceWithBuffer.width; // width is really length

            this.setVisible(true);

            this.trigger(EdgeProximityEvent, {
                proximityToPastHorizon: visibleAreaPastEdge - timePosition,
                proximityToFutureHorizon: timePosition + length - visibleAreaFutureEdge
            });
        };

        MetadataView.prototype.syncSelf = function () {

            var model = this.model;
            var context = this.context;
            var spacing = context.getSpacing();
            var length = model.getLength();
            var breadth = model.getBreadth();
            var timePosition = context.getGraphTimePosition(model);

            this.boundingBox = context.createRectangle(
                context.getGraphTimewise(timePosition, length) * spacing,
                0,
                length * spacing,
                breadth * spacing
            );

            var topChangeset = model.getLastChangeset();
            if (this.lastTopChangeset && (topChangeset !== this.lastTopChangeset)) {
                var oldTopPosition = context.getPositionInContainerCoordinates(this.lastTopChangeset, this.lastTopChangeset.branchRevisions[0], this, this.viewport);
                var newTopPosition = context.getPositionInContainerCoordinates(topChangeset, topChangeset.branchRevisions[0], this, this.viewport);
                var newSpaceAtTop = oldTopPosition.y - newTopPosition.y;

                this.pendingRescrollPx += newSpaceAtTop;
            }
            this.lastTopChangeset = topChangeset;

            this.attributes = {
                scale: 1,
                width: this.boundingBox.width,
                height: this.boundingBox.height
            };

            this.removeDisposedSubViews();
            this.addNewSubViews();
            return this;
        };

        MetadataView.prototype.highlightSelf = function () {
            // Do Nothing
            return this;
        };

        MetadataView.prototype.drawSelf = function () {
            var boundingBox = this.boundingBox;
            var height = boundingBox.height;
            var scrollTop = this.scrollTop || 0;

            if (this.element) {
                this.element.height(height);
                this.$container.scrollTop(scrollTop);
            } else {
                this.element = $('<div class="metadata-dag"/>').height(height).appendTo(this.$container);
                this.$container.scrollTop(scrollTop);
            }
        };

        // Should only be called by destroy.
        MetadataView.prototype.eraseSelf = function () {
            unbindEvents(this);
            if (this.element) {
                this.element.remove();
                this.element = null;
            }
        };

        MetadataView.prototype.reloadMetadata = function () {
            this.subViews && Array.each(this.subViews, function (subView) {
                subView.reloadMetadata();
            });
            return this;
        };

        MetadataView.prototype.resize = function () {
            this.width = this.parentView.width;
            this.viewport = new Rectangle(0, 0, this.width, this.$container.height());
            this.subViews && Array.each(this.subViews, function (subView) {
                subView.resize();
            });
            this.updateVisibility();
            this.drawSelf();
        };

        MetadataView.prototype.translateTo = function (changesetOrPxFromTop) {
            if (typeof changesetOrPxFromTop === 'number') {
                this.scrollTop = changesetOrPxFromTop;
            } else {
                var context = this.context;
                var branchRevision = changesetOrPxFromTop.branchRevisions[0];
                var csPosition = context.getPositionInContainerCoordinates(changesetOrPxFromTop, branchRevision, this, this.viewport);

                this.scrollTop = csPosition.y - this.boundingBox.y;
            }
            this.drawSelf();
        };

        MetadataView.prototype.hasPendingRescroll = function () {
            return !!this.pendingRescrollPx;
        };

        MetadataView.prototype.updateScroll = function () {
            if (this.pendingRescrollPx) {
                var pendingRescrollPx = this.pendingRescrollPx;
                this.scrollTop += pendingRescrollPx;
                this.pendingRescrollPx = 0;
                this.reDraw();

                return pendingRescrollPx;
            }
            return 0;
        };

        return MetadataView;
    })();

    var DualView = (function () {
        var DualView = function (dag, target, $container, initialViewport, initialTranslation, context) {
            this.model = dag;
            this.$graphContainer = $(target.getContainer());
            this.$metadataContainer = $container;
            this.context = context;

            this.graphicalView = new GraphView(dag, target, initialViewport, initialTranslation, context);
            this.graphicalView.parentView = this;
            this.metadataView = new MetadataView(dag, $container, initialViewport, initialTranslation, context);
            this.metadataView.parentView = this;
            this.subViews = [this.graphicalView, this.metadataView];

            this._isVisible = true;
            var self = this;
            var triggerSelf = function (event) {
                self.trigger(EdgeProximityEvent, event);
            };
            Array.each(this.subViews, function (view) {
                view.bind(EdgeProximityEvent, triggerSelf);
            });
        };
        $.extend(DualView.prototype, TopLevelView.prototype);

        DualView.create = function (graph, target, $metadataContainer, horizontalOrVertical, timeDirectionPositive) {
            var horizontal = horizontalOrVertical === 'horizontal';
            var flippedTimewise = !timeDirectionPositive;
            var initialViewport = new Rectangle(
                0,
                0,
                target.getWidth(),
                target.getHeight()
            );
            var context = createContext(horizontal, flippedTimewise, CHANGESET_SPACING, CHANGESET_RADIUS);
            var initialOffset = flippedTimewise ?
                (horizontal ? initialViewport.width : initialViewport.height) :
                0;
            var initialTranslation = context.createVector(
                (CHANGESET_SPACING / 2) + initialOffset,
                CHANGESET_SPACING / 2
            );

            return new DualView(graph, target, $metadataContainer, initialViewport, initialTranslation, context);
        };

        /**
         * If sync is called on DualView, we know it comes from the user (not internal to our view tree).
         * That means we're goin got need a full lifecycle call.
         */
        var ViewSync = View.prototype.sync;
        DualView.prototype.sync = function (forceFullRefresh) {
            // The root level must resync to grab new slices
            this.setSyncRequired(true);

            ViewSync.call(this, forceFullRefresh);

            var graphView = this.graphicalView;
            var metadataView = this.metadataView;

            if (forceFullRefresh) {
                graphView.reDraw();
            }

            /** new metadata might have loaded on top.  This means
             *  the metadata view must increase its height, increase all existing slice tops to make room,
             *  and scroll down so the changesets that were on the screen before are still on the screen.
             *  This scroll should NOT trigger a matched scroll by the graph view, because that would move
             *  them out of sync.
             *  Therefore we must also scroll the graphView UP in prep for the downward scroll event from the metadataView
             */
            if (metadataView.hasPendingRescroll()) {
                //includes a redraw.
                var realignmentPx = metadataView.updateScroll();
                realignmentPx && graphView.translate(new Vector(0, realignmentPx));
            } else if (forceFullRefresh) {
                metadataView.reDraw();
            }

            return this;
        };

        DualView.prototype.syncSelf = function () {
            // Do Nothing
            return this;
        };

        DualView.prototype.updateVisibilitySelf = function () {
            // Do Nothing
            return this;
        };

        DualView.prototype.highlightSelf = function () {
            // Do Nothing
            return this;
        };

        DualView.prototype.drawSelf = function () {
            // Do Nothing
            return this;
        };

        DualView.prototype.eraseSelf = function () {
            // Do Nothing
            return this;
        };

        DualView.prototype.setSyncRequired = function (requiresSync) {
            if (requiresSync) {
                this._isSyncedWithModel = false;
                this.graphicalView.setSyncRequired(requiresSync);
                this.metadataView.setSyncRequired(requiresSync);
            } else {
                this._isSyncedWithModel = !requiresSync;
            }
        };

        // if this is IE8, don't redraw metadata until the end of the scroll.
        var isIE8 = $.browser.msie && $.browser.version < 9;
        var pendingTranslation = new Vector(0, 0);
        var graphTranslate = function (self) {
            self.graphicalView.translate(pendingTranslation);
            pendingTranslation = new Vector(0, 0);
        };
        var throttledGraphTranslate = $.browser.mozilla ?
            graphTranslate : // Firefox sucks at ordering the event execution if you setTimeout.
                             // Performance takes a hit when we don't throttle, but perceived performance
                             // is improved because you don't get huge lags in graph translation.
            $.throttle(isIE8 ? 500 : 50, graphTranslate);
        var metadataRedraw = function (self) {
            self.metadataView.updateVisibility();
            self.metadataView.drawSelf();
        };
        var throttledMetadataRedraw = isIE8 ?
            $.debounce(500, metadataRedraw) :
            $.throttle(250, metadataRedraw);
        DualView.prototype.translate = function (vector) {
            pendingTranslation = pendingTranslation.add(vector);
            throttledGraphTranslate(this);
            throttledMetadataRedraw(this);

            return this;
        };

        DualView.prototype.translateTo = function (changeset) {
            //HACK: this scroll will cause events to fire that will
            // make the ProjectVisualiser tell us to translate the graphView.
            // Very convoluted - we should move scroll handling into the view.
            this.metadataView.translateTo(changeset);

            return this;
        };

        DualView.prototype.resize = function () {
            var $dualContainer = this.$graphContainer.parent();
            var availableWidth = $dualContainer.width();
            var graphSpaceDesired = this.graphicalView.boundingBox.width + METADATA_PADDING_LEFT;
            var graphSpace = Math.min(graphSpaceDesired, availableWidth - 200);

            this.width = graphSpace;
            this.$graphContainer.width(graphSpace);
            this.graphicalView.resize();
            this.metadataView.resize();
        };

        DualView.prototype.reloadMetadata = function () {
            this.metadataView.reloadMetadata();
            this.reHighlight();
        };

        DualView.prototype.getScrolledToTimePosition = function () {
            return this.graphicalView.getScrolledToTimePosition();
        };

        DualView.prototype.init = function (scrollToChangesetId, focusChangesetId) {
            var model = this.model;
            var graphView = this.graphicalView;

            // align graph with metadata view - put first row at the top
            graphView.translateTo(0);

            var scrollToChangeset;
            if (scrollToChangesetId && (scrollToChangeset = model.findChangesetByIdOrTag(scrollToChangesetId))) {
                this.translateTo(scrollToChangeset);
            } else if (scrollToChangesetId) {
                var msg = $("<div></div>")
                    .append("<p></p>").text("Could not find a changeset with the id or tag: " + scrollToChangesetId);
                FECRU.AJAX.appendNotificationMessage(msg, true);
                FECRU.AJAX.showNotificationBox("Changeset not found", 'message');
            }

            var focusChangeset = focusChangesetId === scrollToChangesetId ?
                scrollToChangeset :
            focusChangesetId && model.findChangesetByIdOrTag(focusChangesetId);
            if (focusChangeset) {
                this.setFocus(this.context.highlightContext.getChangeset(focusChangeset));
            }
        };

        return DualView;
    })();

    return {
        createGraphView: DualView.create,
        View: View,
        SliceView: SliceView,
        GraphView: GraphView,
        GraphSliceView: GraphSliceView,
        GraphGroupView: GraphGroupView,
        GraphBranchRevisionView: GraphBranchRevisionView,
        GraphEdgeView: GraphEdgeView,
        MetadataView: MetadataView,
        MetadataSliceView: MetadataSliceView,
        MetadataChangesetView: MetadataChangesetView,
        DualView: DualView,
        EdgeProximityEvent: EdgeProximityEvent
    };
})(AJS.$, AJS.template);
/*[{!renderer_js_6tm252r!}]*/