FE.VIS.POSITIONING = (function ($) {

    /* Imports */
    var EventProducer = FECRU.MIXINS.EventProducer;

    var AbstractPositioner = function (positionChangesetsInBranch) {
        this.positionChangesetsInBranch = positionChangesetsInBranch;
    };
    $.extend(AbstractPositioner.prototype, EventProducer);

    AbstractPositioner.prototype.positionGroups = function (slice) {
        var self = this;
        var nextY = 0;

        Array.each(slice.changesetGroups, function (group) {
            var maxBranchPosition = 0;
            var l = group.branchRevisions.length;

            for (var i = 0; i < l; i++) {
                var branchRevision = group.branchRevisions[i];
                maxBranchPosition = Math.max(maxBranchPosition, branchRevision.branchPosition);
            }

            var groupBreadth = l ?
                Math.max(maxBranchPosition + 1, 2) /* min-width: 2*changeset-spacings */ :
                /*Empty breadth: */ 1;
            var branch = group.branch;

            var oldBranchBreadth = branch.breadth;
            var branchBreadth;

            if (oldBranchBreadth < groupBreadth) {
                branchBreadth = groupBreadth;
                self.trigger(AbstractPositioner.BranchExpandedEvent, {
                    branchName: branch.name,
                    breadth: branchBreadth
                });
            } else {
                branchBreadth = oldBranchBreadth;
            }

            branch.branchPosition = nextY;
            branch.breadth = branchBreadth;
            nextY += branchBreadth; // the group below shifts down by this branch's breadth
        });
    };
    AbstractPositioner.BranchExpandedEvent = "BranchExpanded";

    AbstractPositioner.prototype.positionSlice = function (slice, neighbor, atEnd) {
        if (neighbor) {
            slice.timePosition = atEnd ?
            neighbor.timePosition + neighbor.length :
            neighbor.timePosition - slice.length;
        } else {
            slice.timePosition = 0;
        }
    };
    var timeSpanOnGraph = function (brev1, brev2) {
        return Math.abs(brev1.changeset.absTimePosition - brev2.changeset.absTimePosition);
    };

    var findNearestSpace = function (stack, startIndex) {
        var length = stack.length;
        for (var upI = startIndex, downI = startIndex;
             upI > 0 || downI < length;
             --upI, ++downI) {
            if (upI >= 0) {
                if (stack[upI] === undefined) {
                    return upI;
                }
            }
            if (downI < length) {
                if (stack[downI] === undefined) {
                    return downI;
                }
            }
        }

        return length;
    };

    var methodGroups = {
        ancestorBased: {
            prevNodes: "parents",
            nextNodes: "children",
            prevNodesOnBranch: "parentsOnBranch",
            nextNodesOnBranch: "childrenOnBranch",
            isSparsePrevNode: "isSparseAncestor"
        },
        descendantBased: {
            prevNodes: "children",
            nextNodes: "parents",
            prevNodesOnBranch: "childrenOnBranch",
            nextNodesOnBranch: "parentsOnBranch",
            isSparsePrevNode: "isSparseDescendant"
        }
    };

    AbstractPositioner.prototype.positionChangesetsInTime = function (changesets, slice, ancestorBased) {
        //set changeset positions
        if (ancestorBased) {
            Array.each(changesets, function (cs, i) {
                cs.timePosition = i;
                cs.absTimePosition = slice.timePosition + i;
            });
        } else {
            var lastIndex = changesets.length - 1;
            Array.each(changesets, function (cs, i) {
                var x = cs.timePosition = (lastIndex - i);
                cs.absTimePosition = slice.timePosition + x;
            });
        }
    };

    AbstractPositioner.prototype.positionNewSlices = function (newSlices, newSparseAncestors, newSparseDescendants, newSparseOnBranch, positioningData, neighbor, ancestorBased) {
        var positioner = this;
        var newLiveChangesets = [];

        Array.each(newSlices, function (slice) {
            positioner.positionSlice(slice, neighbor, ancestorBased);
            neighbor = slice;

            var sliceChangesets = slice.getAllChangesets(!ancestorBased);

            positioner.positionChangesetsInTime(sliceChangesets, slice, ancestorBased);

            $.merge(newLiveChangesets, sliceChangesets);
        });
        // position sparse changesets in time as well.
        Array.each(newSparseAncestors, function (cs) {
            cs.timePosition = cs.absTimePosition = -Infinity;
        });
        Array.each(newSparseDescendants, function (cs) {
            cs.timePosition = cs.absTimePosition = Infinity;
        });
        Array.each(newSparseOnBranch, function (cs) {
            cs.timePosition = NaN;
        });

        var firstSparse = (ancestorBased ? newSparseAncestors : newSparseDescendants) || [];
        var lastSparse = (ancestorBased ? newSparseDescendants : newSparseAncestors) || [];

        if (firstSparse.length) {
            // First lot of data to be loaded
            positioner.positionChangesetsInBranch(firstSparse, positioningData, ancestorBased, true);
        }
        positioner.positionChangesetsInBranch(newLiveChangesets || [], positioningData, ancestorBased);
        positioner.positionChangesetsInBranch(lastSparse, positioningData, ancestorBased);

        Array.each(newSparseOnBranch, function (cs) {
            cs.branchPosition = Infinity;
        });

        Array.each(newSlices, function (slice) {
            positioner.positionGroups(slice);
        });
    };

    AbstractPositioner.prototype.getChangesetAtTimePosition = function (graph, timePosition) {
        var sliceWithChangeset;
        var changesetAtTimePosition = null;

        Array.each(graph.dagSlices, function (slice) {
            if (slice.timePosition <= timePosition &&
                slice.getLength() + slice.timePosition >= timePosition) {
                sliceWithChangeset = slice;
                return false;
            }
        });

        if (sliceWithChangeset) {
            var offset = timePosition - sliceWithChangeset.timePosition;
            Array.each(sliceWithChangeset.getAllChangesets(), function (cs) {
                if (cs.timePosition === offset) {
                    changesetAtTimePosition = cs;
                    return false;
                }
            });
        }

        return changesetAtTimePosition;
    };

    // Our positioning algorithm

    function positionBranchRevisionInBranch(branchRevision, positionArray, methods) {
        var nextNodesFuncKey = methods.nextNodesOnBranch;
        var prevNodesFuncKey = methods.prevNodesOnBranch;
        var isSparsePrevNodeFuncKey = methods.isSparsePrevNode;
        var notChangeset = function (node) {
            return node !== branchRevision;
        };

        var prevNodesOnBranch = branchRevision[prevNodesFuncKey]();
        var prevNodeCount = prevNodesOnBranch.length;
        var stackHeight = positionArray.length;
        var newBreadthPos;

        if (prevNodeCount === 0 || branchRevision[isSparsePrevNodeFuncKey]()) { // head, sparse or branch point

            // Find the first empty spot in the array. If there isn't one, use the height
            var availablePosition = $.inArray(undefined, positionArray);
            newBreadthPos = availablePosition !== -1 ? availablePosition : stackHeight;

        } else if (prevNodeCount === 1) {

            var prevNode = prevNodesOnBranch[0];
            var prevNodeNextNodesOnBranch = $.grep(prevNode[nextNodesFuncKey](), notChangeset); // Sibling / spouse
            var prevNodeNextNodeCount = prevNodeNextNodesOnBranch.length;

            if (prevNodeNextNodeCount === 0) {
                // Its a normal commit
                newBreadthPos = prevNode.branchPosition;
            } else {
                var timeSpan = timeSpanOnGraph(branchRevision, prevNode);
                if (Array.any(prevNodeNextNodesOnBranch,
                        function (node) {
                            return timeSpanOnGraph(node, prevNode) > timeSpan;
                        })) {
                    newBreadthPos = findNearestSpace(positionArray, prevNode.branchPosition);
                } else {
                    newBreadthPos = prevNode.branchPosition;
                }
            }

        } else {

            var furthestPrevNode = null;
            var longestTimeSpan = 0;

            Array.each(prevNodesOnBranch, function (prevNode) {
                var timeSpan = timeSpanOnGraph(branchRevision, prevNode);
                if (timeSpan > longestTimeSpan) {
                    longestTimeSpan = timeSpan;
                    furthestPrevNode = prevNode;
                    if (timeSpan === Infinity) {
                        return false; // child is sparse
                    }
                }
            });

            var shouldMatchFurthestPrevNode = true;
            Array.each(prevNodesOnBranch, function (prevNode) {
                var timeSpan = timeSpanOnGraph(branchRevision, prevNode);
                var nextNodeIsFurther = function (nextNode) {
                    return nextNode !== branchRevision && timeSpanOnGraph(nextNode, prevNode) > timeSpan;
                };

                if (!Array.any(prevNode[nextNodesFuncKey](), nextNodeIsFurther)) {
                    positionArray[prevNode.branchPosition] = undefined;
                } else if (prevNode === furthestPrevNode) {
                    /**
                     * This scenario happened
                     * <---Positioner is sweeper this way
                     * o--------------o
                     *           @---/
                     *            \o
                     *
                     * The furthest previous node is not the further previous node of one of its next node
                     */
                    shouldMatchFurthestPrevNode = false;
                }
            });
            newBreadthPos = shouldMatchFurthestPrevNode ?
                furthestPrevNode.branchPosition :
                findNearestSpace(positionArray, furthestPrevNode.branchPosition);
        }
        // Ensure the changeset is only positioned once
        branchRevision.branchPosition = !isNaN(branchRevision.branchPosition) ? branchRevision.branchPosition : newBreadthPos;

        var nextNodes = branchRevision[nextNodesFuncKey]();
        var shouldMakeAvailable = !branchRevision.isSparse()
            && (nextNodes.length === 0 || Array.all(nextNodes, function (brev) {
                return !isNaN(brev.branchPosition);
            }));

        //preserve this position in the positioning array if the changeset isn't sparse and has "next" nodes.
        positionArray[newBreadthPos] = shouldMakeAvailable ? undefined : branchRevision.id;
    }

    function positionChangesetInBranch(changeset, positionArraysByBranch, methods) {
        Array.each(changeset.branchRevisions, function (brev) {
            var branchName = brev.primaryBranch.name;
            if (!positionArraysByBranch[branchName]) {
                positionArraysByBranch[branchName] = [];
            }
            positionBranchRevisionInBranch(brev, positionArraysByBranch[branchName], methods);
        });
    }

    function positionChangesetsInBranch(changesets, positioningData, ancestorBased, initialPositioning) {
        var positionArraysByBranch;
        if (ancestorBased) {
            positionArraysByBranch = positioningData.atEnd = positioningData.atEnd || {};
        } else {
            positionArraysByBranch = positioningData.atStart = positioningData.atStart || {};
        }

        var methods = ancestorBased ? methodGroups.ancestorBased : methodGroups.descendantBased;
        Array.each(changesets, function (cs) {
            positionChangesetInBranch(cs, positionArraysByBranch, methods);
        });
        if (initialPositioning) {
            if (ancestorBased) {
                positioningData.atStart = $.extend(true, {}, positionArraysByBranch);
            } else {
                positioningData.atEnd = $.extend(true, {}, positionArraysByBranch);
            }
        }
    }

    return {
        AbstractPositioner: AbstractPositioner,
        defaultPositioner: new AbstractPositioner(positionChangesetsInBranch)
    };
})(AJS.$);
/*[{!positioner_js_0sha52q!}]*/