AJS.$.extend(FE.VIS.HIGHLIGHTS, (function ($) {

    /* Imports */
    var Map = FECRU.DATA_STRUCTURES.Map;
    var CacheMap = FECRU.DATA_STRUCTURES.CacheMap;
    var EventProducer = FECRU.MIXINS.EventProducer;
    var colorForBranch = FE.VIS.BranchSet.colorForBranch;
    var fallbackBranchColor = FE.VIS.fallbackBranchColor;
    var Iterators = FE.VIS.HIGHLIGHTS.ITERATORS;

    var COLOR_BY_BRANCH = "COLOR_BY_BRANCH";

    function getBranchColorFromChangeset(cs) {
        var primaryBranches = cs._model.primaryBranches;
        if (primaryBranches && primaryBranches.length === 1) {
            var primaryBranch = primaryBranches[0];
            return primaryBranch.name !== 'All Branches' ? //all check is a hack
                primaryBranch.color :
            cs.branches && cs.branches.length && colorForBranch(cs.branches[0]);
        } else if (primaryBranches && primaryBranches.length > 1) {
            /* complex SVN commit */
            return COLOR_BY_BRANCH;
        } else {
            return fallbackBranchColor;
            /* sparseOnBranch */
        }
    }

    function getBranchColorFromEdge(edge) {
        return getBranchColorFromChangeset(edge.parent());
    }

    function HighlightHelper(highlightContext) {
        var context = highlightContext;
        var loadingState = false;

        return {
            setLoading: function (isLoading) {
                if (isLoading !== loadingState) {
                    if (isLoading) {
                        context.trigger(HighlightContext.LoadingStarted);
                    } else {
                        context.trigger(HighlightContext.LoadingFinished);
                        this.forceRedraw();
                    }
                    loadingState = isLoading;
                }
            },
            forceRedraw: function () {
                context.trigger(HighlightContext.RedrawRequested);
            }
        }
    }

    /* Public */

    /**
     * Encapsulates a highlighter's interface with the views.
     * It can create "HighlightableChangesets" and "HighlightableEdges" that
     * will be sent to the highlighter for highlighting.
     * @param highlighter the highlighter this context is for.
     * @param changesetDefaults standard changeset styling attributes to pass to plugins.
     * @param edgeDefaults standard edge styling attributes to pass to plugins.
     */
    var HighlightContext = function (highlighter, changesetDefaults, edgeDefaults) {
        this.highlighter = highlighter = highlighter || {};

        var helperEnabled = !!highlighter.setHelper;
        var edgeHighlightsEnabled = this.edgeHighlightsEnabled = !!highlighter.highlightEdge;
        var changesetHighlightsEnabled = this.changesetHighlightsEnabled = !!highlighter.highlightChangeset;
        var focusEnabled = this.focusEnabled = highlighter.onFocusChanged;
        var relatedChangesetsEnabled = this.relatedChangesetsEnabled = !!highlighter.getAllRelatedChangesets;
        var moreInfoEnabled = this.moreInfoEnabled = !!highlighter.getMoreInfo;
        var onActivationEnabled = this.onActivationEnabled = !!highlighter.onActivation;
        var onDeactivationEnabled = this.onDeactivationEnabled = !!highlighter.onDeactivation;
        var onDataLoadedEnabled = this.onDataLoadedEnabled = !!highlighter.onDataLoaded;
        var onDataUnloadedEnabled = this.onDataUnloadedEnabled = !!highlighter.onDataUnloaded;
        var lozengesEnabled = this.getLozenges = !!highlighter.getLozenges;
        var changesetAnnotationEnabled = this.getChangesetAnnotation = !!highlighter.getChangesetAnnotation;

        changesetDefaults = $.extend(true, {
            Selected: {
                color: '#FFF',
                hasBorder: true,
                borderColor: '#000',
                borderWidth: 4,
                textColor: null,
                backgroundColor: '#3D82F8',
                metadataCssClass: null
            },
            Emphasized: {
                color: '#EA7400',
                hasBorder: true,
                borderColor: '#003366',
                borderWidth: 2,
                textColor: null,
                backgroundColor: '#FFCC00',
                metadataCssClass: null
            },
            Normal: {
                color: '#326CA6',
                hasBorder: false,
                borderColor: null,
                borderWidth: null,
                textColor: null,
                backgroundColor: null,
                metadataCssClass: null
            },
            BranchSelected: {
                color: '#FFF',
                hasBorder: true,
                borderColor: '#000',
                borderWidth: 4,
                textColor: null,
                backgroundColor: '#3D82F8',
                metadataCssClass: null
            },
            BranchEmphasized: {
                color: getBranchColorFromChangeset,
                hasBorder: true,
                borderColor: getBranchColorFromChangeset,
                borderWidth: 2,
                textColor: null,
                backgroundColor: null,
                metadataCssClass: null
            },
            BranchNormal: {
                color: getBranchColorFromChangeset,
                hasBorder: false,
                borderColor: null,
                borderWidth: null,
                textColor: null,
                backgroundColor: null,
                metadataCssClass: null
            },
            Dimmed: {
                color: '#CCC',
                hasBorder: false,
                borderColor: null,
                borderWidth: null,
                textColor: '#999',
                backgroundColor: null,
                metadataCssClass: "metadata-dimmed"
            }
        }, changesetDefaults || {});

        edgeDefaults = $.extend(true, {
            Selected: {
                color: '#000',
                hasBorder: true,
                borderColor: '#000',
                borderWidth: 1
            },
            Emphasized: {
                color: '#CCC',
                hasBorder: 1,
                borderColor: "#CCC",
                borderWidth: 1
            },
            Normal: {
                color: '#CCC',
                hasBorder: false,
                borderColor: null,
                borderWidth: null
            },
            BranchSelected: {
                color: getBranchColorFromEdge,
                hasBorder: true,
                borderColor: getBranchColorFromEdge,
                borderWidth: 1
            },
            BranchEmphasized: {
                color: getBranchColorFromEdge,
                hasBorder: false,
                borderColor: null,
                borderWidth: null
            },
            BranchNormal: {
                color: '#CCC',
                hasBorder: false,
                borderColor: null,
                borderWidth: null
            },
            Dimmed: {
                color: '#CCC',
                hasBorder: false,
                borderColor: null,
                borderWidth: null
            }
        }, edgeDefaults || {});

        this.focusedObject = null;

        //Putting our ancestry cache on the context allows me to isolate it from other plugins (so one can't ruin another).
        this.ancestorCache = new CacheMap(20);
        this.descendantCache = new CacheMap(20);


        var changesetsById = new Map();
        var edgesByLinkedIds = new Map();
        var self = this;

        var getChangeset = function (model) {
            return changesetsById.get(model.id);
        };
        var createChangeset = function (model) {
            var newCs = new HighlightableChangeset(model, self);
            changesetsById.set(model.id, newCs);
            newCs.attr(changesetDefaults.Normal);
            return newCs;
        };
        var getOrCreateChangeset = function (model) {
            return getChangeset(model) || createChangeset(model);
        };
        var getEdge = function (model) {
            return edgesByLinkedIds.get(model.childChangeset.id + "-" + model.parentChangeset.id);
        };
        var createEdge = function (model) {
            var parent = getOrCreateChangeset(model.parentChangeset);
            var child = getOrCreateChangeset(model.childChangeset);
            var edge = new HighlightableEdge(model, parent, child, self);

            edgesByLinkedIds.set(child.csid + "-" + parent.csid, edge);
            edge.attr(edgeDefaults.Normal);
            return edge;
        };
        var getOrCreateEdge = function (model) {
            return getEdge(model) || createEdge(model);
        };

        this.getChangeset = getOrCreateChangeset;
        this.getEdge = getOrCreateEdge;

        function logError(e) {
            FECRU.AJAX.appendErrorResponse(e, false);
            FECRU.AJAX.showNotificationBox(e.toString());
        }

        function getTryFunction(isEnabled, enabledFunc, disabledFunc, getDefaultValue) {
            if (isEnabled) {
                return function () {
                    try {
                        return enabledFunc.apply(this, arguments);
                    } catch (e) {
                        logError(e);
                        return getDefaultValue && getDefaultValue();
                    }
                };
            } else {
                return !disabledFunc ? $.noop : function () {
                    try {
                        return disabledFunc.apply(this, arguments);
                    } catch (e) {
                        logError(e);
                        return getDefaultValue && getDefaultValue();
                    }
                };
            }
        }

        this.highlightEdge = getTryFunction(edgeHighlightsEnabled, function (edge) {
            if (edge === this.focusedObject) {
                this.focusedObject.attr(edgeDefaults.Selected);
                if (!focusEnabled) {
                    return;
                }
            }
            highlighter.highlightEdge(edge, edgeDefaults);
        }/*, noop - do not set selected style on edges if highlightEdges isn't implemented */);

        this.highlightChangeset = getTryFunction(changesetHighlightsEnabled, function (cs) {
            if (cs === this.focusedObject) {
                this.focusedObject.attr(changesetDefaults.Selected);
                if (!focusEnabled) {
                    return;
                }
            }
            highlighter.highlightChangeset(cs, changesetDefaults);
        }, function (cs) {
            /* if they don't implement highlightChangeset, there's probably something terribly wrong going on.
             Regardless, we still want to make sure the selected attributes are applied.
             * */
            if (cs === this.focusedObject) {
                this.focusedObject.attr(changesetDefaults.Selected);
            } else {
                this.focusedObject.attr(changesetDefaults.Normal);
            }
        });

        this.onFocusChanged = getTryFunction(focusEnabled, function (focusedObject) {
            if (focusedObject !== this.focusedObject) {
                this.focusedObject = focusedObject;
                highlighter.onFocusChanged(focusedObject);
            }
        }, function (focusedObject) {
            this.focusedObject = focusedObject;
        });

        this.getAllRelatedChangesets = getTryFunction(relatedChangesetsEnabled, function (focusedChangeset) {
            return highlighter.getAllRelatedChangesets(focusedChangeset);
        }, null, function () {
            return [];
        });

        this.getMoreInfo = getTryFunction(moreInfoEnabled, function (focusedChangeset, $moreInfoContainer) {
            return highlighter.getMoreInfo(focusedChangeset, $moreInfoContainer);
        });

        this.onActivation = getTryFunction(onActivationEnabled, function (allChangesets, focusedObject) {
            highlighter.onActivation(allChangesets);
            this.onFocusChanged(focusedObject);
        }, function (unused, focusedObject) {
            this.onFocusChanged(focusedObject);
        });

        this.onDeactivation = getTryFunction(onDeactivationEnabled, function () {
            highlighter.onDeactivation();
        });

        this.onDataLoaded = getTryFunction(onDataLoadedEnabled, function (loadedChangesets) {
            highlighter.onDataLoaded(loadedChangesets);
        });

        this.onDataUnloaded = getTryFunction(onDataUnloadedEnabled, function (unloadedChangesets) {
            highlighter.onDataUnloaded(unloadedChangesets);
        });

        this.getChangesetAnnotation = getTryFunction(changesetAnnotationEnabled, function (changeset) {
            return highlighter.getChangesetAnnotation(changeset, FE.VIS.HIGHLIGHT_POPUP.createAnnotationList);
        });

        this.getLozenges = getTryFunction(lozengesEnabled, function (changeset) {
            return highlighter.getLozenges(changeset, FE.VIS.HIGHLIGHT_LOZENGE.create);
        });

        if (helperEnabled) {
            try {
                this.highlighter.setHelper(new HighlightHelper(this));
            } catch (e) {
                logError(e);
            }
        }
    };

    $.extend(HighlightContext.prototype, EventProducer);
    HighlightContext.LoadingStarted = "LoadingStarted";
    HighlightContext.LoadingFinished = "LoadingFinished";
    HighlightContext.RedrawRequested = "RedrawRequested";

    function getValue(obj, funcOrValue) {
        return $.isFunction(funcOrValue) ? funcOrValue(obj) : funcOrValue;
    }

    /**
     * These functions are called in attr if the key passed in matches the key in this object.
     * They are also callable directly on some objects.
     */
    var attributeHooks = {
        color: function (funcOrColor) {
            if (funcOrColor !== undefined) {
                this._attributes.color = getValue(this, funcOrColor);
                return this;
            } else {
                return this._attributes.color;
            }
        },
        hasBorder: function (funcOrBool) {
            if (funcOrBool !== undefined) {
                this._attributes.hasBorder = !!getValue(this, funcOrBool);
                return this;
            } else {
                return this._attributes.hasBorder;
            }
        },
        borderColor: function (funcOrColorStr) {
            if (funcOrColorStr !== undefined) {
                this._attributes.borderColor = getValue(this, funcOrColorStr);
                return this;
            } else {
                return this._attributes.borderColor;
            }
        },
        borderWidth: function (funcOrNumber) {
            if (funcOrNumber !== undefined) {
                var width = getValue(this, funcOrNumber);
                if (width !== null && typeof width !== 'number') {
                    throw new Error("borderWidth must be a number or a function that returns a number.");
                }
                this._attributes.borderWidth = width;
                return this;
            } else {
                return this._attributes.borderWidth;
            }
        },
        textColor: function (funcOrColorStr) {
            if (funcOrColorStr !== undefined) {
                this._attributes.textColor = getValue(this, funcOrColorStr) || '';
                return this;
            } else {
                return this._attributes.textColor;
            }
        },
        backgroundColor: function (funcOrColorStr) {
            if (funcOrColorStr !== undefined) {
                this._attributes.backgroundColor = getValue(this, funcOrColorStr) || '';
                return this;
            } else {
                return this._attributes.backgroundColor;
            }
        },
        metadataCssClass: function (funcOrClassStr) {
            if (funcOrClassStr !== undefined) {
                this._attributes.metadataCssClass = getValue(this, funcOrClassStr) || '';
                return this;
            } else {
                return this._attributes.metadataCssClass;
            }
        }
    };

    function isObject(o) {
        return typeof o === 'object' && o !== null;
    }

    var isRelated = function (toCs, iterator, directionalCacheMap) {
        var cache;
        if (directionalCacheMap.has(toCs.csid)) {
            cache = directionalCacheMap.get(toCs.csid);
        } else {
            directionalCacheMap.set(toCs.csid, cache = new Map());
        }

        var endCurrentPath = false;
        var next;

        while (next = iterator(endCurrentPath)) {
            endCurrentPath = false;
            if (cache.get(next.csid) === false) {
                endCurrentPath = true;

            } else if (cache.get(next.csid) === true || next.csid === toCs.csid) {

                var changesetsInPath = iterator.getCurrentPathChangesets();
                Array.each(changesetsInPath, function (cs) {
                    cache.set(cs.csid, true);
                });

                return true;
            }
        }

        //if none of my ancestors were related, everything can be set to unrelated.
        var allCs = iterator.getAllTraversedChangesets();
        Array.each(allCs, function (cs) {
            cache.set(cs.csid, false);
        });
        return false;
    };

    /**
     * A mixin containing the basic functionality of a highlightable object.
     * Changesets/edges we expose to highlight-plugin developers should implement these methods.
     */
    var HighlightableObject = {

        color: attributeHooks.color,

        hasBorder: attributeHooks.hasBorder,

        borderColor: attributeHooks.borderColor,

        borderWidth: attributeHooks.borderWidth,

        metadataCssClass: attributeHooks.metadataCssClass,

        attr: function (attributesOrKey, value) {
            var attributeFunction;

            if (attributesOrKey === void 0) {
                return (this._attributes = this._attributes || {});
            }

            if (isObject(attributesOrKey) && attributesOrKey.hasOwnProperty) { // all attrs
                for (var attribute in attributesOrKey) {
                    if (attributesOrKey.hasOwnProperty(attribute)) {
                        if (attributeFunction = attributeHooks[attribute]) {
                            attributeFunction.call(this, attributesOrKey[attribute]);
                        } else {
                            throw new Error('Unrecognized attribute: "' + attribute + '"');
                        }
                    }
                }
            } else if (attributeFunction = attributeHooks[attributesOrKey]) { // single attr
                return attributeFunction.call(this, value);
            } else {
                if (attributesOrKey === null) {
                    throw new TypeError("Attribute object cannot be null.");
                } else {
                    throw new Error('Unrecognized attribute: "' + attributesOrKey + '"');
                }
            }

            return this;
        },

        isDescendantOf: function (ofObj) {
            if (this === ofObj) {
                return false;
            }

            var endCs = ofObj.isChangeset() ? ofObj : ofObj.child();
            var iterator = this.getAncestorIterator(endCs);

            return isRelated(endCs, iterator, this._context.descendantCache);
        },

        isAncestorOf: function (ofObj) {
            if (this === ofObj) {
                return false;
            }

            var endCs = ofObj.isChangeset() ? ofObj : ofObj.parent();
            var iterator = this.getDescendantIterator(endCs);

            return isRelated(endCs, iterator, this._context.ancestorCache);
        },

        isRelatedTo: function (toCs) {
            return toCs === this || this.isAncestorOf(toCs) || this.isDescendantOf(toCs);
        }
    };


    /**
     * An object that represents a changeset, as it is visible to the highlighter.
     * @param model the actual changeset model backing this HighlightableChangeset
     * @param highlightContext the highlightContext that created this HighlightableChangeset.
     */
    var HighlightableChangeset = function (model, highlightContext) {
        //Private
        this._attributes = {};
        this._context = highlightContext;

        this.init(model);
    };
    $.extend(HighlightableChangeset.prototype, HighlightableObject);

    HighlightableChangeset.prototype.init = function (model) {
        //Private
        this._model = model;

        // Public
        this.csid = model.id;
        this.branches = model.branches.slice(0);
        this.tags = model.tags.slice(0);
        this.order = model.order;
        this.commitDate = new Date(model.date);
        this.author = $.extend(true, {}, model.author);
        this.commentHtml = model.comment;
    };

    HighlightableChangeset.prototype.backgroundColor = attributeHooks.backgroundColor;

    HighlightableChangeset.prototype.textColor = attributeHooks.textColor;

    HighlightableChangeset.prototype.highlightedClass = attributeHooks.highlightedClass;

    HighlightableChangeset.prototype.parents = function () {
        var parents = Array.map(this._model.parents(), this._context.getChangeset);

        if (this._model.isLive()) {
            this.parents = function () {
                return parents;
            };
        }

        return parents;
    };

    HighlightableChangeset.prototype.children = function () {
        var children = Array.map(this._model.children(), this._context.getChangeset);

        if (this._model.isLive()) {
            this.children = function () {
                return children;
            };
        }

        return children;
    };

    HighlightableChangeset.prototype.getAncestorIterator = function (stopAtCs) {
        return Iterators.getDepthFirstChangesetIterator(
            this,
            stopAtCs ?
                (stopAtCs.order < this.order ? stopAtCs : this) :
                Iterators.ANCESTORS);
    };

    HighlightableChangeset.prototype.getDescendantIterator = function (stopAtCs) {
        return Iterators.getDepthFirstChangesetIterator(
            this,
            stopAtCs ?
                (stopAtCs.order > this.order ? stopAtCs : this) :
                Iterators.DESCENDANTS);
    };

    HighlightableChangeset.prototype.parentEdges = function () {
        var branchRevisions = this._model.branchRevisions;
        var edges = [];

        Array.each(branchRevisions, function (brev) {
            if (brev.parentEdges) {
                Array.each(brev.parentEdges, function (edge) {
                    edges.push(edge);
                });
            }
        });

        var parentEdges = Array.map(edges, this._context.getEdge);

        if (this._model.isLive()) {
            this.parentEdges = function () {
                return parentEdges;
            };
        }

        return parentEdges;
    };

    HighlightableChangeset.prototype.childEdges = function () {
        var branchRevisions = this._model.branchRevisions;
        var edges = [];

        Array.each(branchRevisions, function (brev) {
            if (brev.childEdges) {
                Array.each(brev.childEdges, function (edge) {
                    edges.push(edge);
                });
            }
        });

        var childEdges = Array.map(edges, this._context.getEdge);

        if (this._model.isLive()) {
            this.childEdges = function () {
                return childEdges;
            };
        }

        return childEdges;
    };

    HighlightableChangeset.prototype.isChangeset = function () {
        return true;
    };
    HighlightableChangeset.prototype.isEdge = function () {
        return false;
    };


    /**
     * An object that represents an edge, as it is visible to the highlighter.
     * @param model the edge model backing this HighlightableEdge
     * @param parent a HighlightableChangeset representing the parent on this edge
     * @param child a HighlightableChangeset representing the child on this edge
     * @param highlightContext the context that created this HighlightableEdge
     */
    var HighlightableEdge = function (model, parent, child, highlightContext) {
        this._attributes = {};
        this._model = model;
        this._context = highlightContext;
        this._parent = parent;
        this._child = child;

        // Public
    };
    $.extend(HighlightableEdge.prototype, HighlightableObject);

    HighlightableEdge.prototype.parent = function () {
        return this._parent;
    };
    HighlightableEdge.prototype.child = function () {
        return this._child;
    };

    HighlightableEdge.prototype.isChangeset = function () {
        return false;
    };
    HighlightableEdge.prototype.isEdge = function () {
        return true;
    };

    HighlightableEdge.prototype.getAncestorIterator = function (stopAtCs) {
        var parent = this.parent();
        return Iterators.getDepthFirstChangesetIterator(
            this,
            stopAtCs ?
                (stopAtCs.order <= parent.order ? stopAtCs : this.parent()) :
                Iterators.ANCESTORS);
    };

    HighlightableEdge.prototype.getDescendantIterator = function (stopAtCs) {
        var child = this.child();
        return Iterators.getDepthFirstChangesetIterator(
            this,
            stopAtCs ?
                (stopAtCs.order >= child.order ? stopAtCs : this.child()) :
                Iterators.DESCENDANTS);
    };

    var pluginHighlighters = [];
    var addHighlighter = function (highlighter) {
        pluginHighlighters.push(highlighter);//TODO: check for clashes in names etc
    };
    var getPluginHighlighters = function () {
        return pluginHighlighters;
    };

    return {
        HighlightContext: HighlightContext,
        HighlightableObject: HighlightableObject,
        HighlightableChangeset: HighlightableChangeset,
        HighlightableEdge: HighlightableEdge,
        addHighlighter: addHighlighter,
        getPluginHighlighters: getPluginHighlighters,
        COLOR_BY_BRANCH: COLOR_BY_BRANCH
    };
})(AJS.$));
/*[{!highlight_js_izd752w!}]*/