FE.VIS.BranchHeader = (function ($, templateFactory) {
    /* Imports */
    var Vis = FE.VIS;
    var CHANGESET_SPACING = Vis.Configuration.CHANGESET_SPACING;
    var AbstractPositioner = Vis.POSITIONING.AbstractPositioner;
    var colorForBranch = Vis.BranchSet.colorForBranch;
    var BranchSet = Vis.BranchSet;
    var Map = FECRU.DATA_STRUCTURES.Map;
    var Dialog = FECRU.DIALOG;

    // This variable is used to set an id on each branch head lozenge in the main dag - _not_ in the branch dialog
    var BRANCH_HEAD_ID_PREFIX = "branch-head-";
    var MAX_BRANCHES = 20;

    /*privates */
    /**
     * Add a branch to the given container, with the given name and color. Uses the template 'branch-head'
     * @param $container the container to add the branch element to
     * @param branchName the name of the branch
     * @param color the color - all css color formats accepted.
     * @param idCreator an optional function that will return an id for the given branchName
     */
    function appendBranchToContainer($container, branchName, color, idCreator, makeOpaque) {
        if (!branchName) {
            return null;
        }

        var html = templateFactory.load('branch-head').fill({
            'branchName': branchName
        }).toString();

        var $elem = $(html)
            .data("branch-name", branchName)
            .css({
                "background-color": color,
                "border-color": color,
                "opacity": makeOpaque ? 0.5 : 1
            })
            .appendTo($container);

        if (idCreator && $.isFunction(idCreator)) {
            $elem.attr("id", idCreator(branchName));
        }
        return $elem;
    }

    function setupDragging($branchList, onUpdate, axis) {
        $branchList.sortable({
            tolerance: "intersect",
            helper: "clone",
            handle: ".draggable-handle",
            placeholder: "branch-placeholder branch-head",
            opacity: 0.4,
            cursor: "move",
            scroll: false,
            update: onUpdate,
            axis: axis || false
        });
    }

    function isAlreadySelected(branchName, $branchList) {
        var branches = [];
        $branchList.find("input.branch-name-holder").each(function () {
            branches.push($(this).val());
        });
        return Array.any(branches, function (branch) {
            return branch === branchName;
        });
    }

    /*public*/
    /**
     * Create a new branch header.
     * @param $container the containing element of the branch header
     * @param branchSet a reference to the branchset
     * @param positioner the DAG positioner (used for calculating widths)
     */
    var BranchHeader = function ($container, branchSet, positioner) {
        var self = this;
        this.$branchSetContainer = $("#branch-header-list");
        this.initialise(branchSet);

        var branchHeadPopupSelector = ".branch-head:not(.branch-placeholder,.ui-sortable-helper)";

        // If we are in "All branches mode" then only set the popup within the dialog, otherwise use all headers
        if (self.branchSet.isAllInOne()) {
            branchHeadPopupSelector = "#dialog-branch-editor " + branchHeadPopupSelector;
        }

        new BranchHeaderPopup(branchHeadPopupSelector, "branch-head-popup", {});

        if (!self.branchSet.isAllInOne()) {
            setupDragging(this.$branchSetContainer, function (event, ui) {
                var branches = Array.map($(this).sortable("toArray"), function (elem) {
                    return elem.slice(BRANCH_HEAD_ID_PREFIX.length);
                });
                if (branches.length > MAX_BRANCHES) {
                    FECRU.AJAX.appendErrorMessage('You cannot reorder branches as you have more than ' + MAX_BRANCHES + '. Remove branches before reordering.');
                    FECRU.AJAX.showUserErrorBox();
                    $(this).sortable('cancel');
                    return;
                }

                self.branchSet.trigger(BranchSet.BranchesReordered, {
                    // "this" is the sortable element in the context of this callback
                    branches: branches
                });
            }, "x");

            $container.delegate(".branch-remove", "click", function () {
                var $branchHead = $(this).closest(".branch-head");
                var branchName = $branchHead.data("branch-name");
                var visibleBranchNames = self.branchSet.getBranchNames();

                var removedBranchIndex = $.inArray(branchName, visibleBranchNames);
                if (removedBranchIndex === -1) {
                    return; // How did we end up in this state;
                }

                Array.remove(visibleBranchNames, removedBranchIndex);
                $branchHead.remove();

                self.branchSet.trigger(BranchSet.BranchesRemoved, {
                    removedBranches: [
                        branchName
                    ],
                    visibleBranches: visibleBranchNames
                });
            });
        }
        positioner.bind(AbstractPositioner.BranchExpandedEvent, function (e) {
            var width = e.breadth * CHANGESET_SPACING;
            var branch = self.elementByBranchName.get(e.branchName);
            if (branch) {
                branch.width(width - 4 + "px");
            }
        });

        var $dialogContents = $("#dialog-branch-editor");
        var $branchAutocomplete = $(".fecru-autocomplete-wrapper");

        $dialogContents.find("#autocomplete-holder").append($branchAutocomplete);
        this.createDialog($dialogContents);
        if (this.infoMsg) {
            self.branchDialog.showInfoMsg(this.infoMsg);
        }

        $("#branch-modifier").click(function () {
            self.branchDialog.resetAndShow();
            return false;
        });
    };

    BranchHeader.prototype.initialise = function (branchSet) {
        this.elementByBranchName = new Map();
        this.setBranchSet(branchSet);
        if (branchSet.getRepositoryType() === 'GIT') {
            this.infoMsg =
                $("<strong>Note:</strong> <em class='git-branch-notice'>Git repositories (such as this one) can have commits that appear in a number of different branches. "
                    + "To visualise this, we put the commit on the left-most item it occurs. "
                    + "<a href='https://confluence.atlassian.com/display/FISHEYE/Ordering+of+Branches+Important+When+Visualising+Git+Changesets'>Read more...</a></em>");
        }
    };

    BranchHeader.prototype.setBranchSet = function (branchSet) {
        this.branchSet = branchSet;
        var branches = branchSet.isAllInOne() ? [branchSet.getAllInOne()] : branchSet.getAllBranches();
        var $container = this.$branchSetContainer.empty();
        var elementByBranchName = this.elementByBranchName;

        Array.each(branches, function (branch) {
            var $elem = appendBranchToContainer($container, branch.name, branch.color, function (branchName) {
                return BRANCH_HEAD_ID_PREFIX + branchName;
            });
            elementByBranchName.set(branch.name, $elem);
        });
        if (branchSet.isAllInOne()) {
            $("#branch-header-list .branch-remove").remove();
        }
    };

    /**
     * Create a branch dialog. Will only create one.
     */
    var dialogCreated = false;
    BranchHeader.prototype.createDialog = function ($dialogContents) {
        if (dialogCreated) {
            return;
        }
        dialogCreated = true;

        var self = this;
        var isAllBranchesMode = false;
        var $branchList = $dialogContents.find(".branch-list");
        var $branchInput = $dialogContents.find("#branch-modifier-dialog-autocomplete-input");
        var $errorMsgHolder = $dialogContents.find("#dialog-error-message-holder").hide();
        var $infoMsgHolder = $dialogContents.find("#dialog-info-message-holder").hide();
        var $toggle = $dialogContents.find(".toggle-branch-mode");

        var hideDialog = function () {
            $branchInput.trigger("reset.autocomplete");
            self.branchDialog && self.branchDialog.hide();
        };

        var getNewBranchNames = function () {
            var newBranches = [];
            $dialogContents.find("input.branch-name-holder").each(function () {
                newBranches.push($(this).val());
            });
            return newBranches;
        };

        var saveDialog = function () {
            var newBranches = getNewBranchNames();
            if (newBranches.length > MAX_BRANCHES) {
                flashErrorMsg('You have selected too many branches. Please remove branches to bring your selection to less than or equal to ' + MAX_BRANCHES + ' branches.');
                return;
            }
            self.branchSet.saveBranches(newBranches, isAllBranchesMode);
            hideDialog();
        };

        var switchContentsToBranchLaneMode = function () {
            $toggle.text("Switch to all branches mode");
            var newBranchNames = getNewBranchNames();
            $branchList.empty();
            isAllBranchesMode = false;
            Array.each(newBranchNames, function (branchName) {
                addBranch(branchName, isAllBranchesMode);
            });
        };


        $toggle.bind('click', function () {
            isAllBranchesMode = !isAllBranchesMode;
            if (isAllBranchesMode) {
                saveDialog();
            } else {
                switchContentsToBranchLaneMode();
            }
        });

        var flashErrorMsg = function (msg) {
            $errorMsgHolder.text(msg);
            $errorMsgHolder.show();
        };

        var resetAutoComplete = function () {
            $branchInput.triggerHandler("placeholderReset.jQueryPlaceholder");
            $branchInput.trigger("reset.autocomplete");
        };

        var checkForBranchNameError = function (branchName, autoCompletResult) {
            if (!branchName) {
                //just ignore - could be a typo/mis-click
                return false;
            } else if (isAlreadySelected(branchName, $branchList)) {
                //no need to show error msg when selecting the same branch again and again
                resetAutoComplete();
                return false;
            } else if (autoCompletResult && !(autoCompletResult.data.length > 0)) {
                flashErrorMsg("Cannot find the branch '" + branchName + "' - please check that the branch exists.");
                return false;
            } else {
                $errorMsgHolder.hide();
                return true;
            }
        };


        $branchInput.fecruSetOptions({
            isItemChecked: function (value) {
                return isAlreadySelected(value, $branchList);
            }
        });

        var addBranchAndSwitchMode = function (branchName, preserveAutoComplete) {
            if (checkForBranchNameError(branchName)) {
                if (self.branchSet && isAllBranchesMode) {
                    switchContentsToBranchLaneMode();
                }
                addBranch(branchName, isAllBranchesMode, preserveAutoComplete);
            }
        };
        /**
         * @param branchName the string name of the branch to add. Case sensitive.
         */
        var addBranch = function (branchName, isAllBranchesMode, preserveAutoComplete) {
            if (checkForBranchNameError(branchName)) {
                appendBranchToContainer($branchList, branchName, colorForBranch(branchName), undefined, isAllBranchesMode);
                if (!preserveAutoComplete) {
                    resetAutoComplete();
                }
            }
        };
        var removeBranch = function (branchName) {
            if (branchName) {
                $branchList.find('input').filter(function () {
                    return this.value === branchName;
                }).closest('.branch-head').remove();
            }
        };

        self.branchDialog = Dialog.create(600, 400, undefined, {
            preferredFocusSelector: '.toggle-branch-mode'
        })
            .addHeader("Select Branches")
            .addButton("Apply", saveDialog, "branch-dialog-ok-button")
            .addLink("Cancel", hideDialog, "branch-dialog-cancel-button")
            .addPanel("Add Branches", $dialogContents);

        self.branchDialog.resetAndShow = function () {
            $branchList.empty();
            if (self.branchSet) {
                isAllBranchesMode = self.branchSet.isAllInOne();
                Array.each(self.branchSet.getBranchNames(), function (branchName) {
                    addBranch(branchName, isAllBranchesMode);
                });
            }
            self.branchDialog.show();
        };

        self.branchDialog.showInfoMsg = function (msg) {
            $infoMsgHolder.empty().append(msg).show();
        };

        var enterPressed = false;
        $branchInput.bind("result", function (event, data, valueOrValues, unselectedValues) {
            if ($.isArray(valueOrValues)) {
                Array.each(valueOrValues, function (branchName) {
                    addBranchAndSwitchMode(branchName);
                });
                unselectedValues && Array.each(unselectedValues, function (branchName) {
                    removeBranch(branchName);
                });
            } else {
                addBranchAndSwitchMode(valueOrValues);
            }
            enterPressed = false;
            //Disable default autocomplete handling where the selected value is set on the input
            event.stopImmediatePropagation();
        }).bind("selection", function (event, branchName, checked) {
            if (checked) {
                addBranchAndSwitchMode(branchName, true);
            } else {
                removeBranch(branchName)
            }
        }).bind("autocomplete-data-received", function (event, result) {
            var inputVal = $branchInput.val();
            if (enterPressed) {
                if (result.data.length > 0) {
                    var dataList = result.data;
                    //try to find a match in the result. If found, add branch, if not found, dont do anything
                    var match = Array.first(dataList, function (matchedItem) {
                        return matchedItem.data.id === inputVal;
                    });
                    if (match) {
                        addBranchAndSwitchMode(match.data.id);
                    } else if (dataList.length === 1) {
                        addBranchAndSwitchMode(dataList[0].data.id);
                    }
                } else {
                    checkForBranchNameError(inputVal, result);
                }
                enterPressed = false;
            }
        }).keydown(function (event) {
            if (event.which === $.ui.keyCode.ENTER) { //the enter key
                enterPressed = true;
                event.preventDefault();
                event.stopPropagation();

            }
        }).change(function () {
            enterPressed = false;
        });

        $dialogContents.delegate(".branch-remove", "click", function () {
            $(this).closest(".branch-head").remove();
            $branchInput.resetSelectionCache();
            switchContentsToBranchLaneMode();
        });

        setupDragging($branchList, $.noop);
    };

    var BranchHeaderPopup = function (selector, id, opts) {
        var self = this;
        var getArrowAttributes = function () {
            return {
                fill: self.dialogColor,
                stroke: self.dialogColor
            };
        };

        var defaults = {
            onHover: true,
            fadeTime: 0,
            hideDelay: 0,
            showDelay: 0,
            getArrowAttributes: getArrowAttributes,
            displayShadow: false,
            useLiveEvents: true,
            cacheContent: false,
            getArrowPath: function () {
                return "M0,8L6,2,12,8";
            },
            calculatePositions: function (popup, targetPosition, mousePosition, opts) {
                // CONSTANTS
                var PADDING_TOP = 0;
                var PADDING_LEFT = 0;
                var ARROW_HEIGHT = 6;

                var $header = targetPosition.target.closest(selector);
                var branchName = $header.data("branch-name");
                var offset = $header.offset();

                // Get the dialog colour here to avoid a race condition in the dialog
                if (branchName) {
                    self.dialogColor = colorForBranch(branchName);
                }

                var arrowLeft = 2;
                var arrowTop = -ARROW_HEIGHT;
                var popupLeft = offset.left + PADDING_LEFT;
                var popupTop = offset.top + $header.outerHeight() + PADDING_TOP + ARROW_HEIGHT;

                return {
                    displayAbove: false,
                    popupCss: {
                        left: popupLeft,
                        right: "auto",
                        top: popupTop
                    },
                    arrowCss: {
                        position: "absolute",
                        left: arrowLeft,
                        right: "auto",
                        top: arrowTop
                    }
                }
            }

        };

        this.dialogColor = "#333";

        var hoverOpts = $.extend({}, defaults, opts);

        this.initialise(selector, id, hoverOpts);
    };

    BranchHeaderPopup.prototype.initialise = function (selector, id, opts) {
        if (this.popup) {
            return; // don't multi-init
        }
        var self = this;
        var lastTrigger = undefined;

        var hidePopup = function () {
            if (self.popup) {
                // Set the display property rather than calling "hide()" because InlineDialog overrides this method
                // and it does not do what we expect
                self.popup.css("display", "none");
            }
        };

        this.popup = AJS.InlineDialog(selector, id, function (contents, trigger, showPopup) {
            var $contents = $(contents).empty();
            var branchName = $(trigger).data("branch-name");

            if (branchName) {
                self.dialogColor = colorForBranch(branchName);
                $contents.append(
                    $("<span></span>").text(branchName)
                ).css("backgroundColor", self.dialogColor);

                if (lastTrigger && lastTrigger !== trigger) {
                    hidePopup();
                }
                lastTrigger = trigger;
                showPopup();
            } else {
                hidePopup();
            }
        }, opts);
    };

    return BranchHeader;
})(AJS.$, AJS.template);
/*[{!branch_header_js_hqen52o!}]*/