FE.VIS.BranchSet = (function ($) {
    /**
     * Imports
     */
    var Map = FECRU.DATA_STRUCTURES.Map;
    var Fragment = FE.VIS.Fragment;
    var Configuration = FE.VIS.Configuration;
    var EventProducer = FECRU.MIXINS.EventProducer;
    var numericHash = FECRU.numericHash;

    /**
     * The set of colors that branches can have
     */
    var colors = [
        "#5e7084",
        "#326ca6",
        "#8c9ca9",
        "#2b8fb2",
        "#3d4a50",
        "#076c76",
        "#4e7c62",
        "#066006",
        "#7d8712",
        "#767649",
        "#330660",
        "#613991",
        "#85557d",
        "#841484",
        "#600633",
        "#ba6e72",
        "#900909",
        "#b7440b",
        "#ca7f0c",
        "#906c09"
    ];
    var defaultBranchColor = "#003366";

    /*private*/

    function createBranch(name, color) {
        return {
            name: name,
            breadth: 0,
            branchPosition: 0,
            color: color
        };
    }

    /*public*/

    /**
     * Create a branchset that models the set of branch in the graph
     * @param branchNames an array of branch names
     * @param allBranchesMode a boolean indicating whether the current display mode is 'all in one' or multi branch
     * @param options settings (the repo name, type and the url to use to save the branch)
     */
    var BranchSet = function (branchNames, allBranchesMode, options) {
        allBranchesMode = !!allBranchesMode;
        var allInOneBranch = createBranch('All Branches', '#666');
        var self = this;
        var settings = $.extend(options, {});

        this.repositoryName = settings.repositoryName;
        this.repositoryType = settings.repositoryType;
        this.saveBranchesUrl = settings.saveBranchesUrl;
        this.hashFragment = settings.hashFragment || Fragment.fromString("#", {branch: branchNames}, allBranchesMode);

        this.get = function (branchName) {
            return this.branchByName.get(branchName);
        };

        this.getAllBranches = function () {
            return this.iterable.slice(0);
        };

        this.getBranchNames = function () {
            return this.branchNames.slice(0); // clone
        };

        this.isAllInOne = function () {
            return allBranchesMode;
        };

        this.getAllInOne = function () {
            return allInOneBranch;
        };

        this.getRepositoryName = function () {
            return this.repositoryName;
        };

        this.getRepositoryType = function () {
            return this.repositoryType;
        };

        this.addBranch = function (branchName) {
            if (this.isAllInOne()) {
                //switch out of all-in-one.
                reinit(this, [branchName], false);
            } else if (!this.get(branchName)) {
                var branch = createBranch(branchName, BranchSet.colorForBranch(branchName));
                self.branchByName.set(branchName, branch);
                self.iterable.push(branch);
                self.branchNames.push(branchName);
            }
        };

        this.removeBranch = function (branchName) {
            var branch = this.branchByName.get(branchName);
            if (branch) {
                this.branchByName.remove(branchName);
                var branchIndex = $.inArray(branch, self.iterable);
                Array.remove(self.iterable, branchIndex);
            }
        };

        function reinit(self, branchNames, isAllBranchesMode) {
            self.branchByName = new Map();
            self.iterable = [];
            self.branchNames = [];
            allBranchesMode = !!isAllBranchesMode;
            if (allBranchesMode) {
                self.branchNames = branchNames;
                self.iterable.push(allInOneBranch);
            } else {
                branchNames && Array.each(branchNames, function (name) {
                    self.addBranch(name);
                });
                //save the changeset branch if it exists
                if (self.hashFragment.getChangesetBranch()) {
                    //pass in $.noop as callback because the is no need to trigger any call back here - the set of branches,
                    //as far as other components are concerned, has not been modified (so we dont trigger any events either)
                    self.saveBranches(branchNames, allBranchesMode, $.noop, $.noop);
                }
            }
        }

        //TODO: change to FECRU.ajax(), not $.ajax()
        this.saveBranches = function (branchesToSave, allBranchesMode, successCallback, errorCallback) {
            var self = this;
            var onSuccess = successCallback || function () {
                    self.trigger(BranchSet.BranchModified, {
                        branches: branchesToSave
                    });
                };
            var onError = errorCallback || function () {
                    AJS.log("error saving branches " + branchesToSave);
                };
            $.ajax({
                url: self.saveBranchesUrl,
                type: "POST",
                data: {
                    repoName: self.getRepositoryName(),
                    'vbs': '{"' + self.getRepositoryName() + '":{' + (branchesToSave ? ('bl:[' +
                    Array.map(branchesToSave, function (b) {
                        return '"' + FECRU.quoteString(b) + '"';
                    }).join(',') + '], ') : '') + 'm:"' + (allBranchesMode ? 'A' : 'C') + '"}}'
                },
                success: function (data, textStatus, jqXHR) {
                    onSuccess && onSuccess(data, textStatus, jqXHR);
                },
                error: function (jqXHR, textStatus, errorThrown) {
                    onError && onError(jqXHR, textStatus, errorThrown);
                }
            });

            if (window.localStorage) {
                window.localStorage[FE.VIS.localStorageKeys(this.repositoryName).branches] = JSON.stringify(branchesToSave);
                window.localStorage[FE.VIS.localStorageKeys(this.repositoryName).mode] = allBranchesMode ? 'ALL' : 'CUSTOM';
            }
        };

        reinit(this, branchNames, allBranchesMode);
    };

    function isDefaultBranch(branchName) {
        return Configuration.visualiserSetting.defaultBranchName === branchName;
    }

    BranchSet.colorForBranch = function (branchName) {
        return isDefaultBranch(branchName) ? defaultBranchColor : colors[numericHash(branchName, colors.length)];
    };
    //fallback if no branch exists.
    BranchSet.fallbackBranchColor = "#666";

    /**
     * Events that exist on the branchset. You can obtain a reference to the branchset instance and
     * call branchSet.trigger(_event_name_, {...parametersMap...}) to trigger the event (all bound functions to that event will execute), or
     * branchSet.bind(_event_name_, function(parametersMap){...code...}); to bind the event.
     */
    BranchSet.BranchModified = "BranchModified";
    BranchSet.BranchesAdded = "BranchesAdded";
    BranchSet.BranchesRemoved = "BranchesRemoved";
    BranchSet.BranchResized = "BranchResized";
    BranchSet.BranchesReordered = "BranchesReordered";

    $.extend(BranchSet.prototype, EventProducer);

    return BranchSet;
})(AJS.$);
/*[{!branch_set_js_hc1j52h!}]*/