var FE = FE || {};
FE.BRANCH_TAG_SWITCHER = (function ($, Backbone, fecruAjax) {
    'use strict';

    var ResultEntry = Backbone.Model.extend({});
    var DataSet = Backbone.Collection.extend({
        model: ResultEntry,
        currentRequest: undefined,
        initialLoading: true,
        initialize: function (name, options) {
            this.name = name;
            Backbone.Collection.prototype.initialize([], options);
        },
        parse: function (response) {
            this.totalItems = response.totalItems;
            return response.items;
        },
        query: function (query) {
            if (this.currentQuery === query) {
                return;
            }
            var self = this;
            this.currentQuery = query;
            this.trigger('loading');
            if (this.currentRequest) {
                this.currentRequest.abort();
            }
            this.currentRequest = this.fetch({
                reset: true,
                data: {
                    q: query || ''
                }
            });
            this.currentRequest.always(function () {
                self.currentRequest = undefined;
            });
        }
    });
    var BranchTagSwitcher = Backbone.Model.extend({
        defaults: {
            phrase: '',
            searchType: 'branches'
        },
        initialize: function (options) {
            this.dataSets = {
                branches: new DataSet('branches', {
                    url: options.branchesSource
                }),
                tags: new DataSet('tags', {
                    url: options.tagsSource
                })
            };

            var onResponse = function (dataSetName, dataSet) {
                if (this.getSearchType() === dataSetName) {
                    this.trigger('results', dataSet, dataSetName);
                }
            };
            _.each(this.dataSets, function (dataSet, dataSetName) {
                dataSet.on('sync', $.proxy(onResponse, this, dataSetName));
            }, this);
        },
        search: function (phrase) {
            phrase = phrase || '';
            var dataSet = this.getCurrentDataSet();
            this.set('phrase', phrase);
            dataSet.query(phrase);
        },
        getPhrase: function () {
            return this.get('phrase');
        },
        setSearchType: function (searchType) {
            this.set('searchType', searchType);
        },
        getSearchType: function () {
            return this.get('searchType');
        },
        getCurrentDataSet: function () {
            return this.dataSets[this.getSearchType()];
        },
        getDataSets: function () {
            return this.dataSets;
        }
    });

    var DefaultView = (function () {
        var View = function () {
            this.$container = $('#branch-selector');
            this.$button = this.$container.find('.aui-dropdown2-trigger');
            this.$dropdown = $('#branch-selector-dropdown');
            this.$searchContainer = $('#branch-selector-search');
            this.$searchForm = this.$searchContainer.find('form');
            this.$searchInput = this.$searchForm.find('input');
            this.$allBranchesLink = $('#switch-all-branches');
            this.lists = {
                branches: $('#search-branches'),
                tags: $('#search-tags')
            };

            this.includeBranchHeadParam = this.$container.data('include-branch-head-param');

            this.$loader = $('<span class="aui-icon aui-icon-wait loading">Wait</span>');
            this.$listLoader = $('<li class="loading-wrapper"><span class="aui-icon aui-icon-wait loading">Wait</span></li>');

            this.switcher = new BranchTagSwitcher({
                branchesSource: this.lists.branches.data('url'),
                tagsSource: this.lists.tags.data('url')
            });

            this.selectedIndex = -1;
            this.selectableElements = this.$allBranchesLink;

            this.baseUrl = this.$container.data('url');

            this.initResultHandling();
            this.initDropdown();
            this.initSearchInput();
            this.initLoading();
            this.initTabs();
            this.initAutoFocus();
            this.initSelection();
        };

        View.prototype.initDropdown = function () {
            var self = this;
            this.$dropdown.one('aui-dropdown2-show', function () {
                self.switcher.search();
                self._refreshSelectableElements();
            });
        };

        View.prototype.initResultHandling = function () {
            var self = this;
            var templates = {
                noResults: _.template('<li class="no-results">No <%= type %> found</li>'),
                moreElements: _.template('<li class="total-results">' +
                    'Displaying <%= maxResults %> of <%= totalResults %> results.<br/>' +
                    '<%= secondPartOfMessage %>' +
                    '</li>'),
                entry: _.template('<li>' +
                    '<a class="interactive" href="<%= url %>">' +
                    '<span class="aui-icon aui-icon-small aui-iconfont-devtools-<%= icon%>"><%= icon %></span>' +
                    '<span class="reference-name"><%- name %></span>' +
                    '</a>' +
                    '</li>')
            };

            this.switcher.on('results', function (dataSet, searchType) {
                var list = self.lists[searchType].find('ul').get(0);
                var icon = (searchType === 'branches' ? 'branch' : 'tag');
                var dataset = self.switcher.getCurrentDataSet();
                var hasMore = dataSet.totalItems > dataSet.length;
                var resultHtml = '';

                list.setAttribute('data-query', dataSet.currentQuery);
                dataSet.each(function (element) {
                    var url = self.baseUrl + (searchType === 'branches' ? '?wbbr=' : '?wbtag=');
                    url += encodeURIComponent(element.get('displayPrimary'));
                    url += '&wbjump=jump&brjump=jump';
                    if (self.includeBranchHeadParam) {
                        url += '&' + self.includeBranchHeadParam + '=true';
                    }
                    resultHtml += templates.entry({
                        name: element.get('displayPrimary'),
                        icon: icon,
                        url: url
                    });
                });

                if (dataSet.length === 0) {
                    resultHtml += templates.noResults({type: searchType});
                }

                if (hasMore) {
                    resultHtml += templates.moreElements({
                        maxResults: dataset.length,
                        totalResults: dataset.totalItems,
                        secondPartOfMessage: self.switcher.getPhrase() ? 'Continue typing to filter.' : 'Start typing to filter.'
                    });
                }

                list.innerHTML = resultHtml;
                self._refreshSelectableElements();
            });
        };

        View.prototype.initSearchInput = function () {
            var self = this;
            this.$searchInput.keyup($.debounce(200, function () {
                if (this.value !== self.switcher.getPhrase()) {
                    self.switcher.search(this.value);
                }
            }));
        };

        View.prototype.initLoading = function () {
            var self = this;
            var currentlyLoading = 0;
            var dataSets = this.switcher.getDataSets();
            var onLoading = function () {
                currentlyLoading++;
                if (this.initialLoading) {
                    var list = self.lists[this.name].find('ul');
                    if (list.find('.loading-wrapper').length === 0) {
                        list.append(self.$listLoader.clone());
                    }
                } else if (currentlyLoading === 1) {
                    self.$searchForm.append(self.$loader);
                }
            };
            var onSync = function () {
                currentlyLoading--;
                if (currentlyLoading === 0) {
                    self.$loader.remove();
                }
            };
            var onError = function (dataset, request) {
                currentlyLoading--;
                // request.status === 0 if aborted
                if (request.status === 0) {
                    return;
                }

                fecruAjax.appendErrorMessage('Cannot fetch results for branch/tag switcher. ' +
                    'Please check your internet connection or try again later');
                fecruAjax.showErrorBox();

                if (this.initialLoading) {
                    removeInitialLoading.call(this);
                } else {
                    onSync();
                }
            };
            var removeInitialLoading = function () {
                self.lists[this.name].find('ul .loading-wrapper').remove();
                this.initialLoading = false;
            };

            _.each(dataSets, function (dataSet) {
                dataSet.on('loading', onLoading);
                dataSet.on('sync', onSync);
                dataSet.once('sync', removeInitialLoading);
                dataSet.on('error', onError);
            });
        };

        View.prototype.initTabs = function () {
            var self = this;
            this.$searchContainer.on('tabSelect', function (e, eventData) {
                var searchType = eventData.pane.attr('id') === 'search-branches' ? 'branches' : 'tags';
                self.$searchInput.attr('placeholder', 'Filter ' + searchType);
                self.switcher.setSearchType(searchType);
                self.switcher.search(self.switcher.getPhrase());
                self._refreshSelectableElements();
            });
        };

        View.prototype.initAutoFocus = function () {
            var self = this;
            var autoFocus = function () {
                self.$searchInput.focus();
            };

            this.$searchContainer.on('tabSelect', autoFocus);
            this.$dropdown.on('aui-dropdown2-show', autoFocus);
        };

        View.prototype._refreshSelectableElements = function () {
            this.selectableElements = this.$searchContainer.find('.active-pane li > a').add(this.$allBranchesLink);
            this.selectedIndex = -1;
        };

        View.prototype.initSelection = function () {
            var self = this;
            var keyCode = AJS.$.ui.keyCode;
            var $document = $(document);
            var setSelected = function (index) {
                self.selectedIndex = index;
                self.selectableElements.removeClass('active');
                if (index !== -1) {
                    self.selectableElements.eq(index).addClass('active');
                }
            };
            var moveUp = function () {
                var newIndex = self.selectedIndex - 1;
                if (newIndex <= -1) {
                    newIndex = self.selectableElements.length - 1;
                }
                setSelected(newIndex);
            };
            var moveDown = function () {
                var newIndex = self.selectedIndex + 1;
                if (newIndex >= self.selectableElements.length) {
                    newIndex = 0;
                }
                setSelected(newIndex);
            };
            var onKeyDown = function (e) {
                e.stopImmediatePropagation();
                if (!self.selectableElements.length) {
                    return;
                }
                switch (e.keyCode) {
                    case keyCode.TAB:
                        e.shiftKey ? moveUp() : moveDown();
                        break;

                    case keyCode.DOWN:
                        moveDown();
                        break;

                    case keyCode.UP:
                        moveUp();
                        break;

                    case keyCode.ENTER:
                        e.preventDefault();
                        select();
                        break;
                }
            };
            var select = function () {
                if (self.selectedIndex < 0) {
                    return;
                }
                var $link = self.selectableElements.eq(self.selectedIndex);
                var buttonText = $link.is(self.$allBranchesLink) ? $link.html() : $link.find('.reference-name').html();

                self.$button
                    .prop('disabled', true)
                    .trigger('aui-button-invoke')
                    .find('.button-text').html(buttonText).end()
                    .find('.aui-icon').attr('class', 'aui-icon aui-icon-wait');

                // without setTimeout safari won't repaint dropdown and dropdown button
                setTimeout(function () {
                    document.location.href = $link.attr('href');
                }, 0);
            };

            var selectableElementsSelector = '.active-pane li > a, #switch-all-branches';
            this.$dropdown
                .on('aui-dropdown2-show', function () {
                    $document.on('keydown', onKeyDown);
                })
                .on('aui-dropdown2-hide', function () {
                    $document.off('keydown', onKeyDown);
                })
                .on('mouseover', selectableElementsSelector, function (e) {
                    self.selectedIndex = self.selectableElements.index(e.currentTarget);
                })
                // prevent deselection of results when hover on tabs
                .on('mousemove', '.tabs-menu a', function (e) {
                    e.stopImmediatePropagation();
                })
                .on('click', selectableElementsSelector, function (e) {
                    e.preventDefault();
                    select();
                });

            var selectFirst = $.proxy(setSelected, undefined, 0);
            this.switcher.on('results', selectFirst);
            this.$searchContainer.on('tabSelect', selectFirst);
            this.$dropdown.on('aui-dropdown2-show', function () {
                // required to prevent AUI dropdown2 from selecting (therefore deselecting our choice) "first" item
                setTimeout(selectFirst, 0);
            });
        };

        return View;
    })();

    return {
        init: function () {
            return (new DefaultView).switcher
        }
    };

})(AJS.$, Backbone, FECRU.AJAX);
/*[{!branch_tag_switcher_js_b0o6524!}]*/