/**
 * Crucible utility functions that can be used from any crucible page.
 */

window.CRU = window.CRU || {};
CRU.UTIL = {};

(function ($) {
    var cruUtil = CRU.UTIL;
    /**
     * Crucible base url (without trailing '/').
     *
     * @param permaId optional review id
     */
    cruUtil.urlBase = function (permaId) {
        if (permaId) {
            return FECRU.pageContext + '/cru/' + permaId;
        } else {
            return FECRU.pageContext + '/cru';
        }
    };

    /**
     * Crucible JSON base url (without trailing '/').
     *
     * @param permaId optional review id
     */
    cruUtil.jsonUrlBase = function (permaId) {
        if (permaId) {
            return FECRU.pageContext + '/json/cru/' + permaId;
        } else {
            return FECRU.pageContext + '/json/cru';
        }
    };

    cruUtil.isReviewPage = function () {
        return typeof review !== 'undefined';
    };


    cruUtil.startAjaxDialogSpin = function () {
        $('body').addClass('ajax-dialog');
        AJS.dim();
    };

    cruUtil.stopAjaxDialogSpin = function () {
        AJS.undim();
        $('body').removeClass('ajax-dialog');
    };

    cruUtil.isAjaxDialogSpinning = function () {
        return cruUtil.isDimmed() && $('body').hasClass('ajax-dialog');
    };

    var _onReviewStateTransitCallbacks = {};
    var triggerCallbacks = function (command, resp) {
        for (var id in _onReviewStateTransitCallbacks) {
            if (_onReviewStateTransitCallbacks.hasOwnProperty(id)) {
                _onReviewStateTransitCallbacks[id](command, resp);
            }
        }
    };
    /**
     * The callback will be called when the state of review transits.
     *
     * @param callbackId : string -- enable more than one callbacks to be bound
     * @param callback : function ( command, resp )
     */
    cruUtil.onReviewStateTransit = function (callbackId, callback) {
        _onReviewStateTransitCallbacks[callbackId] = callback;
    };

    cruUtil.ajaxDialog = function (url, params, isFromStateTransition) {
        cruUtil.startAjaxDialogSpin();
        FECRU.AJAX.ajaxDo(url, params || {}, function (resp) {
            var complete = function () {
                if (isFromStateTransition) {
                    triggerCallbacks(params.command, resp);
                }
            };

            if (resp.worked) {
                if (resp.showDialog) {
                    cruUtil.stopAjaxDialogSpin();
                    try {
                        FECRU.DIALOG.getAjaxDialogContainer().html(resp.payload);
                        FECRU.DIALOG.triggerAjaxDialogLoaded();

                        complete();
                    } catch (e) {
                        alert(e);
                    }
                } else if (resp.redirect) {
                    complete();
                    // no need to undim
                    window.location = resp.payload;
                }
            }
            // !resp.worked handled by ajaxDo
        });
        return false;
    };

    cruUtil.stateTransition = function (transition, permaId, params) {
        var util = cruUtil;
        var url = util.jsonUrlBase(permaId) + '/changeStateAjax';
        var unsaved = CRU.UNSAVED;

        // Make sure there aren't any unsaved inputs on the page, and if there are, then provide a warning
        // and give the user the ability to cancel and save their inputs.
        if (unsaved) {
            if (!unsaved.confirmUnsubmittedInputs()) {
                return;
            }
            unsaved.clearWatchForUnsavedChanges();
        }

        params = params || {};

        $.extend(params, {
            command: transition
        });

        if (util.isReviewPage() && $.inArray(transition, ['action:completeReview', 'action:summarizeReview', 'action:closeReview']) >= 0) {
            // If completing or summarizing we need to warn if the review has been updated.

            util.startAjaxDialogSpin();
            CRU.REVIEW.UTIL.reviewUpdatedAjax({
                done: function () {
                    var reviewUpdated = $('body').hasClass('review-updated');
                    if (reviewUpdated) {
                        CRU.REVIEW.UTIL.warnAboutReviewUpdates({reshowWarning: true});
                    }
                    util.stopAjaxDialogSpin();
                    return util.ajaxDialog(url, $.extend(params, {reviewUpdated: reviewUpdated}), true);
                }
            });
            return false;

        } else {
            return util.ajaxDialog(url, params, true);
        }
    };

    cruUtil.editDetailsFormChange = false;
    cruUtil.checkEditForm = function (done) {
        if (cruUtil.editDetailsFormChange) {
            CRU.REVIEW.UTIL.postEditDetailsForm(done);
        } else {
            if (done) {
                done();
            }
        }
        return false;//do a link action if called from <a>
    };

    cruUtil.command = function (cmd, pid, button) {
        var perma = pid || permaId;
        if (button) {
            button.disabled = true;
        }
        var donext = function () {
            var url = cruUtil.urlBase(perma);
            FECRU.XSRF.postUri(url + '/' + cmd);
        };
        //check and post the editDetailsForm if it has changed
        cruUtil.checkEditForm(donext);
    };


    cruUtil.createBlankReview = function (params) {
        var url = cruUtil.jsonUrlBase() + '/createReviewDialog';
        return cruUtil.ajaxDialog(url, params);
    };

    cruUtil.createSnippet = function (params) {
        var url = cruUtil.jsonUrlBase() + '/createSnippetDialog';
        return cruUtil.ajaxDialog(url, params);
    };

    cruUtil.addToReview = function (params) {
        var url = cruUtil.jsonUrlBase() + '/createDialog';
        return cruUtil.ajaxDialog(url, params);
    };

    cruUtil.isAnyDialogShowing = function () {
        return AJS && AJS.popup && typeof AJS.popup.current !== 'undefined' && AJS.popup.current !== null;
    };

    cruUtil.isAnyOverlayShowing = function () {
        return $('.review-overlay.review-overlay__open').length > 0;
    };

    cruUtil.isDimmed = function () {
        return !!AJS.dim.dim;
    };

    cruUtil.makeCssRule = function (selector, ruleBody) {
        var styleSheet = document.styleSheets[0];
        var index = 0;
        if (styleSheet.insertRule) {
            styleSheet.insertRule(selector + '{' + ruleBody + '}', index);
            return styleSheet.cssRules[index].style;
        } else {
            // Internet Explorer's version.
            styleSheet.addRule(selector, ruleBody, index);
            return styleSheet.rules[index].style;
        }
    };

    cruUtil.createIdeSrc = function (linkUrl, frxId) {
        var src = linkUrl + '&id=' + Math.floor(Math.random() * 1000);
        if (frxId) {
            src += '&line=' + CRU.REVIEW.UTIL.getTopVisibleLineNumber(frxId);
        }
        return src;
    };

    cruUtil.loadJiraIssueLink = function (issueKey, $target, params, done) {
        if (!issueKey) {
            throw "JIRA issue key required";
        }
        if (!$target) {
            throw "Target container to insert issue details into required";
        }

        var isReviewIssue = ($target.attr("data-review-jira-issue") === "true");
        var hideIssueTitle = ($target.attr("data-hide-issue-title") === "true");

        var defaults = {
            shouldGetIssueMetadata: isReviewIssue,
            hideIssueTitle: hideIssueTitle,
            maxTitleLength: 75,
            key: issueKey
        };

        var data = $.extend({}, defaults, params);

        var getIssueDetails = function () {
            AJS.$.ajax({
                url: FECRU.pageContext + '/json/action/issue-inline.do',
                data: data,
                type: "GET",
                dataType: "json",
                cache: false,
                success: function (resp) {
                    if (resp.foundIssue || (resp.credentialsRequired && resp.credentialsRequired.length)) {
                        // only display html if we found an issue (no errors)
                        $target.html(resp.html);
                    }

                    if (!$target.find(".ual-authenticate").length) {
                        $target.attr("data-jira-issue-lazy-load", "false");
                    }

                    $target.find(".ual-authenticate").bind("click", function () {
                        FECRU.OAUTH.getEventProducer($(this).attr("href")).authorized(function () {
                            FECRU.HOVER.invalidateCache(FECRU.HOVER.CACHE_FOREVER, issueKey);
                            CRU.UI.loadInlineJiraIssues();
                        });
                    });

                    if (data.shouldGetIssueMetadata) {
                        if (resp.canLogWork) {
                            AJS.$("#time-spent").addClass("submit-jira-time");
                            AJS.$("#linked-jira-log-work").show();
                        }
                        if (resp.subtasksConfigured && !resp.issueIsSubtask) {
                            AJS.$("body").addClass("jiraSubtasksVisible");
                        } else {
                            AJS.$("body").removeClass("jiraSubtasksVisible");
                        }
                    }

                    done && done($target);
                },
                error: function (jqXHR, textStatus, errorThrown) {
                    // dont display jira errors in the UI
                    $target.attr("data-jira-issue-lazy-load", "false");
                }
            });
        };

        getIssueDetails();
    };

    // this is a method to handle legacy "checkbox" style user pickers via autocomplete
    cruUtil.addUserCheckbox = function (id, user) {
        var userid = user.dbId;
        var $elTarget = AJS.$('#' + id + '_rc_' + userid);

        if (!$elTarget.length) {
            // checkbox doesnt exist yet
            $elTarget = AJS.$('<span><input type="checkbox"/><label></label></span>');
            $elTarget.attr('id', id + '_rc_' + userid);
            $elTarget.children('input')
                .attr('name', id)
                .attr('value', user.id);
            $elTarget.children('label')
                .attr('for', id + '_rc_' + userid)
                .text(user.displayPrimary);

            AJS.$('#' + id + '_checkboxes').append($elTarget);
        }
        $elTarget.children('input').attr('checked', 'checked');
        $elTarget.show();
    };

})(AJS.$);
/*[{!util_js_6z0350x!}]*/