/**
 * Range slider where the handles meet at a point (but don't overlap).
 */

CRU.WIDGETS = {};
(function () {
    function draggableRangeSlider(options) {

        var $slider = AJS.$(options.elem);
        var activeNub;
        var leftNub;
        var rightNub;

        var prevLeftValue;
        var prevRightValue;
        var prevLeftSlideValue;
        var prevRightSlideValue;

        function focused(nub) {
            return activeNub === nub;
        }

        /*
         * When mutating the slider values, don't cache the values of the slider
         * since mutating causes slider change events to be triggered.
         */
        function val(nub, value) {
            if (value !== undefined) {
                return $slider.slider('values', nub === leftNub ? 0 : 1, value);
            } else {
                return $slider.slider('values', nub === leftNub ? 0 : 1);
            }
        }

        $slider.slider({
            orientation: 'horizontal',
            range: true,
            min: options.min,
            max: options.max,
            values: options.values,

            start: function (event, ui) {
                activeNub = event.target || event.srcElement;
                if (options.fixedMin && focused(leftNub)) {
                    event.preventDefault();
                    event.stopPropagation();
                    return;
                }
                prevLeftValue = prevLeftSlideValue = val(leftNub);
                prevRightValue = prevRightSlideValue = val(rightNub);
            },

            slide: function (event, ui) {
                // Prevent the nubs from overlapping, effectively bumping them along.
                if (val(leftNub) === val(rightNub)) {
                    if (focused(leftNub)) {
                        val(rightNub, val(rightNub) + 1);
                    }
                    if (focused(rightNub)) {
                        val(leftNub, val(leftNub) - 1);
                    }
                } else if (val(rightNub) < options.STEP) {
                    if (options.fixedMin) {
                        event.preventDefault();
                        event.stopPropagation();
                        return;
                    }
                }
            },

            change: function (event, ui) {
                // Prevent the nubs from overlapping at each end.
                if (focused(leftNub) && val(leftNub) === options.max) {
                    val(leftNub, val(leftNub) - 1);
                } else if (focused(rightNub) && val(rightNub) === options.min) {
                    val(rightNub, val(rightNub) + 1);
                }

                if (!(val(leftNub) === prevLeftSlideValue && val(rightNub) === prevRightSlideValue)) {
                    prevLeftSlideValue = val(leftNub);
                    prevRightSlideValue = val(rightNub);
                    return options.slide(val(leftNub), val(rightNub));
                }
            },

            stop: function (event, ui) {
                if (!(val(leftNub) === prevLeftValue && val(rightNub) === prevRightValue)) {
                    return options.change(val(leftNub), val(rightNub));
                }
            }
        });

        var $leftNub = $slider.find('.ui-slider-handle:first');
        leftNub = $leftNub[0];
        rightNub = $slider.find('.ui-slider-handle:last')[0];

        if (options.fixedMin) {
            $leftNub.addClass('ui-slider-handle-locked')
        }
    }

    /**
     * Slider for a given set of datapoints.
     */
    CRU.WIDGETS.datapointSlider = (function () {

        // Must be odd so that there is no single median tickmark between adjacent datapoints.
        var STEP = 6 + 1;

        function tickmarks(datapoints) {
            var result = [];
            for (var i = 0, len = datapoints.length; i < len; i++) {
                result.push(STEP * i + 1);
            }
            return result;
        }

        function quantizeTick(tick) {
            return Math.round((tick - 1) / STEP) * STEP + 1;
        }

        function leftTick(val) {
            return val + 1;
        }

        function leftVal(tick) {
            return tick - 1;
        }

        function rightTick(val) {
            return val;
        }

        function rightVal(tick) {
            return tick;
        }

        function createSet() {
            var result = {};
            var i;
            var len;

            for (i = 0, len = arguments.length; i < len; i++) {
                if (arguments[i]) {
                    result[arguments[i]] = true;
                }
            }
            return result;
        }

        return function (settings) {
            var datapoints = settings.datapoints;
            var tick2dp = {};
            var dp2tick = {};
            (function () {
                var ticks = tickmarks(datapoints);
                for (var i = 0, len = ticks.length; i < len; i++) {
                    tick2dp[ticks[i]] = datapoints[i];
                    dp2tick[datapoints[i]] = ticks[i];
                }
            })();

            var selectedDPs = {
                start: settings.start || datapoints[0],
                end: settings.end || datapoints[datapoints.length - 1]
            };
            var activeDPs = createSet(selectedDPs.start, selectedDPs.end);

            draggableRangeSlider({
                elem: settings.elem,
                min: leftVal(1),
                max: rightVal((datapoints.length - 1) * STEP + 1),
                values: [leftVal(dp2tick[selectedDPs.start]), rightVal(dp2tick[selectedDPs.end])],
                fixedMin: settings.fixedMin,
                STEP: STEP,

                slide: function (leftVal, rightVal) {
                    var newDPs = createSet(tick2dp[quantizeTick(leftTick(leftVal))], tick2dp[quantizeTick(rightTick(rightVal))]);
                    var dp;

                    for (dp in activeDPs) {
                        if (!newDPs[dp]) {
                            settings.inactive(dp);
                        }
                    }
                    for (dp in newDPs) {
                        if (!activeDPs[dp]) {
                            settings.active(dp);
                        }
                    }

                    activeDPs = newDPs;
                },

                change: function (left, right) {
                    var start = tick2dp[quantizeTick(leftTick(left))];
                    var end = tick2dp[quantizeTick(rightTick(right))];

                    // Snap nubs to the exact corresponding slider value for each datapoint.
                    AJS.$(settings.elem).slider('values', 0, leftVal(dp2tick[start]));
                    AJS.$(settings.elem).slider('values', 1, rightVal(dp2tick[end]));

                    if (start && end && !(start === selectedDPs.start && end === selectedDPs.end)) {
                        selectedDPs.start = start;
                        selectedDPs.end = end;
                        return settings.change(selectedDPs.start, selectedDPs.end);
                    }
                }
            });

            (function () {
                for (var dp in activeDPs) {
                    if (activeDPs.hasOwnProperty(dp)) {
                        settings.active(dp);
                    }
                }
            })();

        };
    })();
})();
/*[{!sliders_js_sdt6522!}]*/