(function () {
    /**
     * Bind a callback that fires when scrolling is completed on matching elements.
     *
     * A scroll is considered complete after completionDelay ms have passed since
     * the last scroll event.
     */
    var setCompletedScrollTimeout = function ($object, callback, completionDelay) {
        completionDelay = completionDelay || 100;
        var timeout;

        $object.scroll(function () {
            if (completionDelay > 0 && timeout) {
                clearTimeout(timeout);
            }
            timeout = setTimeout(callback, completionDelay);
        });
    };

    /**
     * Bind a scroll tracker that monitors a scrollable container for matching
     * elements.
     *
     * After each completed scroll, if the currently active element is still in
     * the viewport/container, it remains active; if no longer in the viewport,
     * the top most element becomes active. If no tracked elements are in
     * the container, the active element prior to the completed scroll (if any)
     * remains active.
     *
     * The options.active and options.deactive callbacks are fired when the newly
     * calculated active element differs from the previously active element. The
     * options.outofview callback is called the first time the active element
     * becomes out of view (even partially).
     *
     * @param {jQuery selector|callback} options.selector expression/callback for elements to track
     * @param {jQuery selector} options.containerId id of scrollable container (or the window if unspecified) where the tracked elements live
     * @param {Number} options.threshold maximum number of px from the top/bottom of the container to start tracking elements
     * @param {Function} options.active callback where <tt>this</tt> is bound to the active element
     * @param {Function} options.deactive optional callback where <tt>this</tt> is bound to the previously active element
     * @param {Function} options.outofview optional callback where <tt>this</tt> is bound to the active element
     * @param {Number} options.delay optional delay (in ms) before a scroll is considered completed
     * @return {object} handle to the scroll tracker with methods to manipulate it's behaviour
     */
    CRU.WIDGETS.makeScrollTracker = function (options) {
        var trackingEnabled = true;
        var trackedElements;
        var currentElement;
        var hasBeenOutOfView = false;
        var ignoreNextScroll = false;
        var sticky = false;

        var $container;
        options.containerId = options.containerId || '';
        if (options.containerId) {
            $container = AJS.$('#' + options.containerId);
        } else {
            $container = AJS.$(window);
        }
        options.threshold = options.threshold || 0;

        function lookupElements() {
            trackedElements = AJS.$.isFunction(options.selector) ? options.selector() : AJS.$(options.selector);
        }

        function filterElements(threshold) {
            threshold = typeof threshold === 'number' ? threshold : calcThreshold();
            var result;
            trackedElements.each(function (i) {
                if (AJS.$(this).filter(':in-viewport-vert(' + threshold + ', ' + options.containerId + ')').length === 1) {
                    result = this;
                    return false;  // break
                }
            });
            return result;
        }

        function calcThreshold() {
            return Math.min($container.scrollTop(), options.threshold);
        }

        function topMostElement() {
            if (!trackedElements) {
                lookupElements();
            }
            return filterElements();
        }

        function updateCurrentElementAndFireEvents(newCurrentElement, shouldFireActivation) {
            if (currentElement && options.deactive) {
                options.deactive.call(currentElement);
            }
            currentElement = newCurrentElement;
            hasBeenOutOfView = false;
            if (currentElement && shouldFireActivation) {
                options.active.call(currentElement);
            }
        }

        function fireEventsIfOutOfView() {
            if (options.outofview && isOutOfView(currentElement)) {
                options.outofview.call(currentElement);
                hasBeenOutOfView = true;
            }
        }

        function isOutOfView(element) {
            var $element = AJS.$(element);
            var offset = $element.offset();

            return offset.top < $container.scrollTop() || $container.scrollTop() + $container.height() < offset.top + $element.height();
        }

        function mainloop() {
            if (!trackingEnabled) {
                return;
            }

            if (ignoreNextScroll) {
                ignoreNextScroll = false;
                return;
            }

            var topElement;
            if (!(currentElement && sticky && AJS.$(currentElement).is(':in-viewport-vert(' + calcThreshold() + ', ' + options.containerId + ')'))) {
                topElement = topMostElement();
                if (topElement && topElement !== currentElement) {
                    sticky = false;
                    updateCurrentElementAndFireEvents(topElement, true);
                }
            }
            if (currentElement && !hasBeenOutOfView) {
                fireEventsIfOutOfView();
            }
        }

        setCompletedScrollTimeout(
            $container,
            mainloop,
            options.delay
        );

        AJS.$(document).ready(function () {
            lookupElements();
        });

        return {

            /**
             * Ignore the next scroll event.
             *
             * Useful for when you're programatically scrolling the page and don't want
             * the scroll tracker to recalculate the active element after that scroll.
             *
             * A typical calling pattern might be:
             * <pre>
             *     scrollTracker.ignoreNextScroll();
             *     // programatically scroll to `elem`
             *     scrollTracker.setCurrentElement(elem);
             * </pre>
             *
             * @see setCurrentElement
             */
            ignoreNextScroll: function () {
                ignoreNextScroll = true;
            },

            /**
             * Set the currently active element.
             *
             * Useful after calling ignoreNextScroll to programatically scroll
             * to a specific element.
             *
             * If element is not equal to getCurrentElement() this:
             * - will fire options.deactive,
             * - depending on fireActivation, may or may not fire options.active,
             * - depending on whether the element is out of view, may fire
             *   options.outofview
             *
             * @param {DOMElement} element the element to set as the active one
             * @param {Boolean} fireActivation whether to fire options.active (if applicable) for element (defaults to false)
             *
             * @see ignoreNextScroll
             */
            setCurrentElement: function (element, fireActivation) {
                fireActivation = fireActivation || false;

                if (element !== currentElement) {
                    updateCurrentElementAndFireEvents(element, fireActivation);
                }
                fireEventsIfOutOfView();
            },

            /**
             * Return the currently active element or undefined if there is none.
             */
            getCurrentElement: function () {
                return currentElement;
            },

            /**
             * Reevaluate the set of tracked elements, using the originally
             * provided selector expression/callback.
             *
             * Useful when tracked elements are removed from- or added to the DOM,
             * or their visiblity changes.
             */
            rescan: function () {
                lookupElements();
                // Calculate the new current element.
                mainloop();
            },

            /**
             * Returns the top most element visible in the actual container.
             *
             * @param threshold optional margin for the 'viewport'
             */
            getAbsoluteTopElement: function (threshold) {
                return filterElements(threshold || 0);
            },

            /**
             * Make the current element sticky, so that as long as it is in the viewport
             * it remains the current element.
             */
            makeCurrentElementSticky: function () {
                sticky = true;
            },

            setTrackingEnabled: function (enabled, ignoreNext) {
                trackingEnabled = enabled;
                ignoreNext = ignoreNext || false;
                if (ignoreNext === true) {
                    this.ignoreNextScroll();
                }
            }
        };
    };
})();
/*[{!scroll_tracker_js_09xq521!}]*/