/**
 * @param {Number} id
 * @param {String} type
 * @class Comment
 */
function Comment(id, type) {
    this.m_id = id;
    this.m_type = type;
    this.m_domId = this.m_type + this.m_id;
    this.m_contentDomId = this.m_type.replace('reply', 'comment') + 'Content' + this.m_id;
    this.m_frx = undefined;
    this.m_review = undefined;
    this.m_replyTo = undefined;
    this.m_replies = [];
    this.m_status = undefined;
    this.m_message = undefined;
    this.m_messageAsHtml = undefined;
    this.m_defect = false;
    this.m_fromLineRange = undefined;
    this.m_toLineRange = undefined;
    this.m_draft = false;
    this.m_commentColor = undefined;
    this.m_gutterLine = undefined; // int
    this.m_fromRevId = undefined;
    this.m_toRevId = undefined;
    this.m_metrics = [];
    this.m_issueKey = undefined;
    this.m_issueStatus = undefined;
    this.m_date = undefined;
    this.m_author = undefined;
}

Comment.prototype.id = function () {
    return this.m_id;
};

/**
 * Returns the DOM ID of the comment's content and replies container.
 *
 * <p>
 * For source comments, this returns the inline version of the DOM ID.
 * You will need to regexp replace <code>inline</code> with <code>above</code>
 * to get the DOM ID for the above version of the comment.
 * </p>
 */
Comment.prototype.domId = function () {
    return this.m_domId; // if you update the HTML structure of comments, update visiblePosition()
};

/**
 * Returns the DOM ID of the comment's content container.
 *
 * <p>
 * For source comments, this returns the inline version of the DOM ID.
 * You will need to regexp replace <code>inline</code> with <code>above</code>
 * to get the DOM ID for the above version of the comment.
 * </p>
 *
 * <p>
 * Use this in favour of, e.g., <code>jQuery(comment.domId()).children('div.comment')</code>.
 * </p>
 */
Comment.prototype.contentDomId = function () {
    return this.m_contentDomId;
};

Comment.prototype.getDomElement = function () {
    return AJS.$('#' + this.domId());
};

Comment.prototype.getContentDomElement = function () {
    return AJS.$('#' + this.contentDomId());
};


// It is possible for the comment object to exist, but not be in the dom - eg when we autosave comments.
Comment.prototype.domIdExists = function () {
    return this.getDomElement().length > 0;
};

Comment.prototype.frx = function () {
    return this.m_frx;
};

Comment.prototype.setFrx = function (frx) {
    this.m_frx = frx;
    this.m_review = frx.review();
    return this;
};

Comment.prototype.review = function () {
    return this.m_review;
};

Comment.prototype.setReview = function (review) {
    this.m_review = review;
    return this;
};

Comment.prototype.type = function () {
    return this.m_type;
};

/**
 * the location of the comment: either general, revision or inline.
 */
Comment.prototype.position = function () {
    return this.isReply() ? this.m_type.replace(/reply$/, "") : this.m_type.replace(/comment$/, "");
};

/**
 * Returns the position of the comment which is visible.
 * @return either 'general', 'revision', 'inline' or 'above' depending on which comment is visible. If the comment
 * is hidden, an empty string is returned. If both inline and above comments are shown, the result is indeterminant.
 */
Comment.prototype.visiblePosition = function () {
    var position = this.position();
    var place = this.isReply() ? 'reply' : 'comment';
    if (AJS.$("#" + position + place + this.m_id).is(":visible")) {
        return position;
    } else if (this.isInline()) {
        position = 'above';
        if (AJS.$('#' + position + place + this.m_id).is(":visible")) {
            return position;
        }
    }
    return '';
};

Comment.prototype.visibleDomId = function () {
    if (this.isInline() && AJS.$('body').hasClass('show-above-comments')) {
        return this.m_domId.replace('inline', 'above');
    } else {
        return this.m_domId;
    }
};

Comment.prototype.status = function () {
    return this.m_status;
};

Comment.prototype.setStatus = function (status) {
    this.m_status = status;
    return this;
};

Comment.prototype.isReply = function () {
    return !!this.m_replyTo;
};

Comment.prototype.replyTo = function () {
    return this.m_replyTo;
};

Comment.prototype.setReplyTo = function (replyTo) {
    this.m_replyTo = replyTo;
    return this;
};

Comment.prototype.hasReplies = function () {
    return this.m_replies.length > 0;
};

Comment.prototype.addReply = function (comment) {
    var alreadyAdded = this.getReplies()
        .some(function (reply) {
            return reply.id() === comment.id();
        });
    if (!alreadyAdded) {
        this.m_replies.push(comment);
    }
    return this;
};

Comment.prototype.getReplies = function () {
    return this.m_replies;
};

Comment.prototype.clearReplies = function () { //recursive
    var replies_copy = this.m_replies.slice(); // use a copy because review.removeComment() modifies this.m_replies
    for (var i = 0, len = replies_copy.length; i < len; i++) {
        var reply = replies_copy[i];
        if (reply) {
            reply.clearReplies();
            review.removeComment(reply);
        }
    }
    this.m_replies = [];
};

Comment.prototype.isInline = function () {
    return /^inline/.test(this.m_type);
};

Comment.prototype.isHidden = function () {
    return this.isInline() && !this.fromLineRange() && !this.toLineRange();
};

Comment.prototype.message = function () {
    return this.m_message;
};

Comment.prototype.setMessage = function (message) {
    this.m_message = message;
    return this;
};

Comment.prototype.messageAsHtml = function () {
    return this.m_messageAsHtml;
};

Comment.prototype.setMessageAsHtml = function (message) {
    this.m_messageAsHtml = message;
    return this;
};

Comment.prototype.defect = function () {
    return this.m_defect;
};

Comment.prototype.setDefect = function (defect) {
    this.m_defect = defect;
    return this;
};

Comment.prototype.draft = function () {
    return this.m_draft;
};

Comment.prototype.setDraft = function (draft) {
    this.m_draft = draft;
    return this;
};

Comment.prototype.fromLineRange = function () {
    return this.m_replyTo ? this.m_replyTo.fromLineRange() : this.m_fromLineRange;
};

Comment.prototype.setFromLineRange = function (lr) {
    this.m_fromLineRange = lr;
    return this;
};

Comment.prototype.toLineRange = function () {
    return this.m_replyTo ? this.m_replyTo.toLineRange() : this.m_toLineRange;
};

Comment.prototype.setToLineRange = function (lr) {
    this.m_toLineRange = lr;
    return this;
};

Comment.prototype.fromRevId = function () {
    return this.isReply() ? this.m_replyTo.fromRevId() : this.m_fromRevId;
};

Comment.prototype.setFromRevId = function (id) {
    this.m_fromRevId = id;
    return this;
};

Comment.prototype.toRevId = function () {
    return this.m_replyTo ? this.m_replyTo.toRevId() : this.m_toRevId;
};

Comment.prototype.setToRevId = function (id) {
    this.m_toRevId = id;
    return this;
};

Comment.prototype.commentColor = function () {
    return this.m_commentColor;
};

Comment.prototype.setCommentColor = function (color) {
    this.m_commentColor = color;
    return this;
};

Comment.prototype.gutterLine = function () {
    return this.m_replyTo ? this.m_replyTo.gutterLine() : this.m_gutterLine;
};

Comment.prototype.setGutterLine = function (gutterLine) {
    this.m_gutterLine = gutterLine;
    return this;
};

Comment.prototype.metrics = function () {
    return this.m_metrics;
};

Comment.prototype.setMetrics = function (metrics) {
    this.m_metrics = metrics || [];
    return this;
};

Comment.prototype.issueKey = function () {
    return this.m_issueKey;
};

Comment.prototype.setIssueKey = function (issueKey) {
    this.m_issueKey = issueKey;
    return this;
};

Comment.prototype.issueStatus = function () {
    return this.m_issueStatus;
};

Comment.prototype.setIssueStatus = function (issueStatus) {
    this.m_issueStatus = issueStatus;
    return this;
};

/**
 * @param {Date} date
 * @return {Comment}
 */
Comment.prototype.setDate = function (date) {
    if (typeof date !== 'undefined' && !(date instanceof Date)) {
        throw new Error('Comment date must be an instance of Date');
    }
    this.m_date = date;
    return this;
};

/**
 * @return {(Date|undefined)}
 */
Comment.prototype.date = function () {
    return this.m_date;
};

/**
 * @return {(User|undefined)}
 */
Comment.prototype.author = function () {
    return this.m_author;
};

/**
 * @param {User} author
 * @return {Comment}
 */
Comment.prototype.setAuthor = function (author) {
    if (typeof author !== 'undefined' && !(author instanceof User)) {
        throw new Error('Author must be an instance of User');
    }
    this.m_author = author;
    return this;
};
/*[{!comment_js_yz0j51v!}]*/