window.CRU = window.CRU || {};
if (!CRU.UI) {
    CRU.UI = {};
}

(function ($, cssJs) {
    var cruUi = CRU.UI;
    var $window = $(window);
    var reviewCSSRules = cssJs('review-styles');

    /**
     * Bind textarea to trigger resize events
     *
     * @param {DOMElement} textarea
     *
     */
    var makeTextareaResizable = function (textarea) {
        var $textarea = $(textarea);
        var $document = $(document);

        textarea = $textarea.get(0);

        // it's impossible to resize textarea via mouse without support for resize property
        if (!('resize' in textarea.style)) {
            return;
        }

        var checkIfResized = $.proxy(function () {
            var previousWidth = this.dataset.lastWidth;
            var previousHeight = this.dataset.lastHeight;
            var currentWidth = $textarea.width().toFixed(0);
            var currentHeight = $textarea.height().toFixed(0);

            if (previousWidth !== currentWidth || previousHeight !== currentHeight) {
                this.dataset.lastWidth = currentWidth;
                this.dataset.lastHeight = currentHeight;
                $textarea.trigger('textarea-resize', +currentWidth, +currentHeight);
            }
        }, textarea);

        $textarea
            .on('mouseover', function () {
                this.dataset.lastWidth = $textarea.width().toFixed(0);
                this.dataset.lastHeight = $textarea.height().toFixed(0);
                $document.on('mousemove', checkIfResized);
            })
            .on('blur mouseup', function () {
                $document.off('mousemove', checkIfResized);
            });
    };

    var pageScrolling = (function () {
        var dimensions = {
            reviewToolbarHeight: 0,
            pageHeaderHeight: 0,
            headerHeight: 0,
            footerHeight: 0,
            reviewHeaderHeight: 77, // yep it's constant
            switchToStickyPoint: 0,
            bodyMinHeight: 0,
            messageHeight: 0,
            reviewSubHeaderButtons: 0,
            windowHeight: 0
        };

        var elements = {
            $sidebar: undefined,
            frxPane: undefined,
            $body: undefined,
            page: undefined,
            scrollable: undefined
        };

        var isSticky = false;
        var isScrollTriggeredByWindow = false;
        var isScrolling = false;

        var initStaticDimensions = function () {
            dimensions.reviewToolbarHeight = $('#review-info-container').height();
            dimensions.footerHeight = $('#footer').height();
            dimensions.pageHeaderHeight = $('.aui-page-header').outerHeight(true);
            dimensions.reviewHeaderHeight = $('.review-header').outerHeight(true);
            dimensions.headerHeight = $('#header').outerHeight(true);
            dimensions.messageHeight = $('#content > .aui-message').outerHeight(true);
            dimensions.reviewSubHeaderButtons = $('.review-header--buttons').height();
            dimensions.evalBar = $('.evalBar').height();
            dimensions.licenseStatusBanner = $('#stp-licenseStatus-banner').outerHeight(true);
            dimensions.switchToStickyPoint = dimensions.pageHeaderHeight + dimensions.headerHeight +
                dimensions.reviewHeaderHeight + dimensions.messageHeight - dimensions.reviewSubHeaderButtons +
                dimensions.evalBar + dimensions.licenseStatusBanner;

            $('body').data('switchToStickyPoint', dimensions.switchToStickyPoint);
        };

        var initDynamicDimensions = function () {
            // recount body height if review content height changes
            var contentHeightChangeEvents = [
                'comment-form-added', 'comment-form-removed', 'comment-added', 'comment-removed',
                'comment-collapsed', 'comment-expanded', 'comment-reply-added',
                'objectives-updated', 'objectives-form-visible', 'objectives-form-hidden',
                'summary-updated', 'summary-form-visible', 'summary-form-hidden',
                'review-details-updated', 'diff-mode-changed', 'view-range-change',
                'wiki-preview-on', 'wiki-preview-off',
                'tracked-branch:removed', 'tracked-branch:added'
            ].join(' ');
            var contentDimensionsChangeEvents = ['review-details-updated', 'editable-title-updated', 'comment-view-type-changed'].join(' ');
            var recountPageHeight = function () {
                if (!elements.scrollable) {
                    return;
                }

                var targetHeight = elements.scrollable.scrollHeight;
                if (elements.scrollable.id !== 'generalComments') {
                    targetHeight += dimensions.reviewHeaderHeight;
                }
                setBodyHeight(targetHeight);
            };

            var recountPageHeightWithThrottle = _.throttle(recountPageHeight, 200, {trailing: true});
            var recountPageHeightWithThrottleForWindowsResize = _.throttle(function () {
                dimensions.reviewHeaderHeight = $('.review-header').outerHeight(true);
                initStaticDimensions();
                recountPageHeight();
            }, 400, {trailing: true});

            var imageChangeEvents = ['frx-visible', 'comment-added', 'summary-updated', 'objectives-updated',
                'comment-reply-added'].join(' ');

            var loadImageWithThrottle = _.throttle(function () {
                $(elements.frxPane).find('img').each(function () {
                    var $t = $(this);
                    if (!$t.data('loaded')) {
                        $t.data('loaded', true);
                        $t.load(function () {
                            $(elements.frxPane).trigger('image-loaded');
                        })
                    }
                })
            }, 200, {trailing: true});

            var resizeReviewScrollableContentIfHeaderIsHigherThanNormal = function () {
                var headerHeight = $('.frxouter.activeFrx .frx-header-container').height();
                var $frxInner = $('.frxouter.activeFrx .frxinner');

                if (!$frxInner.length) {
                    return;
                }
                if (headerHeight > 80) { // if is higher than normal
                    var reviewContentHeight = $window.height() - dimensions.footerHeight
                        - dimensions.reviewToolbarHeight - dimensions.reviewSubHeaderButtons;
                    $frxInner.height(reviewContentHeight - headerHeight)
                        .data('hasCustomHeight', true);
                } else if ($frxInner.data('hasCustomHeight')) {
                    $frxInner.height('').data('hasCustomHeight', false);
                }
            };

            $(elements.frxPane)
                .on('frx-visible diff-mode-changed view-range-change current-frx-loaded', '.frxouter.activeFrx', function () {
                    if (this.id === 'generalComments') {
                        elements.scrollable = $('#generalCommentsInner').get(0);
                        return;
                    }
                    elements.scrollable = $(this).find('.frxinner').get(0);
                    CRU.COMMENT.reinitializeScrollTrackerForElement(elements.scrollable);
                    $(elements.scrollable).on('scroll', onScrollableElementScroll);
                })
                // setting body height immediately is very important for those events so they have higher priority
                .on('frx-visible current-frx-loaded', recountPageHeight)
                .on(contentHeightChangeEvents, '.frxouter.activeFrx', recountPageHeightWithThrottle)
                .on(imageChangeEvents, '.frxouter.activeFrx', loadImageWithThrottle)
                .on('image-loaded', recountPageHeightWithThrottle)
                .on(contentDimensionsChangeEvents, initStaticDimensions)
                .on(contentDimensionsChangeEvents, recountPageHeightWithThrottle)
                .on('comment-form-added', function (e) {
                    var $textarea = $(e.target).find('textarea');
                    makeTextareaResizable($textarea);
                    $textarea.on('textarea-resize', recountPageHeightWithThrottle);
                })
                .on('frx-header-unshelved view-range-change diff-mode-changed',
                resizeReviewScrollableContentIfHeaderIsHigherThanNormal);

            $window.on('resize', recountPageHeightWithThrottleForWindowsResize)
                .on('resize', _.throttle(resizeReviewScrollableContentIfHeaderIsHigherThanNormal, 100));


            setBodyHeight($window.height() - dimensions.reviewToolbarHeight - dimensions.footerHeight);

            // Fix the layout when STP banner added/removed
            AJS.bind('stp-licenseStatus-banner:created stp-licenseStatus-banner:destroyed', function () {
                initStaticDimensions();
                recountPageHeight();
            });

            $(document).on('aui-message-close', function (event, messageElement) {
                // analytics plugin does not deliver information about message element
                // so we use this fact to detect that analytics acknowlegment message has been closed
                if (typeof messageElement === 'undefined') {
                    initStaticDimensions();
                    recountPageHeight();
                }
            });
        };

        var initScroll = function () {
            $window.on('scroll', _.throttle(onScroll, 20, {trailing: true}));
        };

        var setBodyHeight = function (height) {
            dimensions.windowHeight = $window.height();
            dimensions.bodyMinHeight = dimensions.switchToStickyPoint + dimensions.windowHeight;
            var targetHeight = height + dimensions.switchToStickyPoint +
                dimensions.reviewToolbarHeight + dimensions.footerHeight + dimensions.reviewSubHeaderButtons;
            elements.$body.css({
                height: targetHeight < dimensions.bodyMinHeight ? dimensions.bodyMinHeight : targetHeight
            });
        };

        var onScroll = function () {
            triggerScrollStart();
            var windowScrollTop = $window.scrollTop();
            if (windowScrollTop >= dimensions.switchToStickyPoint) {
                if (!isSticky) {
                    isSticky = true;
                    elements.$body.trigger('sticky-headers-change', true);
                    movePage(dimensions.switchToStickyPoint);
                }
                setScrollableElementScrollPositionFromWindowScrollTop(windowScrollTop);
            } else {
                movePage(windowScrollTop);
                if (isSticky) {
                    isSticky = false;
                    elements.$body.trigger('sticky-headers-change', false);
                    setScrollableElementScrollPositionFromWindowScrollTop(windowScrollTop);
                }
            }
            triggerScrollEnd();
        };

        var setScrollableElementScrollPositionFromWindowScrollTop = function (windowScrollTop) {
            if (elements.scrollable) {
                isScrollTriggeredByWindow = true;
                if (windowScrollTop >= dimensions.switchToStickyPoint) {
                    elements.scrollable.scrollTop = windowScrollTop - dimensions.switchToStickyPoint;
                } else {
                    elements.scrollable.scrollTop = 0;
                }
                turnOffWindowScrollingFlag();
            }
        };

        var triggerScrollStart = function () {
            if (!isScrolling) {
                $window.trigger('scrollstart');
                isScrolling = true;
            }
        };

        var triggerScrollEnd = _.debounce(function () {
            if (isScrolling) {
                $window.trigger('scrollend');
                isScrolling = false;
            }
        }, 200);

        var turnOffWindowScrollingFlag = _.debounce(function () {
            isScrollTriggeredByWindow = false;
        }, 200);

        var onScrollableElementScroll = function (e) {
            if (isScrollTriggeredByWindow) {
                return;
            }
            var $target = $(e.target);
            var targetScrollTop = e.target.scrollTop;

            if ($target.data('lastScrollTop') === targetScrollTop) {
                // that means user scrolls horizontally so we don't care
                return;
            }
            $target.data('lastScrollTop', targetScrollTop);
            exports.scrollToPosition(targetScrollTop);
        };

        var movePage = function (position) {
            elements.page.scrollTop = position;
        };

        /**
         * Handle hiding of .frxSlider element during scrolling to improve IE9 rendering performance
         */
        var speedUpIE9 = function () {
            $window
                .on('scrollstart', function () {
                    $('.frxSlider').addClass('hidden');
                })
                .on('scrollend', function () {
                    $('.frxSlider').removeClass('hidden');
                });
        };

        var exports = {
            init: function () {
                elements.$sidebar = $('#content-resizable');
                elements.frxPane = document.getElementById('frx-pane');
                elements.$body = $(document.body);
                elements.page = document.getElementById('page');
                initStaticDimensions();
                initDynamicDimensions();
                initScroll();

                if ($.browser.msie && $.browser.version === '9.0') {
                    speedUpIE9();
                }
            },
            scrollToPosition: function (positionTop) {
                $window.scrollTop(positionTop + dimensions.switchToStickyPoint);
            },
            ensureElementVisible: function (element, options) {
                options = options || {};
                var $element = $(element);
                element = $element[0];//we handle element as string or DOM
                var distanceFromScrollTop = element.getBoundingClientRect().top - elements.scrollable.getBoundingClientRect().top;
                var elementHeight = $element.height();
                if (distanceFromScrollTop < 0 || elementHeight >= elements.scrollable.clientHeight) {
                    this.scrollToElement(element, options);
                }
                else if (distanceFromScrollTop + elementHeight > elements.scrollable.clientHeight) {
                    options.offset = elements.scrollable.clientHeight - elementHeight - options.offset;
                    this.scrollToElement(element, options);
                }
                else {
                    $element.trigger('scroll-to-element:finished');
                    if (options.onAfter) {
                        options.onAfter();
                    }
                }

            },
            scrollToElement: function (element, options) {
                // We have to set position of scrollable element properly
                // since it might not be synchronized with window scroll YET
                // Example case: switching between frxes
                setScrollableElementScrollPositionFromWindowScrollTop($window.scrollTop());
                options = options || {};
                element = $(element)[0];
                var scrollOffset = options && options.offset || 0;

                var elementTopInsideScrollable = element.getBoundingClientRect().top - elements.scrollable.getBoundingClientRect().top + elements.scrollable.scrollTop;
                var scrollToPosition = elementTopInsideScrollable + dimensions.switchToStickyPoint - scrollOffset;
                if (scrollToPosition < dimensions.switchToStickyPoint) {
                    scrollToPosition = dimensions.switchToStickyPoint;
                }

                var triggerScrollToElementFinished = function () {
                    $(element).trigger('scroll-to-element:finished');
                };
                if (options && options.initialScroll) {
                    // this has to be done to prevent browser to set saved scroll position before refresh
                    if (document.location.hash) {
                        $window.one('scroll', function (e) {
                            e.stopImmediatePropagation();
                            $window.scrollTop(scrollToPosition);
                        });
                    } else {
                        $window.scrollTop(scrollToPosition);
                        triggerScrollToElementFinished();
                    }
                } else {
                    var jQueryScrollToOptions = {
                        onAfter: FECRU.sequence(options.onAfter, triggerScrollToElementFinished)
                    };
                    //noinspection JSCheckFunctionSignatures Idea thinks that this is window.scrollTo(x,y)
                    $window.stop(true).scrollTo(scrollToPosition, jQueryScrollToOptions);
                }
            },
            getContentScrollTop: function () {
                var position = $window.scrollTop() - dimensions.switchToStickyPoint;
                return position < 0 ? 0 : position;
            }
        };
        return exports;
    })();

    CRU.UI.scrollReviewContentTo = pageScrolling.scrollToPosition;
    CRU.UI.ensureElementVisible = pageScrolling.ensureElementVisible;
    CRU.UI.scrollToElement = _.debounce(pageScrolling.scrollToElement, 100);
    CRU.UI.getReviewContentScrollTopPosition = pageScrolling.getContentScrollTop;

    cruUi.columnFillHeight = function () {
        var foot = $('#footer').outerHeight();
        //window - footer - subheaders buttons - roundingerrorhack
        var windowHeight = $(window).height() - foot - 1.3 - $('.review-header--buttons').height();
        var contentHeight = windowHeight + 1; // TODO where does this +1 come from
        var toolbarHeight = $('#review-navigation-top').outerHeight(true);
        var targetHeight = windowHeight - toolbarHeight;

        $("#content-shield").css({
            height: contentHeight
        });

        $("#content-navigation-panel, #frx-pane").css({
            height: targetHeight
        });

        cruUi.frxFillPane(targetHeight);
        reviewCSSRules.set('.frxinner', {
            height: (targetHeight - 77) + 'px'
        });
    };

    cruUi.initColumnResize = function (min) {
        $("#content-resizable").resizable({
            ghost: true,
            handles: "e",
            maxWidth: 600,
            minWidth: min || 310
        });

        var selector = "#content-resizable .ui-resizable-handle, #content-shield, .shielded";
        $(document).delegate(selector, "mousedown", function () {
            $("html").addClass("shielded");
        }).delegate(selector, "mouseup", function () {
            $("html").removeClass("shielded");
        });
    };

    cruUi.toggleTree = function () {
        var $pane = $("#content-resizable");
        if ($pane.is(".collapsed")) {
            $pane.width($pane.data("expandedWidth"))
                .removeClass("collapsed")
                .removeData("expandedWidth");
        } else {
            $pane.data("expandedWidth", $pane.width())
                .width("25px")
                .addClass("collapsed");
        }
        $(window).trigger('panes-resized');
    };

    cruUi.highlightElements = function ($elem, type) {
        var setBackgroundImage = function (value) {
            $elem.css('background-color', value);
        };
        var doHighlight = function (color, duration, delay) {
            if (delay) {
                setBackgroundImage(color);
                doHighlight(color, delay);
            }
            $elem.effect('highlight', {color: color}, duration, function () {
                setBackgroundImage('');
            });
        };

        switch (type) {
            case 'with-delay':
                doHighlight('#fffddd', 500, 2000);
                break;
            default:
                doHighlight('#fffee8', 1000);
                break;
        }
    };

    cruUi.frxFillPane = function (pane) {
        var isCompliant = !!$.support.opacity;// w3c or IE
        var full = (isCompliant ? pane : (pane - 1)) + "px";
        var half = (pane * 0.5) + "px";

        reviewCSSRules.set('#frxs .frxouter:last-child .frxbackground', {
            marginBottom: half
        });

        reviewCSSRules.set('#generalCommentsInner', {
            height: full
        });
    };

    // list of all issues that we have loaded so far
    var $loadedIssues = $();

    cruUi.loadInlineJiraIssues = function (issueKey, callback) {
        var $linkedIssues = $(".jira-issue-lazy-load").filter("[data-jira-issue-lazy-load='true']");
        if (issueKey) {
            $linkedIssues = $linkedIssues.filter("[data-jira-key='" + issueKey + "']");
        }
        $loadedIssues = $loadedIssues.add($linkedIssues);
        $linkedIssues.each(function (index) {
            var $issue = $(this);
            var jiraIssueKey = $issue.data("jira-key");
            var cruProjKey = $issue.data("cru-project-key");
            if (jiraIssueKey) {
                var params = {};
                if (cruProjKey) {
                    params.projectKey = cruProjKey;
                }
                CRU.UTIL.loadJiraIssueLink(jiraIssueKey, $issue, params, callback);
            }
        });
    };

    cruUi.reloadInlineJiraIssues = function (issueKeys, callback) {
        $loadedIssues.each(function (index) {
            var $issue = $(this);
            var jiraIssueKey = $issue.data("jira-key");
            var cruProjKey = $issue.data("cru-project-key");
            if (jiraIssueKey && (!issueKeys || issueKeys.indexOf(jiraIssueKey) !== -1)) {
                var params = {};
                if (cruProjKey) {
                    params.projectKey = cruProjKey;
                }
                CRU.UTIL.loadJiraIssueLink(jiraIssueKey, $issue, params, callback);
            }
        });
    };

    $(document).ready(function () {
        if ($("#content").length === 0) {
            // IT'S A TRAP! (might be an iframe that we're loading from)
            return;
        }
        // Give a warning if firebug is running
        // NB: do this before columnFillHeight() because this function may insert content into the page
        FECRU.UI.warnAboutFirebug(function () {
            cruUi.columnFillHeight();
        });

        // When resizing the window, resize the panes
        var resizeDelay = 200;
        FECRU.UI.setCompletedResizeTimeout(window, function () {
            cruUi.columnFillHeight();
            $window.trigger('panes-resized');
        }, resizeDelay);

        $("#content-resizable").bind('resizestop', function () {
            $window.trigger('panes-resized');
        });


        // And make sure the page is setup appropriately on first load
        cruUi.initColumnResize();
        cruUi.columnFillHeight();
        cruUi.loadInlineJiraIssues();
        pageScrolling.init();
    });
})(AJS.$, cssJs);
/*[{!crucible_ui_js_lxtg50s!}]*/