define('FECRU/component/blame/trigger-view', [
    'backbone',
    'global-ns/fecru-ajax',
    'global-ns/fecru-event-bus'
], function (Backbone, FeCruAjax, eventBus) {

    /**
     * Blame button will be shared between all source screens.
     * At the same time we display only one source.
     * That's why we have to properly initialize and unload the view to avoid conflicts.
     * Once the source screen is changed we have to abort data fetching if any.
     */
    return Backbone.View.extend({

        el: '.blame-trigger-wrapper',
        events: {
            'click .blame-trigger': 'clickHandler'
        },

        attachListeners: function () {
            this.listenTo(this.model, 'sync', this.onSync);
            this.listenTo(this.model, 'error', this.onError);
            this.listenTo(this.model, 'change:enabled', this.onEnabledChange);
            this.listenTo(this.model, 'change:fetching', this.onFetchingChange);
            this.listenTo(this.model, 'change:data', this.onDataChange);
        },

        /**
         * This method executes every time the source screen is set.
         * Listen to ui/custom events.
         * Enable blame button.
         * Revert button's state according to the model.
         */
        init: function () {
            var isEnabled = this.model.getEnabled();

            this.attachListeners();
            this.delegateEvents();
            this.toggleTriggerDisabledState(false);
            this.toggleTriggerPressedState(isEnabled);

            // Trigger "ready" event after some timeout.
            // Sometimes view can be loaded/unloaded few times (on Review page).
            if (!isEnabled) {
                this.analyticsReadyTimeout = setTimeout(eventBus.trigger.bind(eventBus, 'BLAME_ANALYTICS:READY'), 100);
            }
        },

        /**
         * This method executes every time the source screen is unset.
         * Abort fetching and clear all listeners.
         * Revert the view to default state without changing model's fields, they should be saved.
         */
        reset: function () {
            this.model.abort();
            this.undelegateEvents();
            this.stopListening();
            this.toggleTriggerDisabledState(true);
            this.toggleTriggerPressedState(false);
            this.toggleLoadingSpinner(false);

            // Clear "ready" event execution, the view might be unloaded immediately.
            clearTimeout(this.analyticsReadyTimeout);
        },

        clickHandler: function () {
            var model = this.model;
            var isFetching = model.getFetching();

            model.toggleEnabled();

            if (model.getEnabled()) {
                if (model.isFetched()) {
                    eventBus.trigger('BLAME_ANALYTICS:SHOWN_CACHED');
                } else {
                    eventBus.trigger('BLAME_ANALYTICS:REQUEST');
                }
            } else {
                isFetching || eventBus.trigger('BLAME_ANALYTICS:HIDE');
            }
        },

        /**
         * Handle the error.
         * The trick here is that FeCruAjax.ajaxFailure doesn't handle the case when responseText
         *   is empty (e.g. when status = 0, server offline or Internet connection is down).
         *   To handle it we parse the response an if it's not provided we show specific message then.
         * @param model
         * @param xhr
         * @param options
         */
        onError: function (model, xhr, options) {
            this.model.setEnabled(false);
            if (xhr.statusText !== 'abort') {
                eventBus.trigger('BLAME_ANALYTICS:FETCH_FAILED', options.took, xhr.status);

                var json;
                try {
                    json = JSON.parse(xhr.responseText);
                }
                catch (e) {}
                json ?
                    FeCruAjax.ajaxFailure(xhr) :
                    FeCruAjax.showServerUnreachableBox();
            } else {
                eventBus.trigger('BLAME_ANALYTICS:FETCH_ABORTED', options.took);
            }
        },

        onSync: function (model, response, options) {
            eventBus.trigger('BLAME_ANALYTICS:SHOWN_FETCHED', options.took);
        },

        onEnabledChange: function (model, state) {
            this.toggleTriggerPressedState(state);
            if (state) {
                this.showBlameInfo();
                this.model.isFetched() || this.model.fetch();
            }
            else {
                this.hideBlameInfo();
                this.model.abort();
            }
        },

        onFetchingChange: function (model, state) {
            this.toggleLoadingSpinner(state);
        },

        onDataChange: function () {
            this.renderBlameInfo();
        },

        getTrigger: function () {
            return this.$('.blame-trigger');
        },

        getTriggerSpinner: function () {
            return this.$('.blame-trigger-spinner');
        },

        toggleLoadingSpinner: function (state) {
            state ?
                this.getTriggerSpinner().spin() :
                this.getTriggerSpinner().spinStop();
        },

        toggleTriggerDisabledState: function (state) {
            this.getTrigger()
                .attr('aria-disabled', state)
                .prop('disabled', state);
        },

        toggleTriggerPressedState: function (state) {
            this.getTrigger().attr('aria-pressed', state);
        },

        showBlameInfo: function () {
            eventBus.trigger('SHOW_BLAME_AREA');
        },

        hideBlameInfo: function () {
            eventBus.trigger('HIDE_BLAME_AREA');
        },

        renderBlameInfo: function () {
            eventBus.trigger('RENDER_BLAME_INFO');
        }

    });

});
