define('FECRU/component/blame/renderers/table-renderer-view', [
    'jquery',
    'underscore',
    'FECRU/component/blame/renderers/abstract-renderer-view',
    'FECRU/component/blame/templates'
], function ($, _, AbstractRendererView, templates) {

    var AUTHOR_SELECTOR = '.author';
    var REVISION_SELECTOR = '.revision';

    return AbstractRendererView.extend({

        /**
         * @param tr {Node}
         * @returns {boolean} Whether blame info was rendered or not
         */
        renderUnifiedBlameLine: function (tr) {
            var $tr = $(tr);
            var lineNumbers = this.getLinesNumbers($tr);
            var key = lineNumbers.to ? 'to' : 'from';
            var $author = $tr.children(AUTHOR_SELECTOR);
            var $revision = $tr.children(REVISION_SELECTOR);

            return renderBlameContent.call(this, key, lineNumbers, $author, $revision);
        },

        /**
         * @param tr {Node}
         * @returns {boolean} Whether blame info was rendered or not
         */
        renderSideBySideBlameLine: function (tr) {
            var $tr = $(tr);
            var lineNumbers = this.getLinesNumbers($tr);
            var isRendered = false;

            var $authorFrom = $tr.children(AUTHOR_SELECTOR + ':eq(0)');
            var $revisionFrom = $tr.children(REVISION_SELECTOR + ':eq(0)');
            isRendered = renderBlameContent.call(this, 'from', lineNumbers, $authorFrom, $revisionFrom) || isRendered;

            var $authorTo = $tr.children(AUTHOR_SELECTOR + ':eq(1)');
            var $revisionTo = $tr.children(REVISION_SELECTOR + ':eq(1)');
            isRendered = renderBlameContent.call(this, 'to', lineNumbers, $authorTo, $revisionTo) || isRendered;

            return isRendered;
        },

        /**
         * @override
         * @see AbstractRendererView.prototype.collectMandatoryLines
         */
        collectMandatoryLines: function () {
            var $allSourceLines = this.getAllSourceLines();
            var $allSkippedLines = this.getAllSkippedLines();
            var $firstLine = $allSourceLines.eq(0);
            var trs = [
                $firstLine
            ];

            var isPreviousWasDiff = false;
            var isPreviousWasDiffTo = false;
            $allSourceLines.each(function (index, tr) {
                var $tr = $(tr);
                var isDiff = $tr.hasClass('is-diff');
                var isDiffTo = isDiff ? $tr.get(0).hasAttribute('data-to') : false;

                if (isDiff) {
                    if (isPreviousWasDiff) {
                        // Issue: https://jira.atlassian.com/browse/FECRU-6380
                        // This fallback is needed to handle "Whitespace" preferences.
                        // Make the line mandatory every time the diff block type changes.
                        if (isDiffTo !== isPreviousWasDiffTo) {
                            trs.push($tr);
                        }
                    }
                    // Only the first line of the diff block is mandatory
                    else {
                        trs.push($tr);
                    }
                } else if (isPreviousWasDiff) {
                    trs.push($tr);
                }
                isPreviousWasDiff = isDiff;
                isPreviousWasDiffTo = isDiffTo;
            });

            // First source line after context break ("skipped" tr) is mandatory
            var getNextSourceLine = this.getNextSourceLine.bind(this);
            $allSkippedLines.each(function (index, tr) {
                var sourceLine = getNextSourceLine(tr);
                if (sourceLine) {
                    trs.push(sourceLine);
                }
            });

            var lines = trs.map(this.getLinesNumbers, this);
            this.mandatoryLines = {
                to: _.compact(_.pluck(lines, 'to')),
                from: _.compact(_.pluck(lines, 'from'))
            };
        },

        /**
         * @param tr {Node|jQuery}
         * @returns {?jQuery} Next source line if any
         */
        getNextSourceLine: function (tr) {
            var $tr = $(tr);
            while (($tr = $tr.next()).length) {
                if ($tr.hasClass('sourceLine')) {
                    return $tr;
                }
            }
            return null;
        }
    });

    /**
     * @param key {string}
     * @param lineNumbers {Object}
     * @param $author {jQuery}
     * @param $revision {jQuery}
     * @returns {boolean} Whether blame info was rendered or not
     */
    function renderBlameContent(key, lineNumbers, $author, $revision) {
        var lineNumber = lineNumbers[key];
        var mandatoryLines = this.mandatoryLines[key];
        var details = key === 'to' ?
            this.getChangesetDetailsForLineTo(lineNumber) :
            this.getChangesetDetailsForLineFrom(lineNumber);

        if (details == null) {
            return false;
        }

        var isMandatory = function (lineNumber) {
            return mandatoryLines.indexOf(lineNumber) !== -1;
        };

        if (lineNumber === details.found[key] || isMandatory(lineNumber)) {
            var metadata = this.model.getMetadata();
            var options = {
                changesetId: details.found.changeset,
                metadata: metadata,
                details: details
            };
            $author.html(templates.authorCell(options));
            $revision.html(templates.revisionCell(options));
            return true;
        }

        return false;
    }

});
