/**
 * Model that stores the state of blame component.
 * It contains only 3 attributes:
 *   1) `enabled` - is "blame" button enabled.
 *   2) `fetching` - is blame info fetching from the server at the moment.
 *   3) `data` - blame info itself.
 *
 * Some notes:
 * Potentially for `enabled` attribute we can create another model but then it would add some extra
 *   complexity because of theirs synchronization with "blame" button (by default view accepts only one model,
 *   it is possible to use two, but it's not cool in this simple case).
 */

define('FECRU/component/blame/model', [
    'jquery',
    'underscore',
    'backbone'
], function ($, _, Backbone) {

    var legendMemo = null;
    var changesetMemo = null;
    var resetMemos = function () {
        legendMemo = {};
        changesetMemo = {};
    };

    return Backbone.Model.extend({

        defaults: {
            enabled: false,
            fetching: false,
            data: null
        },

        initialize: function () {
            this.listenTo(this, 'sync error', this.onFetchEnd);
            this.listenTo(this, 'change:data', resetMemos);
        },

        /**
         * @override
         * @param responses {Array<Object>} list of responses
         * @returns {Object}
         */
        parse: function (responses) {
            var from = responses[0] || {legend: []};
            var to = responses[1] || {legend: []};
            var data = {
                authors: _.defaults({}, to.authors, from.authors),
                changesets: _.defaults({}, to.changesets, from.changesets),
                legend: to.legend.concat(from.legend),
                meta: _.defaults({}, to.meta, from.meta)
            };

            return {
                data: data
            };
        },

        /**
         * @override
         */
        fetch: function () {
            var links = this.url;
            var model = this;
            var startedAt = performance.now();

            var fetch = function (url) {
                return $.ajax({
                    url: url,
                    type: 'get',
                    contentType: 'application/json'
                });
            };

            var normalizeTo = function (response) {
                return _.extend({}, response, {
                    legend: response.legend.map(function (item) {
                        return _.extend(_.omit(item, 'from'), {
                            to: item.from
                        });
                    })
                });
            };

            var getOptions = function () {
                return {
                    took: performance.now() - startedAt
                }
            };

            this.__requests = [
                links.from ? fetch(links.from) : null,
                links.to ? fetch(links.to) : null
            ];

            $.when.apply($, this.__requests)
                .then(function (from, to) {
                    var response = [];
                    if (from) {
                        response.push(from[0]);
                    }
                    if (to) {
                        response.push(normalizeTo(to[0]));
                    }
                    model.set(model.parse(response));
                    model.trigger('sync', model, response, getOptions());
                })
                .fail(function (xhr) {
                    model.trigger('error', model, xhr, getOptions());
                });

            this.setFetching(true);
        },

        abort: function () {
            this.setFetching(false);
            try {
                _.invoke(_.compact(this.__requests), 'abort');
            }
            catch (e) {}
        },

        onFetchEnd: function () {
            this.setFetching(false);
        },

        /**
         * @param state {boolean}
         */
        setEnabled: function (state) {
            this.set('enabled', state);
        },

        toggleEnabled: function () {
            this.set('enabled', !this.getEnabled());
        },

        /**
         * @returns {boolean}
         */
        getEnabled: function () {
            return this.get('enabled');
        },

        /**
         * @param state {boolean}
         */
        setFetching: function (state) {
            this.set('fetching', state);
        },

        /**
         * @returns {boolean}
         */
        getFetching: function () {
            return this.get('fetching');
        },

        /**
         * @param value {Object}
         */
        setData: function (value) {
            this.set('data', value);
        },

        /**
         * @returns {Object}
         */
        getData: function () {
            return this.get('data');
        },

        /**
         * @returns {?Array<Object>} Legend if fetched
         */
        getLegend: function () {
            return this.isFetched() && this.getData().legend || null;
        },

        /**
         * @returns {?Object} Changesets if fetched
         */
        getChangesets: function () {
            return this.isFetched() && this.getData().changesets || null;
        },

        /**
         * @returns {?Object} Authors if fetched
         */
        getAuthors: function () {
            return this.isFetched() && this.getData().authors || null;
        },

        /**
         * @returns {?Object} Metadata if fetched
         */
        getMetadata: function () {
            return this.isFetched() && this.getData().meta || null;
        },

        /**
         * Get changeset details by its id.
         * @param changeset {string} id
         * @returns {?Object} changeset if any
         */
        getChangesetDetails: function (changeset) {
            if (this.isFetched() === false) {
                return null;
            }

            var cachedValue = changesetMemo[changeset];
            if (cachedValue) {
                return cachedValue;
            }

            var details = this.getChangesets()[changeset];
            if (details) {
                return changesetMemo[changeset] = _.extend({}, details, {
                    author: this.getAuthors()[details.author] || null
                });
            }
            return null;
        },

        /**
         * Get changeset details for passed line.
         * In case there is no legend for the line (it means that the line is not the first one
         *   of the chunk where a change has been made) we have to grab the info from the closest one.
         * This method is called per each line so it has to be as fast as possible that's why
         *   we have to use slow third-party libraries' methods as less as possible.
         * To speed up this process we will map each line to the legend and provide a function
         *   to retrieve that legend by line's number. This operation will be performed just once (per key).
         * @param line {number}
         * @param key {string} line key - from or to.
         * @returns {?Object} changeset if any
         */
        getChangesetDetailsForLine: function (line, key) {
            if (this.isFetched() === false) {
                return null;
            }

            // Performs just once (per key)
            if (legendMemo[key] === void 0) {
                var legend = this.getLegend();
                var knownLines = [];
                var lineToLegendMap = {};

                // Pick all known lines from the legend, map them together.
                for (var legendIndex = 0; legendIndex < legend.length; legendIndex++) {
                    var lineNumber = legend[legendIndex][key];
                    if (lineNumber !== void 0) {
                        knownLines.push(lineNumber);
                        lineToLegendMap[lineNumber] = legend[legendIndex];
                    }
                }

                if (knownLines.length === 0) {
                    legendMemo[key] = function () {};
                    return null;
                }

                knownLines = _.sortBy(knownLines, Number);

                var maxKnownLine = _.max(knownLines);
                var linesMemo = new Array(maxKnownLine);
                var currentKnownLine = knownLines.shift();

                // Map each line [1, maxKnownLine] to the legend.
                // In case there is no legend for current line - take it from the current known line.
                for (var lineIndex = 1; lineIndex <= maxKnownLine; lineIndex++) {
                    if (lineIndex >= knownLines[0]) {
                        currentKnownLine = knownLines.shift();
                    }
                    linesMemo[lineIndex] = lineToLegendMap[currentKnownLine];
                }

                /**
                 * Get legen by line number.
                 * @param lineNumber {number}
                 * @returns {?Object}
                 */
                legendMemo[key] = function (lineNumber) {
                    // We do not know how many lines the source code has,
                    // so the maximum known line might be not the last one.
                    if (lineNumber > maxKnownLine) {
                        return linesMemo[linesMemo.length - 1];
                    }
                    else {
                        return linesMemo[lineNumber] || null;
                    }
                };
            }

            var found = legendMemo[key](line);
            if (found) {
                return _.extend({}, this.getChangesetDetails(found.changeset), {
                    found: found
                });
            }

            return null;
        },

        /**
         * @returns {boolean} Is data fetched
         */
        isFetched: function () {
            return this.getData() !== null;
        }

    });

});
