/**
 * An entry point of blame component.
 * In exposes an `init` method which has to be called to initialize the component.
 */

define('FECRU/component/blame', [
    'jquery',
    'underscore',
    'backbone',
    'global-ns/fecru-event-bus',
    'FECRU/component/blame/urls-manager',
    'FECRU/component/blame/blame-instance',
    'FECRU/component/blame/blame-instances-manager',
    'FECRU/component/blame/blame-analytics'
], function ($, _, Backbone, eventBus, urlsManager, BlameInstance, BlameInstancesManager, BlameAnalytics) {

    var blameInstancesManager = new BlameInstancesManager();
    var analytics = new BlameAnalytics();

    /**
     * Reset the state of the current instance if any and init requested one.
     * Initialize the instance only in case it can fetch blame info, it make no sense to enable blame button otherwise.
     * But in the same time we want to have an instance to be saved as a current one so once url will be
     *   registered we can "reload" the source.
     * @param instanceKey {?string}
     * @param [page] {string} Page identificator
     * @param [$source] {jQuery} Source code element
     */
    function switchTheSource(instanceKey, page, $source) {
        var currentInstance = blameInstancesManager.getCurrentInstance();
        if (currentInstance) {
            currentInstance.reset();
            analytics.unlisten();
        }

        var instance = blameInstancesManager.ensureInstance(instanceKey, page, $source);
        if (instance) {
            blameInstancesManager.switchToInstance(instance);
            if (instance.canFetchBlameInfo()) {
                analytics.listen(page);
                instance.init();
            }
        }
    }

    function onSourceShown(options) {
        switchTheSource(options.key, options.page, options.$source);
    }

    function onSourceHidden() {
        switchTheSource(null);
    }

    /**
     * Handle `source-code:reset` event.
     * It applies to any instance.
     * If the instance we have to reset is the current one - unload it first.
     * Remove the instance so the next time the source will be loaded we will create a brand new one.
     * @param options {Object}
     */
    function onSourceReset(options) {
        var key = options.key;
        if (blameInstancesManager.isCurrentInstance(key)) {
            switchTheSource(null);
        }
        blameInstancesManager.removeInstance(key);
    }

    /**
     * Handle `registered` event.
     * If registered url is used by the current blame instance - reload the source.
     * Here we convert both instance and config keys to string just to be on the safe side.
     * @param config {Object}
     */
    function onUrlRegistered(config) {
        var key = config.key;
        var instance = blameInstancesManager.getCurrentInstance();
        if (instance && String(instance.key) === String(key)) {
            switchTheSource(null);
            blameInstancesManager.removeInstance(key);
            switchTheSource(key, instance.page, instance.$source);
        }
    }

    return {
        /**
         * Initialize blame component.
         * What does source-code:* event mean?
         *   shown - triggers when source code is shown on the screen
         *   hidden - triggers when source code was hidden from the screen
         *   reset - triggers when source code is going to be modified (e.g. after changing "diff mode")
         */
        init: _.once(function () {
            eventBus.on('source-code:shown', onSourceShown);
            eventBus.on('source-code:hidden', onSourceHidden);
            eventBus.on('source-code:reset', onSourceReset);

            urlsManager.on('registered', onUrlRegistered);
        })
    };

});
