define('FECRU/component/blame/blame-instances-manager', [
    'FECRU/component/blame/blame-instance'
], function (BlameInstance) {

    /**
     * Thing that manages all blame instances.
     * @constructor
     */
    function BlameInstancesManager() {
        this.currentInstance = null;
        this.instances = {};
    }

    BlameInstancesManager.prototype = {
        /**
         * Return an existing instance or create a new one and return it.
         * @param key {?string}
         * @param [page] {string} Page identificator
         * @param [$source] {jQuery} Source code element
         * @returns {?BlameInstance}
         */
        ensureInstance: function (key, page, $source) {
            if (key == null) {
                return null;
            }
            return this.instances[key] || (this.instances[key] = new BlameInstance(key, page, $source));
        },

        /**
         * Make passed instance current one.
         * @param instance {?BlameInstance}
         * @returns {?BlameInstance} Current instance
         */
        setCurrentInstance: function (instance) {
            return this.currentInstance = instance || null;
        },

        /**
         * Check is passed instance is the current one.
         * @param instance {BlameInstance|string} Instance itself or instance key
         * @returns {boolean}
         */
        isCurrentInstance: function (instance) {
            if (instance instanceof BlameInstance === false) {
                instance = this.getInstance(instance);
            }
            return this.getCurrentInstance() === instance;
        },

        /**
         * @returns {?BlameInstance}
         */
        getCurrentInstance: function () {
            return this.currentInstance;
        },

        /**
         * @param key {string}
         * @returns {?BlameInstance}
         */
        getInstance: function (key) {
            return this.instances[key] || null;
        },

        /**
         * Remove the instance by the reference of by its key.
         * Unset the current instance if it has to be removed.
         * @param instance {BlameInstance|string} Instance itself or instance key
         */
        removeInstance: function (instance) {
            if (instance instanceof BlameInstance === false) {
                instance = this.getInstance(instance);
            }
            if (this.isCurrentInstance(instance)) {
                this.setCurrentInstance(null);
            }
            if (instance) {
                this.instances[instance.key] = null;
            }
        },

        /**
         * Switch the current instance.
         * In case there is no instance with passed key - create a new one.
         * @param instance {BlameInstance|string} Instance itself or instance key
         * @returns {?BlameInstance} Current instance
         */
        switchToInstance: function (instance) {
            if (instance instanceof BlameInstance === false) {
                instance = this.ensureInstance(instance);
            }

            return this.setCurrentInstance(instance);
        }
    };

    return BlameInstancesManager;

});
