/**
 * Sends analytics events for blame interactions
 * @see https://extranet.atlassian.com/display/FECRU/Blame+2.0+Analytics
 */

define('FECRU/component/blame/blame-analytics', [
    'backbone',
    'global-ns/fecru-event-bus',
    'global-ns/fecru-analytics'
], function (Backbone, eventBus, analytics) {

    var PAGE_TO_PREFIX_MAP = {
        review: 'cru.view.review.frx',
        changeset: 'fe.changeset',
        source: 'fe.source',
        diff: 'fe.diff'
    };

    /**
     * Convert passed to time to time range.
     * @param time {number} In milliseconds
     * @returns {string}
     */
    var toTimeRange = function (time) {
        if (time >= 0 && time < 100) {
            return '0..100ms';
        } else if (time < 500) {
            return "100ms..500ms";
        } else if (time < 1000) {
            return "500ms..1s";
        } else if (time < 4000) {
            return "1s..4s";
        } else if (time < 30000) {
            return "4s..30s";
        } else if (time >= 30000) {
            return "30s..infinity";
        }
    };

    /**
     * Create a blame analytics handler function.
     * @param prefix {string} Event prefix
     * @param eventName {string} Event name
     * @param [dataTransformer] {function} Data transformer to forward properties from triggered event
     * @returns {function}
     */
    var createSendFunction = function (prefix, eventName, dataTransformer) {
        return function () {
            var name = prefix + '.blame.' + eventName;
            var data = _.isFunction(dataTransformer) ? dataTransformer.apply(null, arguments) : void 0;
            analytics.send(name, data);
        }
    };

    /**
     * @param time {number}
     * @returns {Object}
     */
    var timeToAnalyticsObject =  function (time) {
        return {
            time: toTimeRange(time)
        };
    };

    /**
     * @param time {number}
     * @param errorCode {number}
     * @returns {Object}
     */
    var timeAndErrorCodeToAnalyticsObject =  function (time, errorCode) {
        return {
            time: toTimeRange(time),
            errorCode: errorCode
        };
    };


    /**
     * @constructor
     */
    function BlameAnalytics () {
        this.eventBus = _.extend({}, Backbone.Events);
    }

    BlameAnalytics.prototype = {

        /**
         * @param page {string} Page identificator - review|changeset|source|diff
         */
        listen: function (page) {
            var prefix = PAGE_TO_PREFIX_MAP[page];
            if (prefix) {
                var on = this.eventBus.listenTo.bind(this.eventBus, eventBus);
                var send = createSendFunction.bind(null, prefix);

                on('BLAME_ANALYTICS:READY', send('ready'));
                on('BLAME_ANALYTICS:REQUEST', send('request'));
                on('BLAME_ANALYTICS:HIDE', send('hide'));
                on('BLAME_ANALYTICS:SHOWN_CACHED', send('shown.cached'));
                on('BLAME_ANALYTICS:SHOWN_FETCHED', send('shown.fetched', timeToAnalyticsObject));
                on('BLAME_ANALYTICS:FETCH_ABORTED', send('fetch.aborted', timeToAnalyticsObject));
                on('BLAME_ANALYTICS:FETCH_FAILED', send('fetch.failed', timeAndErrorCodeToAnalyticsObject));
            }
        },

        unlisten: function () {
            this.eventBus.stopListening();
        }

    };

    return BlameAnalytics;

});
