package com.atlassian.confluence.extra.dynamictasklist;

import com.atlassian.confluence.core.ContentEntityObject;
import com.atlassian.confluence.extra.dynamictasklist.model.TaskList;
import com.atlassian.confluence.renderer.PageContext;
import com.atlassian.confluence.renderer.radeox.macros.MacroUtils;
import com.atlassian.confluence.util.VelocityUtils;
import com.atlassian.renderer.RenderContext;
import com.atlassian.renderer.v2.RenderMode;
import com.atlassian.renderer.v2.RenderUtils;
import com.atlassian.renderer.v2.macro.BaseMacro;
import com.atlassian.renderer.v2.macro.MacroException;
import org.apache.log4j.Category;

import java.util.Map;

/**
 * This macro defines a simple clickable task list.
 * <p>
 * Usage:
 * <pre>
 * {dynamictasklist:Shopping List}
 * Buy apples
 * Buy milk
 * Collect laundry
 * {dynamictasklist}
 * </pre>
 *
 * The task list model is maintained by a separate model class {@link TaskList}.
 */
public class DynamicTaskListMacro extends BaseMacro
{
    private static final Category log = Category.getInstance(DynamicTaskListMacro.class);
    private TaskListManager taskListManager;

    public boolean isInline() {
        return false;
    }

    public boolean hasBody() {
        return true;
    }

    public RenderMode getBodyRenderMode() {
        return RenderMode.NO_RENDER;
    }

    public String execute(Map parameters, String body, RenderContext renderContext) throws MacroException {
        PageContext pageContext = (PageContext) renderContext;

        // If we are rendering for thr WYSIWYG 'preview' tab, display a message and don't render the macro
        if (PageContext.PREVIEW.equals(pageContext.getOutputType()))
            return RenderUtils.error("The dynamic task list macro cannot be rendered in 'preview' mode.");

        // retrieve the first parameter passed, in our case the name of the task list.
        String listName = (String) parameters.get("0");

        // if no list name is given, the name of the list will be "_".
        // This will show as an empty String in the frontend. 
        if (listName == null)
            listName = "_";

        //replace all spaces in a String with underscores
        listName = listName.replaceAll("\\s", "_");

        // retrieve a reference to the body object this macro is in
        ContentEntityObject contentObject = ((PageContext)renderContext).getEntity();

        TaskList taskList = taskListManager.getTaskList(contentObject, listName);

        // now create a simple velocity context and render a template for the output
        Map contextMap = MacroUtils.defaultVelocityContext();
        contextMap.put("tasklist", taskList);
        contextMap.put("content", contentObject);
        contextMap.put("promptOnDelete", parameters.get("promptOnDelete"));

        try
        {
            return VelocityUtils.getRenderedTemplate("templates/extra/dynamictasklist/dynamictasklistmacro.vm", contextMap);
        }
        catch (Exception e)
        {
            log.error("Error while trying to display TaskList!", e);
            throw new MacroException(e.getMessage());
        }
    }

    public void setTaskListManager(TaskListManager taskListManager)
    {
        this.taskListManager = taskListManager;
    }
}
