package com.atlassian.confluence.rpc.soap.services;

import com.atlassian.confluence.pages.BlogPost;
import com.atlassian.confluence.pages.PageManager;
import com.atlassian.confluence.rpc.RemoteException;
import com.atlassian.confluence.rpc.soap.AbstractSoapTestCase;
import com.atlassian.confluence.rpc.soap.beans.RemoteBlogEntry;
import com.atlassian.confluence.rpc.soap.beans.RemoteBlogEntrySummary;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.spaces.SpaceManager;
import com.mockobjects.dynamic.C;
import com.mockobjects.dynamic.Mock;

import java.util.Date;
import java.util.Calendar;
import java.util.List;
import java.util.ArrayList;

public class TestBlogsSoapService extends AbstractSoapTestCase
{
    BlogsSoapService service;
    Mock mockPageManager;
    Mock mockServiceHelper;
    Mock mockSpaceManager;

    public void setUp() throws Exception
    {
        super.setUp();

        mockPageManager = new Mock(PageManager.class);
        mockServiceHelper = new Mock(SoapServiceHelper.class);
        mockSpaceManager = new Mock(SpaceManager.class);

        service = new BlogsSoapService();
        service.setPageManager((PageManager) mockPageManager.proxy());
        service.setSoapServiceHelper((SoapServiceHelper) mockServiceHelper.proxy());
        service.setSpaceManager((SpaceManager) mockSpaceManager.proxy());
    }

    public void testGetBlogEntry() throws RemoteException
    {
        Space space = makeSpace("test");
        BlogPost blogPost = makeBlogPost(space, 1, "title");
        blogPost.setCreationDate(new Date(100000));
        blogPost.setCreatorName("bob");
        blogPost.setContent("test content");

        mockServiceHelper.expectAndReturn("assertCanView", blogPost, true);
        mockPageManager.matchAndReturn("getAbstractPage", C.args(C.eq((long) 1)), blogPost);

        assertEquals(new RemoteBlogEntry(blogPost), service.getBlogEntry(blogPost.getId()));

        mockServiceHelper.verify();
    }

    public void testGetBlogEntries() throws Exception
    {
        // retrieve the space
        String SPACE_KEY = "FRUIT";
        Space space = makeSpace(SPACE_KEY);
        BlogPost bananas = makeBlogPost(space, 1, "Bananas");
        bananas.setCreationDate(new Date(100));
        bananas.setCreatorName("fred");
        BlogPost apples = makeBlogPost(space, 2, "Apples");
        apples.setCreationDate(new Date(100));
        apples.setCreatorName("bob");

        List posts = new ArrayList();
        posts.add(bananas);
        posts.add(apples);
        mockSpaceManager.matchAndReturn("getBlogPosts", C.args(C.eq(space), C.IS_TRUE), posts);

        mockServiceHelper.expectAndReturn("retrieveSpace", C.args(C.eq(SPACE_KEY)), space);
        RemoteBlogEntrySummary[] summaries = service.getBlogEntries(SPACE_KEY);
        assertEquals(2, summaries.length);
        assertEquals(new RemoteBlogEntrySummary(bananas), summaries[0]);
        assertEquals(new RemoteBlogEntrySummary(apples), summaries[1]);
    }

    public void testCreateBlogPost() throws RemoteException
    {
        String SPACE_KEY = "FRUIT";
        Space space = makeSpace(SPACE_KEY);
        RemoteBlogEntry testRemoteBlogEntry = new RemoteBlogEntry();
        testRemoteBlogEntry.setSpace(space.getKey());
        testRemoteBlogEntry.setTitle("Apples");
        testRemoteBlogEntry.setContent("test content");

        BlogPost addedBlogEntry = new BlogPost();
        addedBlogEntry.setTitle(testRemoteBlogEntry.getTitle());
        addedBlogEntry.setContent(testRemoteBlogEntry.getContent());
        addedBlogEntry.setSpace(space);

        mockServiceHelper.expectAndReturn("retrieveSpace", C.args(C.eq(SPACE_KEY)), space);
        mockServiceHelper.expect("assertCanView", C.args(C.eq(space)));
        mockServiceHelper.expect("assertCanCreateBlogPost", C.args(C.eq(space)));
        mockPageManager.expectAndReturn("getBlogPost", C.args(C.eq(space.getKey()), C.eq(testRemoteBlogEntry.getTitle()), C.isA(Calendar.class)), null);
        mockPageManager.expect("saveContentEntity", C.args(C.eq(addedBlogEntry), C.IS_NULL));

        service.storeBlogEntry(testRemoteBlogEntry);

        mockPageManager.verify();
        mockServiceHelper.verify();
    }

    public void testUpdateBlogPost() throws RemoteException
    {
        String SPACE_KEY = "FRUIT";
        Space space = makeSpace(SPACE_KEY);

        // create a blogpost that we want to try and edit later by posting a updated RemoteBlogEntry object
        BlogPost testBlogPost = new BlogPost();
        testBlogPost.setSpace(space);
        testBlogPost.setId(123L);
        String TITLE = "Bananas";
        testBlogPost.setTitle(TITLE);
        testBlogPost.setContent("test content");

        mockServiceHelper.matchAndReturn("retrieveSpace", C.args(C.eq(SPACE_KEY)), space);
        mockServiceHelper.expect("assertCanModify", C.args(C.eq(testBlogPost)));
        mockPageManager.expectAndReturn("getBlogPost", C.args(C.eq(123L)), testBlogPost); // should return the BlogPost we just created
        mockPageManager.expect("saveContentEntity", C.anyArgs(2));

        // now lets udpate the title and body of the post
        RemoteBlogEntry updatedRemoteBlogEntry = new RemoteBlogEntry();
        updatedRemoteBlogEntry.setId(123L);
        updatedRemoteBlogEntry.setSpace(SPACE_KEY);
        updatedRemoteBlogEntry.setContent("test content has changed");

        service.storeBlogEntry(updatedRemoteBlogEntry);

        mockPageManager.verify();
        mockServiceHelper.verify();
    }


}
