package com.atlassian.confluence.rpc.soap.services;

import com.atlassian.confluence.pages.BlogPost;
import com.atlassian.confluence.pages.PageManager;
import com.atlassian.confluence.rpc.RemoteException;
import com.atlassian.confluence.rpc.soap.SoapUtils;
import com.atlassian.confluence.rpc.soap.beans.RemoteBlogEntry;
import com.atlassian.confluence.rpc.soap.beans.RemoteBlogEntrySummary;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.spaces.SpaceManager;

import java.util.Calendar;
import java.util.GregorianCalendar;

public class BlogsSoapService
{
    PageManager pageManager;
    SoapServiceHelper soapServiceHelper;
    private SpaceManager spaceManager;

    public RemoteBlogEntry getBlogEntryByDayAndTitle(String spaceKey, int dayOfMonth, String postTitle) throws RemoteException
    {
        // Parse the dateString into a Calendar object
        Calendar calDate = new GregorianCalendar();
        calDate.set(Calendar.DATE, dayOfMonth);

        // Try and retrieve it
        BlogPost blogPost = (BlogPost) pageManager.getBlogPost(spaceKey, postTitle, calDate);

        soapServiceHelper.assertCanView(blogPost);

        return new RemoteBlogEntry(blogPost);
    }

    public RemoteBlogEntry getBlogEntry(long entryId) throws RemoteException
    {
        BlogPost entry = (BlogPost) pageManager.getAbstractPage(entryId);
        soapServiceHelper.assertCanView(entry);
        return new RemoteBlogEntry((BlogPost) pageManager.getAbstractPage(entryId));
    }

    public RemoteBlogEntrySummary[] getBlogEntries(String spaceKey) throws RemoteException
    {
        Space space = soapServiceHelper.retrieveSpace(spaceKey);
        return SoapUtils.getBlogEntrySummaries(spaceManager.getBlogPosts(space, true));
    }

    public RemoteBlogEntry storeBlogEntry(RemoteBlogEntry remoteBlogEntry) throws RemoteException
    {
        if (remoteBlogEntry.getId() <= 0)
            return createBlog(remoteBlogEntry);
        else
            return updateBlog(remoteBlogEntry);
    }

    private RemoteBlogEntry createBlog(RemoteBlogEntry remoteBlogEntry) throws RemoteException
    {
        Space space = soapServiceHelper.retrieveSpace(remoteBlogEntry.getSpace());

        soapServiceHelper.assertCanView(space);
        soapServiceHelper.assertCanCreateBlogPost(space);

        Calendar publishDateCalendar = Calendar.getInstance();

        if (remoteBlogEntry.getPublishDate() != null)
            publishDateCalendar.setTime(remoteBlogEntry.getPublishDate());

        // check if a blog with this date and title already exists
        BlogPost blogPost = pageManager.getBlogPost(space.getKey(), remoteBlogEntry.getTitle(), publishDateCalendar);

        if (blogPost != null)
            throw new RemoteException("A blog post with title '" + remoteBlogEntry.getTitle() + "' already exists for date: " + publishDateCalendar);

        blogPost = new BlogPost();
        blogPost.setSpace(space);
        blogPost.setTitle(remoteBlogEntry.getTitle());
        blogPost.setContent(remoteBlogEntry.getContent());
        blogPost.setCreationDate(remoteBlogEntry.getPublishDate());

        pageManager.saveContentEntity(blogPost, null);
        return new RemoteBlogEntry(blogPost);
    }

    /**
     *
     * @param remoteBlogEntry - you will need to set the id and spacekey of the BlogPost you are updating
     * @return
     * @throws RemoteException
     */
    private RemoteBlogEntry updateBlog(RemoteBlogEntry remoteBlogEntry) throws RemoteException
    {
        BlogPost blogPost = pageManager.getBlogPost(remoteBlogEntry.getId());

        soapServiceHelper.assertCanModify(blogPost);

        if (blogPost == null)
        {
            throw new RemoteException("The blog post with id " + remoteBlogEntry.getId() + " you are trying to update does not exist.)");
        }

        if (!blogPost.getSpace().getKey().equals(remoteBlogEntry.getSpace()))
        {
            throw new RemoteException("You can't change an existing page's space.");
        }
        if (remoteBlogEntry.getTitle() != null && !blogPost.getTitle().equals(remoteBlogEntry.getTitle()))
        {
            throw new RemoteException("You currently can't rename a blog post");
        }

        // udpate content of post
        blogPost.setContent(remoteBlogEntry.getContent());

        pageManager.saveContentEntity(blogPost, null);
        return new RemoteBlogEntry(blogPost);
    }

    public void setPageManager(PageManager pageManager)
    {
        this.pageManager = pageManager;
    }

    public void setSpaceManager(SpaceManager spaceManager)
    {
        this.spaceManager = spaceManager;
    }

    public void setSoapServiceHelper(SoapServiceHelper soapServiceHelper)
    {
        this.soapServiceHelper = soapServiceHelper;
    }
}
