package com.atlassian.confluence.rpc.soap.services;

/**
 * A Soap service holding functionality for Confluence Administrators.
 */

import com.atlassian.confluence.importexport.ImportExportManager;
import com.atlassian.confluence.importexport.ExportContext;
import com.atlassian.confluence.importexport.ImportExportException;
import com.atlassian.confluence.rpc.RemoteException;
import com.atlassian.confluence.rpc.soap.beans.RemoteServerInfo;
import com.atlassian.confluence.user.AuthenticatedUserThreadLocal;
import com.atlassian.confluence.util.GeneralUtil;
import com.atlassian.confluence.setup.BootstrapManager;
import com.atlassian.confluence.security.GateKeeper;
import com.atlassian.confluence.search.lucene.ConfluenceIndexManager;
import com.atlassian.user.User;

import java.io.IOException;

public class AdminSoapService
{
    private ConfluenceIndexManager indexManager;
    private ImportExportManager importExportManager;
    private BootstrapManager bootstrapManager;
    private GateKeeper gateKeeper;
    SoapServiceHelper soapServiceHelper;

    public void setImportExportManager(ImportExportManager importExportManager)
    {
        this.importExportManager = importExportManager;
    }


    /**
     * @param user - the current authenticated user
     * @return true if the user has the SpacePermission.ADMINISTRATE_CONFLUENCE_PERMISSION, otherwise false
     * @throws RemoteException
     */
    public boolean isConfluenceAdministrator(User user) throws RemoteException
    {
        return GeneralUtil.isSuperUser(user);
    }

    public String exportSite(boolean exportAttachments) throws RemoteException
    {
        String downloadPath = null;

        //1. get the user for the export
        User user = AuthenticatedUserThreadLocal.getUser();

        //2. permission check
        if (!isConfluenceAdministrator(user))
            throw new RemoteException("Authenticated user [" + user + "] is not a Confluence Administrator.");

        //3. export
        try
        {
            ExportContext exportContext = importExportManager.buildExportContext(exportAttachments, false, false, ImportExportManager.EXPORT_XML_TYPE_ALL);

            String archivePath = importExportManager.exportAs(ImportExportManager.TYPE_ALL_DATA, exportContext);
            downloadPath = importExportManager.prepareDownloadPath(archivePath);

            gateKeeper.addKey(downloadPath, user);
        }
        catch (ImportExportException e)
        {
            return "Could not export space: " + e;
        }
        catch (IOException e)
        {
            return "Could not export space: " + e;
        }

        String finalPath = bootstrapManager.getBaseUrl() + downloadPath;

        return finalPath;
    }

    public void setBootstrapManager(BootstrapManager bootstrapManager)
    {
        this.bootstrapManager = bootstrapManager;
    }

    public void setGateKeeper(GateKeeper gateKeeper)
    {
        this.gateKeeper = gateKeeper;
    }

    public void setSoapServiceHelper(SoapServiceHelper soapServiceHelper)
    {
        this.soapServiceHelper = soapServiceHelper;
    }

    public void setIndexManager(ConfluenceIndexManager indexManager)
    {
        this.indexManager = indexManager;
    }

    public RemoteServerInfo getServerInfo() throws RemoteException
    {
        // As of Confluence 2.0, we will allow the user to view server information even if they're not an admin (CONF-4566)
        return new RemoteServerInfo();
    }

    //---( Indexing related Remote API. This can be moved into its own service when it becomes a little larger.)---

    /**
     * Support remote interface for flushing the index queue.
     *
     */
    public boolean flushIndexQueue() throws RemoteException
    {
        soapServiceHelper.assertCanAdministrate();
        indexManager.flushQueue();
        return true;
    }

    /**
     * Support remote interface for clearing out the index queue.
     *
     */
    public boolean clearIndexQueue() throws RemoteException
    {
        soapServiceHelper.assertCanAdministrate();
        indexManager.resetIndexQueue();
        return true;
    }
}
